#include "FitScanResult.h"
#include "RawScanResult.h"
#include "ModuleConfiguration.h"
#include "ScanPoints.h"
#include "Defect.h"
#include "FitObject.h"

#include "Sct/SctParameters.h"
#include <TH1.h>
#include <iostream>

using namespace Sct;

namespace SctData {

FitScanResult::FitScanResult() throw() : m_channelFits("ChannelFits"), m_chipFits("ChipFits"), m_linkFits("LinkFits"),
										 m_summaryHist("summaryHist") {}

FitScanResult::FitScanResult(const ResultHeader& header, const ModuleConfiguration& config, const ScanPoints& points) throw()
					        : ScanResult(header, config, points), m_channelFits("ChannelFits"),  m_chipFits("ChipFits"), m_linkFits("LinkFits"),
							m_summaryHist("summaryHist") {}

FitScanResult::FitScanResult(const RawScanResult& raw) throw () : ScanResult(raw), m_channelFits("ChannelFits"),  m_chipFits("ChipFits"), 
							m_linkFits("LinkFits"), m_summaryHist("summaryHist") {}

FitScanResult::~FitScanResult() throw() {
    for (vector<TH1D*>::iterator ih = m_summaryHist.begin(); ih != m_summaryHist.end(); ++ih ) {
        delete *ih;
    }
}

string FitScanResult::getClassName() const throw() {
    return "SctData::FitScanResult";
}

string FitScanResult::getUniqueID() const throw() {
    return getHeader().getUniqueID();
}

string FitScanResult::getUniqueID(const ResultHeader& header) throw() {
    return header.getUniqueID();
}

void FitScanResult::initializeChannelFits(const FitObject& prototype) throw() {
    resizeAndFill(m_channelFits, Sct::nChannelModule, prototype);
}

void FitScanResult::initializeChipFits(const FitObject& prototype) throw() {
    resizeAndFill(m_chipFits, Sct::nChipModule, prototype);
}
void FitScanResult::initializeLinkFits(const FitObject& prototype) throw() {
    resizeAndFill(m_linkFits, Sct::nLinkModule, prototype);
}

FitObject& FitScanResult::getLinkFit( const unsigned ilink ) const throw(LogicError) {
    return *m_linkFits[ilink] ;
}

FitObject& FitScanResult::getChipFit( const unsigned ilink, const unsigned ichip ) const throw (LogicError) {
    return this->getChipFit( ilink*6+ichip );
}

FitObject& FitScanResult::getChipFit( const unsigned ichip ) const throw (LogicError) {
    return *m_chipFits[ichip] ;
}

FitObject& FitScanResult::getChannelFit( const unsigned ilink, const unsigned ichannel ) const throw (LogicError ) {
    return this->getChannelFit(ilink*nChannelLink+ichannel);
}

FitObject& FitScanResult::getChannelFit( const unsigned ichannel ) const throw(LogicError) {
    return *m_channelFits[ichannel] ;
}

void FitScanResult::setChannelFit(unsigned int channel, shared_ptr<FitObject> fit) throw(LogicError) {
    m_channelFits[channel] = fit;
}
    
void FitScanResult::setChipFit(unsigned int chip, shared_ptr<FitObject> fit) throw(LogicError) {
    m_chipFits[chip] = fit;
}
    
void FitScanResult::setLinkFit(unsigned int link, shared_ptr<FitObject> fit) throw(LogicError) {
    m_linkFits[link] = fit;
}

unsigned FitScanResult::getNLinkFits() const throw() {
    return m_linkFits.size();
}

unsigned FitScanResult::getNChipFits() const throw() {
    return m_chipFits.size();
}

unsigned FitScanResult::getNChannelFits() const throw() {
    return m_channelFits.size();
}

void FitScanResult::setNChannelFits(unsigned int n) throw() {
    m_channelFits.resize(n);
}
    
void FitScanResult::setNChipFits(unsigned int n) throw() {
    m_chipFits.resize(n);    
}
    
void FitScanResult::setNLinkFits(unsigned int n) throw() {
    m_linkFits.resize(n);    
}


const TH1D& FitScanResult::getHist(unsigned i) const throw(LogicError) {
    return *(m_summaryHist[i]);
}

}

ostream& operator<<(ostream& os, const SctData::FitScanResult& r) throw() {
    os << r.getClassName() << "  " << r.getUniqueID();
    return os;
}
