#ifndef SCTDATA_FITOBJECT_H
#define SCTDATA_FITOBJECT_H

#include "Sct/Streamable.h"
#include <vector>
#include <string>
#include <iostream>
#include <memory>
#include "Sct/Exception.h"
#include "Sct/LogicErrors.h"

class TF1;

using namespace std;
using Sct::LogicError;

namespace SctData{

/**
 * FitObject is an abstract base class fit object to store the
 * output to various types of fit.
 * </P><P>
 * It can be published to and retrieved from IS.
 * It can generate a ROOT TF1, or be created from one.
 * Derived classes must implement a makeBasicRootTF1() method
 * which use a suitable TF1 constructor.
 *  </P><P>
 * So far I can't see any sensible use for `number of dimensions' != 1
 * but the class could be extensible to multi-dimensional functions.
 * </P><P>
 * The formula provided may be in a form comprehensible to ROOT
 * e.g. "[0]*x*sin([1]*x)" for a 1-dimensional (x), 2-parameter ([0],[1]) function.
 * </p><p>
 * The analytical form of the functions themselves are in `FitFunctions.cpp'.
 * </p>
 * @author Alan Barr
 * @date 6 December 2002
 */
class FitObject : public virtual Sct::Streamable {
 public:

    /**
       Allows any sub-class of FitObject to be used as a prototype
       @return a new FitObject of the appropriate type which is a copy of this.
    */
    virtual shared_ptr<FitObject> clone() const throw() = 0;
    
    /**
   * Conversion utility functions;
   * <p>
   * Make a new ROOT TF1 from this object. Calls the virtual function
   * makeBasicRootTF1()
   */
    virtual auto_ptr<TF1> makeRootTF1() const throw(LogicError);

  /**
   * Set the values of this FitObject equal to those of a ROOT TF1 function
   * to be implimented in sub-classes
   */
    FitObject& operator=(const TF1&) throw(LogicError);

    /// Print this to cout

    void print() const throw(LogicError) ;

    //Serializable override
    //virtual string getClassName() const throw() ;

    /// Get the dimensionality
    inline int getNDim() const throw(){ return m_varMax.size();}

    /// Get the number of parameters
    inline int getNPar() const throw() { return m_parameter.size();}

    /// Get name of parameter ipar
    char* getParName(const int ipar) const throw(LogicError) ;

    /**
       Return the parameter index corresponding to name.
       The index can be used in the other functions.
    */
    int getParIndex(const string& name) const throw(LogicError) ;

    //@{
    ///  Get the value of parameter
    /// @param the parameter number
    double getParameter(const int ipar) const throw(LogicError);
    ///  Get the value of parameter
    /// @param the parameter name
    double getParameter(const string& name) const throw(LogicError) ;
    //@}

    /// @{
    /// Get the error on parameter
    /// @param the parameter number
    double getParError(const int ipar) const throw(LogicError);

    /// Get the error on parameter
    /// @param the parameter name
    /// @}
    double getParError(const string& name) const throw(LogicError);

    ///  Get the chisquared
    double getChiSquared() const throw() {return m_chiSquared;}

    ///  Get the number of degrees of freedom
    int getNDF() const throw() { return m_nDF;}

    ///  Get the maximum value of the variable
    double getVarMax(const int ivar) const throw(LogicError);

    /// Get the minimum value of the variable
    double getVarMin(const int ivar) const throw(LogicError);

    ///  Set the maximum value of the variable:
    void setVarMax(const int ivar, const double value) throw(LogicError) ;

    /// Set the minimum value of the variable
    void setVarMin(const int ivar, const double value) throw(LogicError) ;

    /// get the Formula from the m_type object
    char* getFormula() const throw() {
	return const_cast <char*> ( m_formula.c_str() ) ;
    }

    /// Fixes or unfixes parameter ipar
    /// @throws OutOfRangeError if ipar > NPar
    void fixParameter(int ipar, bool fix) ;
    
    /// Returns whether parameter ipar is fixed
    /// @throws OutOfRangeError if ipar > NPar    
    bool isFixed(int ipar) const;
    
    ///  Set the value of parameter ipar
    void setParameter(const int ipar, const double value) throw (LogicError) ;

    ///  Set the error of parameter ipar
    void setParError(const int ipar, const double value) throw (LogicError) ;

    //// Set the value of parameter ipar. This should be done in the derived class

    void setParName(const int ipar, const char* name) throw (LogicError) ;
    
    ///  Set the chiSquared
    void setChiSquared(const double c) throw() { m_chiSquared=c; }

    ///  Set the number of degrees of freedom
    void setNDF(const int d) throw() { m_nDF=d; }


    /// Destructor does nothing
    virtual ~FitObject() throw() {;}

    /// Sets parameters, errors, chisquared and NDF to zero.
    void reset() throw() ;
protected:
    //Most data objects protected. Accessible through setXxx and getXxx functions.
    vector <double> m_parameter;     ///< values of parameters
    vector <double> m_parError;      ///< error on parameters
    vector <string> m_parName;       ///< names of parameters
    vector <bool> m_parFixed;        ///< true if parameter is fixed
    vector <double> m_varMax;        ///< maximum value of variables
    vector <double> m_varMin;        ///< minimum value of variables
    double m_chiSquared;             ///< chisquared parameter
    int m_nDF;                       ///< degrees of freedom
    string m_formula ;               ///< formula of type understood by root


    /// Create a default FitObject
    ///@note Nothrow
    FitObject(string formula, unsigned int nPar, const vector<string>& parNames, unsigned int nDim=1);

    /// Create a FitObject from a ROOT TF1 object
    FitObject(string formula, unsigned int nPar, const vector<string>& parNames, const TF1& f) ;

    /**
     * Make a TF1 using the appropriate constructor which
     * can then be used in makeRootTF1().
     * To be implimented by <B>derived</B> classes.
     */
    virtual auto_ptr<TF1> makeBasicRootTF1() const throw(LogicError) =0;
}; // end declaration of FitObject

//Inlines
inline double FitObject::getParameter(const string& name) const throw(LogicError) {
	return getParameter(getParIndex(name));
}

inline double FitObject::getParError(const string& name) const throw(LogicError) {
	return getParError(getParIndex(name));
}

ostream& operator << (ostream&, const SctData::FitObject& ) throw (LogicError) ;

} 


#endif // #ifndef SCTDATA_FITOBJECT_H
