#ifndef SCTDATA_DEFECTPROTOTYPE_H
#define SCTDATA_DEFECTPROTOTYPE_H

#include <string>
#include <map>

using std::string;
using std::map;
   
namespace SctData {

/**
  Describes how severe defects are
  */
enum DefectSeverity {
    NONE = 0,		///< Used to indicate that there is no defect associated with the ModuleElement - should not be used for DefectPrototypes
    DODGY = 1,		///< ModuleElements affected by this defect should not have any special action taken	    
    SERIOUS = 2, 	///< ModuleElements affected by this defect should be excluded when determining parameters (e.g. trim range)	
    UNUSEABLE = 3 	///< ModuleElements affected by this defect are unuseable and should be masked
};       
    

/**
   Prototypes for making Defect s out of.
 */
    
class DefectPrototype {
public:    
    /**
    Gets the name of this defect
    */
    string getName() const throw() {return name;}

    /**
     Get a human readable description for this ModuleDefect
    */
    string getDescription() const throw() {return description;}

    /**
     Gets a parameter that can be used to determine if something is defective.
     */
    double getParameter() const throw() {return parameter;}
    
    /**
      Gets the severity of this type of defect.
      */
    DefectSeverity getSeverity() const throw() {return severity;}
    
    /**
    Returns true if the prototypes are the same.
    */
    bool operator==(const DefectPrototype& defect) const throw() {return id==defect.id;}

private:
    /**
    Construct a DefectPrototype
    @param id The unique identifier
    @param severity The severity
    @param name The name
    @param d The description in a form that will be passed to sprintf
    @param parameter The cut parameter (will be passed to sprintf to form the description
    @throws InvalidArgumentError if a DefectPrototype with the given id already exists
    */
    DefectPrototype(int id, DefectSeverity severity, string name, string d, double parameter);    
    
    //Data
    int id;			///< The internal id
    DefectSeverity severity;	///< The level of severity
    string name;                ///< Name of the defect
    string description;	        ///< The human readable description
    double parameter;           ///< A parameter for deciding if something is defective.

    //IOHelper stuff
    friend class DefectIOHelper;
    friend class StandardDefects;
    static map<int, DefectPrototype*>& getMap(); ///< static method to get the map
};

}

#endif //SCTDATA_DEFECTPROTOTYPE_H
