#ifndef MODULEDEFECTLIST_H
#define MODULEDEFECTLIST_H

#include "DefectPrototype.h"
#include "Defect.h"
#include "Sct/Streamable.h"
#include <list>
#include <memory>
#include <boost/shared_ptr.hpp>

using std::list;
using std::auto_ptr;
using Sct::Streamable;
using boost::shared_ptr;

namespace SctData {

class ModuleElement;

/**
This class represents all the defects in a module.
@author Matthew Palmer
*/
class DefectList : public virtual Streamable {
public:
    typedef list<Defect> DefectCollection;
    
    /**
    Create a new empty ModuleDefects object.
    @note Nothrow
    */
    DefectList();

    /**
    Add a defect. Neighbouring defects of the same type are merged.
    @param defect The Defect to add
    @note Nothrow
    */
    void addDefect(Defect defect);
    /**
       Get all defects in the ModuleDefectList.
       @return A reference to a list of all defects.
       @note Nothrow
    */
    const DefectCollection& getAllDefects() const;

    //@{ @name Methods enquiring as to defects on ANY PART of a module element
    
    /**
       Get all the defects affecting a given element
       @param element The ModuleElement
       @return a DefectList
       @note Nothrow
    */
    auto_ptr<DefectList> getDefectsAffectingElement(ModuleElement element) const;
    
    /** 
      Get defect severity
      @returns the Severity using NONE if there is no defect
      @note Nothrow
      */
    DefectSeverity defectSeverityAffectingElement(ModuleElement element) const;
    
    /** 
      Finds if there is a an identical defect to defect in this list
      @note Nothrow
      */
    bool containsDefect(Defect defect) const;
    
    //@}
    
    
    //@{ @name Methods enquiring as to defects which affect the WHOLE of an element.
    
    /**
       Get all defects who contain the element
       @note Nothrow
    */
    auto_ptr<DefectList> getDefectsEncompassingElement(ModuleElement element) const;
    
    /**
      Check for severe defects.
      @note Nothrow
      */
    DefectSeverity defectSeverityEncompassingElement(ModuleElement element) const;
    //@}
    
    virtual string getClassName() const throw();

    /**
      Add defects from another list.  Filters out identical defects.
      @note Nothrow
      */
    DefectList& operator +=(const DefectList&);

private:
    DefectCollection defectList;		///< All defects
};

}

#endif //#ifndef MODULEDEFECTLIST_H
