#ifndef CONFIGURATIONVARIABLE_H
#define CONFIGURATIONVARIABLE_H

#include "Sct/Streamable.h"
#include <boost/shared_ptr.hpp>
#include <string>
#include <map>

using std::string;
using std::map;
using boost::shared_ptr;

namespace SctData {
    
class ModuleConfiguration;

/**
  ConfigurationVariable represents a variable within ModuleConfiguration/ABCDModule.
  Sub-classes represent the particular (logical) variables and the different ways of
  setting them (e.g. ThresholdVariable represents the logical variable threshold and
  mVThresholdVariable represents setting the threshold in mV, fCThresholdVariable represents 
  setting it in fC).  The sub-classes are intended to be singletons (as they are really strategies and 
  flyweights).
  @author Matthew Palmer
  */
class ConfigurationVariable : public virtual Sct::Streamable {
public:
    virtual ~ConfigurationVariable() throw() {}
 
    string getClassName() const throw();
    
    bool operator== (const ConfigurationVariable& var) const throw();
    
    bool operator!= (const ConfigurationVariable& var) const throw();

    //Informational stuff
    string getVariableName() const throw();
    string getStrategyDescription() const throw();
    
    //Conversions
    virtual double getLinkActualPoint(double logicalPt, const ModuleConfiguration& config, unsigned int link) const throw() = 0;    
    virtual double getChipActualPoint(double logicalPt, const ModuleConfiguration& config, unsigned int chip) const throw() = 0;
    virtual double getChannelActualPoint(double logicalPt, const ModuleConfiguration& config, unsigned int channel) const throw() = 0;    
    
    virtual double getLinkLogicalPoint(double actualPt, const ModuleConfiguration& config, unsigned int link) const throw() = 0;
    virtual double getChipLogicalPoint(double actualPt, const ModuleConfiguration& config, unsigned int chip) const throw() = 0;    
    virtual double getChannelLogicalPoint(double actualPt, const ModuleConfiguration& config, unsigned int channel) const throw() = 0;    
    
    //ModuleConfiguration interaction
    virtual double getChipActualPoint(const ModuleConfiguration& config, unsigned int chip) const throw() = 0;
    virtual void setChipActualPoint(double actualPt, ModuleConfiguration& config, unsigned int chip) const throw() = 0;
    
    virtual double getChipLogicalPoint(const ModuleConfiguration& config, unsigned int chip) const throw();
    virtual void setChipLogicalPoint(double logicalPt, ModuleConfiguration& config, unsigned int chip) const throw();
    
    virtual double getModuleLogicalPoint(const ModuleConfiguration& config) const throw();
    virtual void setModuleLogicalPoint(double logicalPt, ModuleConfiguration& config) const throw();
    
protected:
    ConfigurationVariable(unsigned int typeVal, string variableName, string strategyDescription);
    
private:
    ConfigurationVariable(unsigned int typeVal);

    unsigned short typeVal;
    string variableName;
    string strategyDescription;
    typedef map<unsigned short, shared_ptr<ConfigurationVariable> > VariableTypeMap; 
    static VariableTypeMap& getMap() throw();
    friend class ConfigurationVariableIOHelper;
};


class ConfigurationVariableIOHelper {
public:
    static unsigned short getTypeRep(const ConfigurationVariable& var) throw();
    static shared_ptr<ConfigurationVariable> getFromTypeRep(unsigned short typeRep) throw();
};


}

#endif //#ifndef CONFIGURATIONVARIABLE_H
