#ifndef CACHEDFUNCTION1D_H
#define CACHEDFUNCTION1D_H

#include <iostream>
#include <cmath>
using namespace std;


namespace SctData{

    /**
     * Caches the values of a 1-d function into a look-up table so that it
     * can later be evaluated very quickly.
     * @author Alan Barr
     * @date 3 April 2003
     */
    class CachedFunction1D{
    public:
	/**
	 * Constructor creates a look-up table for the function in the range [min,max) 
	 * with step delta between x-values. <P>
	 * e.g. CachedFunction1D(0., 1., 0.001, &TMath::erf ); 
	 @param the minimum x-value
	 @param the maximum x-value
	 @param the x step size -> decrease for better precision.
	 @param the address of the function ``double f(double x)''
	 */
	CachedFunction1D(double min, double max, double delta, double (*function)(double) ) throw() ;
	/**
	 * Deletes the table
	 */
	~CachedFunction1D() throw() ;
	/**
	 * evaluate the function by picking a value from the look-up table.
	 * @param the value of the function at which to evaluate
	 * @return the appropriate value from the look-up table.
	 * if <B>x<min<B> or <B>x>max<B> then f(min) or f(max) are returned respectively.
	 */

	inline double eval(double x) const throw() ;
    private:
	/// not allowed to do this.
	CachedFunction1D() throw() ;
	/// not allowed to do this.
	CachedFunction1D& operator=(const CachedFunction1D&) throw() ;
	void makeTable() throw();    ///< called by the constructor to make the look-up table.
	double min;                  ///< min value of x for which the function is calculated.
	double max;                  ///< min value of x for which the function is calculated.
	double delta;                ///< the step in x between points in the table.
	double invdelta;             ///< cached value of 1/delta.
	int nvalue;                  ///< the number of values in the table.
	double * values;             ///< the table itself.
	double (*function) (double); ///< a pointer to the function itself.
    };
    
// inline function
    double CachedFunction1D::eval(double x) const throw() {
	/*
	int ipos=(int)((x-min)*invdelta);
	if (ipos<0)            { return values[0];        }
	else if (ipos>=nvalue) { return values[nvalue-1]; }
	else                   { return values[ipos];     }
	*/
	double pos = (x-min)*invdelta;
	int ipos=(int)(pos);
	if (ipos<0)            { return values[0];        }
	else if (ipos>=nvalue-1) { return values[nvalue-1]; }
	//interpolate
	double frac = pos - ipos;	
	double y1 = values[ipos];
	double y2 = values[ipos+1];
	//double x1 = ipos * delta + min;
	//double x2 = (ipos+1) * delta + min;
	//cout << x1 << "  "<< x2 << " "  << y1 << "  " << y2 << "  " << x << "  " << pos << "   " << ipos << "  " << frac << "  " << (x-x1)/(x2-x1) << endl;
	return y1 + (y2-y1) * frac;
    }

}



#endif // #ifndef CACHEDFUNCTION_H
