package SctData;

import hep.aida.ref.histogram.VariableAxis;
import hep.aida.IAxis;
import java.util.*;
import Sct.*;

/**
 * This class represents the points in a scan.  It contains methods to get the number of events, the number
 * of error events and the variable value for each point.  There are also some utility methods to get "bin"
 * values for use in setting histogram axes and to set a TAxis directly.
 * @author Matthew Palmer
 */
public class ScanPoints implements Streamable {
    /** Create a ScanPoints from data that is already known. */
    public ScanPoints(float[] points, int[] nEvents, int[] nErrorEvents) throws IllegalArgumentException{
        init(points, nEvents, nErrorEvents);
    }

    public ScanPoints(ScanPoints pts) {
        for (int i=0; i<pts.points.size(); ++i) {
            
        }
    }

    public int getNPoints() {
        return points.size();
    }

    public double getPoint(int i) {
        return ((ScanPoint)points.get(i)).point;
    }

    public int getNEvents(int i) {
        return ((ScanPoint)points.get(i)).nEvents;
    }

    public int getNErrorEvents(int i) {
        return ((ScanPoint)points.get(i)).nErrorEvents;
    }

    public void addPoint(double point, int nEvents, int nErrorEvents) {
        ScanPoint p = new ScanPoint(point, nEvents, nErrorEvents);
        points.add(p);
    }

    public boolean ascending(){
	if (getNPoints()<2) return true;
	boolean up=false, down=false;
	double lastPoint=getPoint(0);
	for (int ipoint=1; ipoint<getNPoints(); ++ipoint){
	    double point=getPoint(ipoint);
	    if (point<lastPoint) down=true;
	    if (point>lastPoint) up=true;
	    if (point==lastPoint || (up && down)) throw new IllegalStateException("ScanPoints not in ascending or descending order, or has repeated points");
	    lastPoint=point;
	}
	return up;
    }

    /** Returns an IAxis. */
    public IAxis getAxis() {
        if (getNPoints() < 2) throw new IllegalStateException("Must have at least 2 points");
        double[] edges = new double[getNPoints() + 1];
        
	if ( ascending() ){
	    edges[0] = 3./2. * getPoint(0) - 0.5 * getPoint(1);
	    for (int i=1; i<getNPoints(); ++i) {
		edges[i] = 0.5*(getPoint(i-1) + getPoint(i));
	    }
	    edges[getNPoints()] = 3./2. * getPoint(getNPoints()-1) - 0.5*getPoint(getNPoints()-2);
	} else{
	    edges[0] = 3./2. * getPoint(getNPoints()-1) - 0.5 * getPoint(getNPoints()-2);
	    for (int i=1; i<getNPoints(); ++i) {
		edges[i] = 0.5*(getPoint( getNPoints()-i-1 ) + getPoint( getNPoints()-i));
	    }
	    edges[getNPoints()] = 3./2. * getPoint(0) - 0.5*getPoint(1);
	}
	
        return new VariableAxis(edges);
    }

    public String getClassName() {
        return "SctData.ScanPoints";
    }

    public static ScanPoints read(IStream s, ObjectManager o) throws java.io.IOException {
        short nPoints = s.readShort("NPoints");
        float[] pts = s.readFloatArray("Points");
        int[] nEvents = s.readIntArray("nEvents");
        int[] nErrorEvents = s.readIntArray("nErrorEvents");
        
        if (nPoints != pts.length) throw new java.io.StreamCorruptedException("Arrays read in were not expected length\nExpected: " + nPoints + " Got: " + pts.length);
        try {
            return new ScanPoints(pts, nEvents, nErrorEvents);
        } catch (IllegalArgumentException e) {
            throw new java.io.StreamCorruptedException("Arrays read in were different lengths");
        }        
    }
    
    public void write(OStream s, ObjectManager o) throws java.io.IOException {
        short nPoints = (short)getNPoints();
        s.writeShort("NPoints", nPoints, false);
        
        //Copy to array
        float[] pts = new float[nPoints];
        int[] nEvents = new int[nPoints];
        int[] nErrorEvents = new int[nPoints];
        for (int i=0; i<nPoints; ++i) {
            pts[i] = (float)getPoint(i);
            nEvents[i] = getNEvents(i);
            nErrorEvents[i] = getNErrorEvents(i);
        }
        
        s.writeFloatArray("Points", pts);
        s.writeIntArray("nEvents", nEvents, false);
        s.writeIntArray("nErrorEvents", nErrorEvents, false);
    }

    
    private void init(float[] points, int[] nEvents, int[] nErrorEvents) throws IllegalArgumentException{
        if (points.length != nEvents.length || points.length != nErrorEvents.length)
            throw new IllegalArgumentException("Arrays must be the same length!!");
        
        for (int i=0; i<points.length; ++i) {
            addPoint(points[i], nEvents[i], nErrorEvents[i]);
        }
    }

    
    private ArrayList points = new ArrayList();

    private class ScanPoint {
        public ScanPoint(double point, int nEv, int nErr) {
            this.point = point;
            this.nEvents = nEv;
            this.nErrorEvents = nErr;
        }

        public double point;
        public int nEvents;
        public int nErrorEvents;
    }
}
