package SctData;

import hep.aida.*;
import hep.aida.ref.histogram.*;
import Sct.*;

/**
 * A RawScanResult represents the raw data for a single scan. Note that this object (like all others in the package) owns its
 * members (i.e. it will delete them when it is deleted). This can cause problems with the 4 argument constructor if you give
 * it local vars!!!! The get***Occupancy functions all return a TH1D with a constant name.
 * It is up to the calling code to rename it as required if it does not want the hist deleted the next time
 * any of these functions is called.
 * @author Matthew Palmer
 */
public class RawScanResult extends ScanResult {
    public RawScanResult(ResultHeader header, ModuleConfiguration moduleConfig, ScanPoints points, IHistogram2D scanData) {
        super(header, moduleConfig, points);
        this.scanData = scanData;
    }

    public String getClassName() {
        return "SctData.RawScanResult";
    }

    public String getUniqueID() {
        return "RawScanResult." + header.getUniqueID();
    }

    public static String getUniqueID(ResultHeader header) {
        return "RawScanResult." + header.getUniqueID();
    }

    /** Gets the scan data as a ROOT histogram. */
    public IHistogram2D getData() {
        return scanData;
    }

    /** Utility function that gets a general occupancy hist. Returns a TH1D containing the occupancy. */
    
    public static RawScanResult read(IStream s, ObjectManager o) throws java.io.IOException {
        return new RawScanResult(s, o);
    }
    
    public void write(OStream s, ObjectManager o) throws java.io.IOException {
        super.write(s, o);
        
        int nPoints = getPoints().getNPoints();
	//System.out.println("Write n points = "+nPoints);
	int x_size  = scanData.xAxis().bins();
	//System.out.println("Write x_size = "+x_size);
        int size = (x_size/2 + 2) * (nPoints + 2);
	//System.out.println("Write size = "+size);
        s.writeInt("Size", size, false);
        s.writeShort("Type", ROOTHIST, false);
        s.writeShort("Width", WIDTH_64BIT, false);

	/// ROOT-data specific bits:
	s.writeShort("OriginalType", originalType, false);
        
        //Copy data into arrays as necessary for Root format.  Ignore under and overflow bins
        double[] link0Data = new double[size];
        double[] link1Data = new double[size];
        
        for (int i=0; i<x_size/2; ++i) {
            for (int j=0; j<nPoints; ++j) {
                link0Data[(x_size/2+2)*(j+1)+i+1] = scanData.binHeight(i, j);                
            }
        }
        for (int i=0; i<x_size/2; ++i) {
            for (int j=0; j<nPoints; ++j) {
                link1Data[(x_size/2+2)*(j+1)+i+1] = scanData.binHeight(i+x_size/2, j);
            }
        }
        
        s.writeDoubleArray("Link0Data", link0Data);
        s.writeDoubleArray("Link1Data", link1Data);
    }
    
    private RawScanResult(IStream s, ObjectManager o) throws java.io.IOException {
        super(s, o);
        readData(s);
    }
    
    private void readData(IStream s) throws java.io.IOException {
        int size = s.readInt("Size");
        short type = s.readShort("Type");
        short width = s.readShort("Width");	

	originalType=type;
	
        switch (type) {
            case SLICE:
                readSliceData(s, size, width);
                return;
                
            case ROOTHIST:
                readRootData(s, size, width);
                return;

	   case RAWHIST:
	       readRawData(s, size, width);
               return;
                
            default:
                throw new java.io.StreamCorruptedException("Unsupported raw data type: " + type);
        }
    }

    Histogram2D setHistSize(int x_size){
	// need to set x_size of histogram here!
	//System.out.println("Set hist size "+x_size);
	FixedAxis xAxis = new FixedAxis(x_size, -0.5, x_size+0.5);
	Histogram2D data = new Histogram2D(getUniqueID(), getUniqueID(), xAxis, getPoints().getAxis());
	scanData = data;
	return data;
    }

    private int getWidthMultiplier(short width){
	//System.out.println("Width = " + width);
	switch (width) {
	case WIDTH_16BIT: return 1;
	case WIDTH_32BIT: return 2;
	case WIDTH_64BIT: return 4;
	default: return 0;
	}
    }
	
    private void readRawData(IStream s, int size, short width) throws java.io.IOException {
	int nPoints = getPoints().getNPoints();
	int quotient = getWidthMultiplier(width) * nPoints;
	if (quotient==0 || size % quotient != 0 ) throw new java.io.StreamCorruptedException("Data size  = " + size + " but trying to divide by " + quotient);
	int x_size = size/quotient;
	Histogram2D scan_data = setHistSize(x_size);
	boolean ascending=getPoints().ascending();

	switch (width) {
	case WIDTH_16BIT:{
	    short[] data = s.readShortArray("16BitSliceData");
	    if (data.length != size) {			
		throw new java.io.StreamCorruptedException("Incorrect data size.  Got: " + data.length + " but header says: " + size);
	    }
	    double [][] heights = new double[x_size+2][nPoints+2];
	    double [][] errors = new double[x_size+2][nPoints+2];
	    for (int i=0; i<x_size; ++i) {
		for (int j=0; j<nPoints; ++j) {
		    int jpt=0;
		    if (ascending) {jpt=j; }else{jpt=nPoints-j-1;}
		    heights[i+1][j+1] = data[jpt*x_size + i];
		    errors[i+1][j+1] = Math.sqrt(heights[i+1][j+1]);
		}
	    }
	    scan_data.setContents(heights, errors, null, null, null, null, null);
	}
	    return;
	case WIDTH_32BIT:{
	    int[] data = s.readIntArray("32BitSliceData");
	    if (data.length != size) {			
		throw new java.io.StreamCorruptedException("Incorrect data size.  Got: " + data.length + " but header says: " + size);
	    }
	    double [][] heights = new double[x_size+2][nPoints+2];
	    double [][] errors = new double[x_size+2][nPoints+2];
	    for (int i=0; i<x_size; ++i) {
		for (int j=0; j<nPoints; ++j) {
		    int jpt=0;
		    if (ascending) {jpt=j; }else{jpt=nPoints-j-1;}
		    heights[i+1][j+1] = data[jpt*x_size + i];
		    errors[i+1][j+1] = Math.sqrt(heights[i+1][j+1]);
		}
	    }
	    scan_data.setContents(heights, errors, null, null, null, null, null);
	}
	    return;

	}
    }

    private void readSliceData(IStream s, int size, short width) throws java.io.IOException {
	    int nPoints = getPoints().getNPoints();
	    long expectedSize=2048 * nPoints;
	    	
	    Histogram2D scan_data = setHistSize(Parameters.nChannelModule);

	    if (size != expectedSize)
		throw new java.io.StreamCorruptedException("Incorrect data size. Have a size of: " + size + " but have " + nPoints + 
							   "scan points which implies a size of: " + expectedSize);	    
	    
	    boolean ascending=getPoints().ascending();
	    switch (width) {
		case WIDTH_16BIT:{
		    short[] data = s.readShortArray("16BitSliceData");
		    		    
		    if (data.length != expectedSize) {			
			throw new java.io.StreamCorruptedException("Incorrect data size.  Got: " + data.length + " but expected: " + expectedSize);
		    }
		    
		    double [][] heights = new double[Parameters.nChannelModule+2][nPoints+2];
		    double [][] errors = new double[Parameters.nChannelModule+2][nPoints+2];
		    		    
		    for (int i=0; i<Parameters.nChannelLink; ++i) {
			for (int j=0; j<nPoints; ++j) {
			    int jpt=0;
			    if (ascending) {jpt=j; }else{jpt=nPoints-j-1;}
			    heights[i+1][j+1] = data[jpt*2048 + i];
			    errors[i+1][j+1] = Math.sqrt(heights[i+1][j+1]);
			    
			    heights[i+Parameters.nChannelLink+1][j+1] = data[jpt*2048 + i+1024];
			    errors[i+Parameters.nChannelLink+1][j+1] = Math.sqrt(heights[i+Parameters.nChannelLink+1][j+1]);
			}
		    }
		    
		    scan_data.setContents(heights, errors, null, null, null, null, null);
		}
		return;
		
		case WIDTH_32BIT: {
		    int[] data = s.readIntArray("32BitSliceData");
		    		    
		    if (data.length != expectedSize) {			
			throw new java.io.StreamCorruptedException("Incorrect data size.  Got: " + data.length + " but expected: " + expectedSize);
		    }
		    
		    double [][] heights = new double[Parameters.nChannelModule+2][nPoints+2];
		    double [][] errors = new double[Parameters.nChannelModule+2][nPoints+2];
		    		    
		    for (int i=0; i<Parameters.nChannelLink; ++i) {
			for (int j=0; j<nPoints; ++j) {
			    int jpt=0;
			    if (ascending) {jpt=j; }else{jpt=nPoints-j-1;}
			    heights[i+1][j+1] = data[jpt*2048 + i];
			    errors[i+1][j+1] = Math.sqrt(heights[i+1][j+1]);
			    
			    heights[i+Parameters.nChannelLink+1][j+1] = data[j*2048 + i+1024];
			    errors[i+Parameters.nChannelLink+1][j+1] = Math.sqrt(heights[i+Parameters.nChannelLink+1][j+1]);
			}
		    }
		    
		    scan_data.setContents(heights, errors, null, null, null, null, null);
		}
		return;
	    }
    }
    
    //Read in ROOT data - copy into arrays for internal rep
    //Ignore under and overflow bins.
    private void readRootData(IStream s, int size, short width) throws java.io.IOException {
	
        originalType = s.readShort("OriginalType");

	int nPoints = getPoints().getNPoints();
	//System.out.println(" READ n points = "+nPoints);
	//System.out.println(" READ size = "+size);

	int quotient = (nPoints+2);
	if (quotient==0 || size % quotient != 0 ){
	    throw new java.io.StreamCorruptedException("Cannot divide data of size " + 
          	       size + " into " + quotient + " bins!");
	}
	int x_size = (size / quotient - 2)*2;
	Histogram2D scan_data = setHistSize(x_size);

        double [] link0Data = s.readDoubleArray("Link0Data");
        double [] link1Data = s.readDoubleArray("Link1Data");
        
        if (link0Data.length != link1Data.length || link0Data.length != (x_size/2 + 2) * (nPoints+2) ) {
            throw new java.io.StreamCorruptedException("Data arrays are not correct length.  Expected: " + 
                      (x_size/2 + 2) * (nPoints+2) + " Link0Data length: " + 
                      link0Data.length + " Link1Data length: " + link1Data.length);
        }
        
        double [][] heights = new double[x_size+2][nPoints+2];
        double [][] errors = new double[x_size+2][nPoints+2];

        IAxis xAxis = scanData.xAxis();
        IAxis yAxis = scanData.yAxis();
        
        for (int i=0; i<x_size; ++i) {
            for (int j=0; j<nPoints; ++j) {
                heights[i+1][j+1] = i<x_size/2 ? link0Data[(j+1)*(x_size/2+2)+i            + 1] : 
		                                 link1Data[(j+1)*(x_size/2+2)+i%(x_size/2) + 1];
                errors[i+1][j+1] = Math.sqrt(heights[i+1][j+1]);
            }
        }
                
        scan_data.setContents(heights, errors, null, null, null, null, null);
    }
    
    private static final short SLICE = (short)2;
    private static final short SCURVE = (short)4;
    private static final short SLICE_COMPRESSED = (short)6;
    private static final short SCURVE_COMPRESSED = (short)8;
    private static final short FIT = (short)10;
    private static final short HITS_PER_EVENT = (short)12;
    private static final short ROOTHIST = (short)14;
    private static final short RAWHIST = (short)16;
    
    private static final short WIDTH_64BIT = (short)64;
    private static final short WIDTH_32BIT = (short)32;
    private static final short WIDTH_16BIT = (short)16;
    
    private IHistogram2D scanData;
    private short originalType;
}
