package SctData;

import Sct.*;
import java.util.*;

/**
 * This class represents a possible defect with a module.
 * A ModuleDefect has a description and can be classified as severe or not severe.
 * @todo Is this enough?  Do we need to add more classes here?  Should we be able to add
 * other ModuleDefect s anywhere in our code or is having a central repository here enough?
 * Should this class be renamed ChannelDefect?	
 * @author Matthew Palmer
 */
public class ModuleDefect implements Streamable, Cloneable {       
    
    private static HashMap defectMap = new HashMap(20);
    
    //@{ @name Fit Defects
    public static ModuleDefect DEAD = new ModuleDefect(1, true, "DEAD", "Dead channel - output always <1%", 0.01);                      ///< Output always v small
    public static ModuleDefect STUCKON = new ModuleDefect(2, true, "STUCKON", "Dead channel - output always > 98% occupancy", 0.98);    ///< Output always v high
    public static ModuleDefect OVER = new ModuleDefect(4, false, "OVER", "Occupancy greater than 100%", 1.);                            ///< Occupancy larger than max
    public static ModuleDefect UNDER = new ModuleDefect(3, true, "UNDER", "Occupancy never reaches 100%", 1.);                          ///< Occupancy never reaches mas
    public static ModuleDefect NOINIT = new ModuleDefect(18, true, "NOINIT", "Couldn't initialize fit", 0);                             ///< Couldn't initialize the fit.
    public static ModuleDefect FIT_UNDER = new ModuleDefect(5, true, "FIT_UNDER", "Fitted mean out of bounds low", 0);
    public static ModuleDefect FIT_OVER = new ModuleDefect(6, true, "FIT_OVER", "Fitted mean out of bounds high", 0);
    public static ModuleDefect SIG_UNDER = new ModuleDefect(7, true, "SIG_UNDER", "Fitted sigma out of bounds low", 0);
    public static ModuleDefect SIG_OVER = new ModuleDefect(8, true, "SIG_OVER", "Fitted sigma out of bounds high", 0);
    //@}
    ///{ @name NPt Gain Defects
    public static ModuleDefect LO_GAIN = new ModuleDefect(9, true, "LO_GAIN", "Gain is less than 0.75*chip average", 0.75);             ///< Gain < 0.75 * chip average
    public static ModuleDefect HI_GAIN = new ModuleDefect(10, true, "HI_GAIN", "Gain is higher than 1.25*chip average", 1.25);          ///< Gain > 1.25 * chip average
    public static ModuleDefect LO_OFFSET = new ModuleDefect(11, true, "LO_OFFSET", "Offset is less than -100", -100);                   ///< Offset < -100
    public static ModuleDefect HI_OFFSET = new ModuleDefect(12, true, "HI_OFFSET", "Offset is higher than 12-", 120);                   ///< Offset > 120
    public static ModuleDefect UNBONDED = new ModuleDefect(13, true, "UNBONDED", "Noise is less than 750", 750);                        ///< Noise <= 750
    public static ModuleDefect PARTBONDED = new ModuleDefect(14, false, "PARTBONDED", "Noise is less than 1100", 1100);                 ///< Noise <= 1100
    public static ModuleDefect NOISY = new ModuleDefect(15, true, "NOISY", "Noise is > 1.15*chip average", 1.15);                       ///< Noise > 1.15* av chip noise
    ///}
    ///{ @name Trim Defects
    public static ModuleDefect TR_RANGE = new ModuleDefect(22, false, "TR_RANGE","Unusual chip trim step size",0.);                     ///< Unexpected chip trim step size
    public static ModuleDefect TR_STEP = new ModuleDefect(23, false, "TR_STEP","Channel trim step size different from chip",4.);        ///< Channel step different from chip
    public static ModuleDefect TR_OFFSET = new ModuleDefect(24, false, "TR_OFFSET","Channel trim range offset different from chip",4.); ///< Channel offset different from chip
    public static ModuleDefect TR_NOTRIM = new ModuleDefect(25, true, "TR_NOTRIM","Untrimmable channel",0.);                            ///< Untrimmable channel
    ///}
    ///{ @name Other Defects
    public static ModuleDefect NO_HI = new ModuleDefect(19, true, "NO_HI", "High noise occupancy", 0.0005);                             /// high noise occupancy
    public static ModuleDefect MEAN_ERROR = new ModuleDefect(16, true, "MEAN_ERROR", "Error in the mean", 0);
    public static ModuleDefect SIG_ERROR = new ModuleDefect(17, true, "SIG_ERROR", "Error in the sigma", 0);
    public static ModuleDefect STUCK_CELL = new ModuleDefect(21, false, "STUCK_CELL","Stuck cell in the pipeline",0.);                  ///< pipeline
    public static ModuleDefect DEAD_CELL = new ModuleDefect(20, false, "DEAD_CELL","Dead cell in the pipeline",0.);                     ///< pipeline

    
    /** Public constructor: Copy the given ModuleDefect and give ModuleElement */
    public ModuleDefect(ModuleDefect prototype, ModuleElement element) {
        this.id = prototype.id;
        this.severe = prototype.severe;
        this.name = prototype.name;
        this.description = prototype.description;
        this.parameter = prototype.parameter;
        this.element = element;
    }    

    /** Return true if this ModuleDefect is severe. */
    public boolean isSevere() {
        return severe;
    }

    /** Gets the module element affected by this defect. */
    public ModuleElement getModuleElement() {
        return element;
    }

    /** True if this is a prototype, false if it actually represents a defect. @todo is this necessary? */
    public boolean isPrototype() {
        return element == null;
    }

    /** returns true if defect is serious enough to prevent fitting to this channel. */
    public boolean isUnfittable() {
        return isSameTypeAs(OVER) || isSameTypeAs(UNDER) || isSameTypeAs(STUCKON);        
    }

    /** SCTDAQ definitition of unusable is dead or stuckon. */
    public boolean isUnuseable() {
        return isSameTypeAs(DEAD) || isSameTypeAs(STUCKON);
    }

    /** SCTDAQ definitition of `dodgy' is any defect other than dead or stuckon. */
    public boolean isDodgy() {
        return !(isSameTypeAs(DEAD) || isSameTypeAs(STUCKON));
    }

    /** Gets the name of this defect */
    public String getName() {
        return name;
    }

    /** Get a human readable description for this ModuleDefect */
    public String getDescription() {
        return description;
    }

    /** Gets a parameter that can be used to determine if something is defective. */
    public double getParameter() {
        return parameter;
    }

    /**
    Comparison operators 
    @{  Returns true if the defect types are the same.
    AND defects refer to the same channels.
    */
    public boolean isSameTypeAs(ModuleDefect defect) {
        return id == defect.id;
    }

    /**
     * Returns true if the defect types are the same.
     * Returns false if either is a prototype.
     * @note defects need NOT refer to the same (physical) channels.
    */
    public boolean equals(Object o) {
        if (o instanceof ModuleDefect) {
            ModuleDefect d = (ModuleDefect)o;
            if (d.isPrototype() || this.isPrototype()) return false;
            return id == d.id && element.equals(d.element);
        }
        return false;
    }

    public String getClassName() {
        return "SctData.ModuleDefect";
    }
    
    /**
     * For I/O use
     */
    public static ModuleDefect read(IStream s, ObjectManager o) throws java.io.IOException {
        int id = s.readInt("Id");
        ModuleDefect proto = (ModuleDefect)defectMap.get(new Integer(id));
        if (proto == null) throw new java.io.StreamCorruptedException("Can't create a ModuleDefect for id: "+ id);
        
        ModuleElement element = (ModuleElement)o.readObject(s, "Element", "SctData.ModuleElement");
        ModuleDefect d = new ModuleDefect(proto, element);
        
        return d;
    }    
    
    public void write(OStream s, ObjectManager o) throws java.io.IOException {
        s.writeInt("Id", id, false);
        o.writeObject(s, "Element", element, false);
    }       
    
    /** Private constructor */
    private ModuleDefect(int id, boolean severe, String name, String d, double parameter) {
        this.id = id;
        this.severe = severe;
        this.name = name;
        this.description = description;
        this.parameter = parameter;
        defectMap.put(new Integer(id), this);
    } 
        
    private int id;
    private boolean severe;
    private String name;
    private String description;
    private double parameter;
    private ModuleElement element;    
}
