/*
 * ChipConfiguration.java
 *
 * Created on 16 September 2003, 16:35
 */

package SctData;

import Sct.*;
import sctConf.*;

/**
 *
 * @author  Matthew Palmer
 */
public class ChipConfiguration implements Streamable {
    
    public ChipConfiguration(ABCDChip data) {
        this.data = data;
        options = new ChipOptions(data.basic.config);
    }
    
    public ABCDChip getData() {
        return data;
    }
    
    public ChipOptions getOptions() {
        return options;
    }
    
    /// @return true if chip is participating in a scan
    public boolean isActive() {
        return data.active != 0;
    }
    
    /// set the chip activity status @param active if set true
    public void setActive(boolean active) {
        data.active = active ? (byte)1 : (byte)0;
    }
    
    /** There is a had-coded address associated with each chip position on the module.
	@return the address.
    */
    public byte getAddress() {
        return data.address;
    }
    
    /// @return the trim target
    public byte getTrimTarget() {
        return data.target;
    }
    
    /// set the trim target @param target in mV?
    public void setTrimTarget(byte target) {
        data.target = target;
    }
    
    /// get the trim  @param channel 0->127 @return trim 0->15
    public byte getTrim(int channel) {
        return data.trim[channel];
    }
    
    ///set the trim @param channel 0->127 @param trim 0->15
    public void setTrim(int channel, byte trim) throws IllegalArgumentException {
        if (trim > 15 || trim < 0) throw new IllegalArgumentException("Trim must be 0->15, channel: " + channel + " trim: " + trim);
        data.trim[channel] = trim;
    }
    
    /// get the value of the threshold DAC setting?
    public byte getThreshold() {
        return data.basic.vthr;
    }
    
    /// set the threshold DAC setting - 0->255
    ///No need for exception as threshold can take full range of byte values
    public void setThreshold(byte threshold) {        
        data.basic.vthr = threshold;
    }
    
    /// get the value of the calibration charge DAC setting
    public byte getCalCharge() {
        return data.basic.vcal;
    }
    
    /// set the value of the calibration charge DAC setting
    ///No need for exception as calCharge can take full range of byte values
    public void setCalCharge(byte calCharge) {        
        data.basic.vcal = calCharge;
    }
    
    /// get the value of the strobe delay DAC setting.
    public byte getStrobeDelay() {
        return data.basic.delay;
    }
    
    /// set the value of the strobe delay DAC setting
    public void setStrobeDelay(byte strobeDelay) throws IllegalArgumentException {
        if (strobeDelay > 63 || strobeDelay < 0) throw new IllegalArgumentException("delay must be 0->63, delay: " + strobeDelay);
        data.basic.delay = strobeDelay;
    }
    
    
    /// unmask all channels
    public void resetMask() {
       for (int i=0; i<data.basic.mask.length; ++i) {
           data.basic.mask[i] = 0;
       }
    }
    
    /// mask mutator @param channel index 0->127 @param mask if true, unmask if false
    public void setMask(int ichannel, boolean value) {
        if (value) mask(ichannel); 
        else unmask(ichannel);
    }
    
    /// mask mutator @param channel index 0->127
    public void mask(int ichannel) {
        data.basic.mask[ichannel/32] &= ~(1<<ichannel%32);
    }
    
    /// mask mutator @param channel index 0->127
    public void unmask(int ichannel) {
        data.basic.mask[ichannel/32] |= (1<<ichannel%32);
    }
    
    /// mask access @return true if channel is masked @param channel index 0->127
    public boolean isMasked(int ichannel) {
        return (data.basic.mask[ichannel/32] & (1<<ichannel%32)) == 0;
    }
    
     /** A functional representation of the response
     * curve (equivalent threshold as a function of charge)
     * so that one may request ROD to set the threshold to
     * a value specified in fC. @{ get the index (type) of the RC function
     *  @return 0 - no calibration information
     *  1 - second order polynomial
     *  2 - "grillo" function
     *  3 - exponential
     *  4 - straight line fit */
    public byte getRcFunctionIndex() {
        return data.caldata.rc_function;
    }
    
    /// Response curve function index. @param 
    public void setRcFunctionIndex(byte index)  throws IllegalArgumentException {
        if (index > 4 || index <= 0) throw new IllegalArgumentException("rcFunction must be 1->4, rcFunction: " + index);
        data.caldata.rc_function = index;        
    }
    
    ///	Get the response curve parameter @param index of parameter 0->2; @return value of parameter
    public double getRcParam(int ipar) {
        return data.caldata.rc_params[ipar];
    }
    
    /**
       Set the response curve parameter @param index of parameter 0->2;
       @param the value to set that parameter
    */
    public void setRcParam(int ipar, double val) {
        data.caldata.rc_params[ipar] = (float)val;
    }
    
    ///@}

    /**  
	 theshold calibration factor by which the nominal threshold must be multiplied
	 @{ mutator @param the factor
    */
    public void setCalFactor(float factor) {
        data.caldata.c_factor = factor;
    }
    
    /// accessor @}
    public float getCalFactor() {
        return data.caldata.c_factor;
    }    
    
    public String getClassName() {
        return "SctData.ChipConfiguration";
    }
    
    public static Streamable read(IStream s, ObjectManager o) throws java.io.IOException {
        ABCDChip data = new ABCDChip();
        data.basic = new ABCDBasic();
        data.caldata = new ABCDCaldata();        
        ChipConfiguration c = new ChipConfiguration(data);
        c.readObject(s, o);
        return c;
    }
    
    public void readObject(IStream s, ObjectManager o) throws java.io.IOException {        
        data.active = s.readByte("Active");
        data.address = s.readByte("Address");
        data.target = s.readByte("Target");
        
        options = (ChipOptions)o.readObject(s, "ChipOptions", "SctData.ChipOptions");
        data.basic.config = options.getData();
        
        setThreshold(s.readByte("Threshold"));
        setCalCharge(s.readByte("CalCharge"));
        setStrobeDelay(s.readByte("Delay"));
        data.basic.preamp = s.readByte("Preamp");
        data.basic.shaper = s.readByte("Shaper");
        
        data.basic.mask = s.readIntArray("Masks");
        
        data.caldata.rc_function = s.readByte("RcFunction");
        data.caldata.rc_params = s.readFloatArray("RcParams");
        data.caldata.c_factor = s.readFloat("CalFactor");
        
        data.trim = s.readByteArray("Trims");        
    }
    
    public void write(OStream s, ObjectManager o) throws java.io.IOException {        
        s.writeByte("Active", data.active, false);
        s.writeByte("Address", data.address, false);
        s.writeByte("Target", data.target, false);        
        
        o.writeObject(s, "ChipOptions", options, false);
        
        s.writeByte("Threshold", getThreshold(), false);
        s.writeByte("CalCharge", getCalCharge(), false);
        s.writeByte("Delay", getStrobeDelay(), false);
        s.writeByte("Preamp", data.basic.preamp, false);
        s.writeByte("Shaper", data.basic.shaper, false);
        
        s.writeIntArray("Masks", data.basic.mask, false);
        
        s.writeByte("RcFunction", data.caldata.rc_function, false);
        s.writeFloatArray("RcParams", data.caldata.rc_params);
        s.writeFloat("CalFactor", data.caldata.c_factor);
        
        s.writeByteArray("Trims", data.trim, false);
    }
    
    //Data
    ABCDChip data;
    ChipOptions options;
}
