#include "Streamer_Test.h"
#include "../src/ResponseCurve.h"
#include "../src/RawScanResultIOHelper.h"
#include "../src/RawScanResult.h"
#include "../src/FitScanResult.h"
#include "../src/TopHatFitObject.h"
#include "../src/DefaultVariable.h"
#include "../src/NullVariable.h"
#include "../src/mVThresholdVariable.h"
#include "../src/DcsData.h"
#include "../src/TrimRangeTestResult.h"
#include "Sct/IOManagerDummy.h"
#include "Sct/Archive/IOManagerArchiveFile.h"
#include "Sct/Archive/IONameArchiveFile.h"
#include "Sct/SctNames.h"
#include <boost/test/unit_test.hpp>
#include "ScanResultWriter/dataTypes.h"
#include "TH2.h"

using boost::unit_test_framework::test_suite;
using namespace Sct;
using namespace Sct::Archive;

namespace SctData{

  void checkStreaming(const Streamable& s){
    BOOST_CHECK_EQUAL( IOManagerDummy::instance().testStreaming1(s),
		       IOManagerDummy::instance().testStreaming2(s) );
  }

  void testLinearRC(){
    LinearResponseCurve ob;
    checkStreaming(ob);
  }

  RawScanResult createRaw(){
    const ResultHeader header(43,22,"not_a_module",mVThresholdVariable::instance());
    const ModuleConfiguration config;
	  
    const float pts[4] = {1., 2., 4., 8.};
    const unsigned nevts[4] = {12, 12, 12, 13};
    const unsigned nerr[4] = {0, 1, 0, 0};
    const ScanPoints points(4, pts, nevts, nerr);
    
    TH2D* link0 = new TH2D("link0","data link 0", 768, -0.5, 767.5, 4,-0.5, 3.5);
    TH2D* link1 = new TH2D("link1","data link 1", 768, -0.5, 767.5, 4,-0.5, 3.5);
    link0->FillRandom("gaus",500);
    link1->FillRandom("gaus",400);
    
    RawScanResult raw(header, config, points, *link0, *link1);
    RawScanResultIOHelper helper; helper.set(raw);
    helper.setDataType(SR_DT_ROOTHIST);
    return raw;
  }
  
  void testRaw(){
    checkStreaming(createRaw());
  }
  
  void testArchRaw(){
    IOManagerArchiveFile::instance().read("./SctData::RawScanResult.1008.0.20220330200020.gz");
  }

  void testArchFile(){
    RawScanResult ob = createRaw();
    BOOST_CHECK(system("rm -f $SCT_PERSISTENT_DIR/SctData::RawScanResult.22.43.not_a_module.gz") == 0 );
    IOManagerArchiveFile::instance().write(ob);
    //cout << "written to ArchFile!" << endl;
    IONameArchiveFile name(ob.getUniqueID(),ob.getClassName());
    IOManagerArchiveFile::instance().read(name.getIOName());
    BOOST_CHECK_THROW(IOManagerArchiveFile::instance().write(ob), IoException);
  }
  
  void testFitted1(){
    RawScanResult raw = createRaw();
    FitScanResult fit (raw.getHeader(), 
		       raw.getConfiguration(),
		       raw.getPoints());
    checkStreaming(fit);
    fit.initializeChannelFits(TopHatFitObject());
    checkStreaming(fit);
  }
  void testFitted2(){
    RawScanResult raw = createRaw();
    FitScanResult fit (raw.getHeader(), 
		       raw.getConfiguration(),
		       raw.getPoints());
    BOOST_CHECK(system("rm -f $SCT_PERSISTENT_DIR/SctData::FitScanResult.22.43.not_a_module.gz") == 0 );

    IOManagerArchiveFile::instance().write(fit);
    IONameArchiveFile name(fit.getUniqueID(),fit.getClassName());
    IOManagerArchiveFile::instance().read(name.getIOName());
  }

  void testFitted3(){
    IOManagerArchiveFile::instance().read("./SctData::FitScanResult.1063.1.20220330200020.gz");
  }

  void testTrimRange(){
    shared_ptr<const Serializable> ob = IOManagerArchiveFile::instance().read("./SctData::TrimRangeTestResult.551.16.20220330200020.gz");
    checkStreaming(*ob);
  }

  void testStrobeDelay(){
    shared_ptr<const Serializable> ob = IOManagerArchiveFile::instance().read("./SctData::StrobeDelayTestResult.1039.0.20220330200020.gz");
    checkStreaming(*ob);
  }

  shared_ptr<DcsData> getDcsData(){
    shared_ptr<DcsData> dcs(new DcsData());
    dcs->setVcc(10.);
    dcs->setVdd(1.5);
    dcs->setIcc(3.);
    dcs->setVcc(4.);
    dcs->setT0(5.);
    dcs->setT1(7.);
    dcs->setVbias(8.);
    dcs->setIbias(9.);
    return dcs;
  }

  void testDcsData(){
    checkStreaming(*getDcsData());
  }

  void testTestDataWithDcs(){
    shared_ptr<const Serializable> ob = IOManagerArchiveFile::instance().read("./SctData::StrobeDelayTestResult.1039.0.20220330200020.gz");
    shared_ptr<const TestResult> r =  dynamic_pointer_cast<const TestResult>(ob);
    TestResult* r2 = const_cast<TestResult*> (r.get());
    r2->setDcsData(getDcsData());

    checkStreaming(*r2);
  }

  void testOldStreamers(){
    string old_name = "SctData::FitScanResult.22.43.old_not_a_module.gz";
    string cmd = "cp -fv ";
    cmd += old_name;
    cmd += " $SCT_PERSISTENT_DIR";
    BOOST_CHECK(system(cmd.c_str()) ==0);
    IOManagerArchiveFile::instance().read(old_name);
  }

  void testConfigurationVariable(){
    checkStreaming( *DefaultVariable::instance(56) );
    checkStreaming( NullVariable::instance() );
    checkStreaming( mVThresholdVariable::instance() );
  }
  
  Streamer_Test::Streamer_Test() {
    string cmd = "rm -f " + SctNames::getPersistentDir() + "/22.43.not_a_module.SctData::RawScanResult.gz";
    system(cmd.c_str());      
    add(BOOST_TEST_CASE(&testLinearRC));
    add(BOOST_TEST_CASE(&testConfigurationVariable));
    add(BOOST_TEST_CASE(&testRaw));
    add(BOOST_TEST_CASE(&testArchRaw));
    add(BOOST_TEST_CASE(&testArchFile));
    add(BOOST_TEST_CASE(&testFitted1));
    add(BOOST_TEST_CASE(&testFitted2));
    add(BOOST_TEST_CASE(&testFitted3));
    add(BOOST_TEST_CASE(&testTrimRange));
    add(BOOST_TEST_CASE(&testOldStreamers));
    add(BOOST_TEST_CASE(&testStrobeDelay));
    add(BOOST_TEST_CASE(&testDcsData));
    add(BOOST_TEST_CASE(&testTestDataWithDcs));
  }
}
