#ifndef SCT_STREAMER_H
#define SCT_STREAMER_H

#include <boost/shared_ptr.hpp>
#include "OStream.h"
#include "IStream.h"
#include "IOManager.h"
#include "Streamable.h"

//All these #includes for convenience

using boost::shared_ptr;

namespace Sct {
  /**
     A streamer converts to and from a streamed representation of a data object.
     The data object must impliment Streamable.
     The actual representation of the streamed data (e.g. file, IS, ... )
     is decided by the sub-class of IStream or OStream, which is provided by the
     sub-class of IOManager.
     <P>Streamers can make use of other streamers, but <B>MUST</B> use the IOManager to find the streamer.
     The (e.g.) RawScanResultStreamer must <B>not</b> do ScanResultStreamer.write(ob);
     RawScanResultStreamer can use ScanResultStreamer, not directly, but
     by calling manager.getStreamer("SctData::ScanResult").write(ob);
   */
class Streamer {
public:
    virtual ~Streamer() {}
    
    /**
    Publishes ob to out.
    @param out The OStream.
    @param ob The Streamable object to stream.
    @param manager the IOManager to use
    @throw IoException if something bad happens
    */
    virtual void write(OStream& out, const Streamable& ob, const IOManager& manager) const = 0;

    /**
    Create a new Streamable object of the appropriate type from IS.
    @param in The IStream.
    @param manager the IOManager to use
    @return a new Streamable object
    @throw IoException if something bad happens    
    */
    virtual shared_ptr<Streamable> read(IStream& in, const IOManager& manager) const = 0;

    /**
    Refreshes ob from in.
    @param in The IStream
    @param ob The Streamable object to be refreshed.
    @param manager the IOManager to use
    @throw IoException if something bad happens    
    */
    virtual void read(IStream& in, Streamable& ob, const IOManager& manager) const = 0;

    virtual unsigned getVersion() const throw()= 0;
};

}

#endif //SCT_STREAMER_H
