#ifndef SCT_RANGEDVECTOR_H
#define SCT_RANGEDVECTOR_H

#include <vector>
#include <string>
#include <sstream>
#include "LogicErrors.h"
#include "OutOfRangeError.h"

using std::vector;
using std::string;

namespace Sct {

/**
 * Class is like a vector, but checks the array bounds of the vector. 
 * @throw LogicError if array bounds are violated.
 * @note does not range check if NDEBUG is defined
 */
template <class T>
class RangedVector : public vector<T> {
public:
    /// Constructor @param the name of the vector.
    RangedVector(string name) : name(name) {}
    
    /// over-write vector operator[]
    ///@throw LogicException if out of range
    T& operator[] (const unsigned i) ;
    
    /// over-write vector operator[] l-value
    ///@throw LogicException if out of range
    const T& operator[] (const unsigned i) const ;
    
    /// asignment
    RangedVector<T>& operator=(const vector<T>& copyme);
    
protected:
    RangedVector() throw();                              ///< not allowed to do this
    OutOfRangeError<int> createException(unsigned int i)const;  ///< creates the message text
    string name;                                         ///< the name of the vector is used in the error message
};

template <class T>
inline OutOfRangeError<int> RangedVector<T>::createException(unsigned int i) const {
    std::ostringstream text;
    text << name << " [] ";
    return OutOfRangeError<int>(text.str(), __FILE__, __LINE__, i,0,this->size()-1);
}

template <class T>
inline T& RangedVector<T>::operator[] (const unsigned i) {
#ifndef NDEBUG
    if (i>this->size()) {
        throw createException(i);
    }
#endif
    return vector<T>::operator[](i);
}

template <class T>
inline const T& RangedVector<T>::operator[] (const unsigned i) const {
#ifndef NDEBUG
    if (i>this->size()) {
        throw createException(i);
    }
#endif
    return vector<T>::operator[](i);
}

template <class T>
inline RangedVector<T>& RangedVector<T>::operator=(const vector<T>& copyme) {
    this->vector<T>::operator=(copyme);
    return *this;
}

}
#endif //SCT_RANGEDVECTOR_H
