#ifndef SCT_OUTOFRANGEERROR_H
#define SCT_OUTOFRANGEERROR_H

#include "LogicErrors.h"

namespace Sct {

/**
 * Thrown to show that arguement value is not in the allowed range.
 * Templated class: the type of class (e.g. int) which has gone out of range.
 * This class must be able to << to an ostream (e.g. {int, long, short, unsigned} )
 * @ingroup Exceptions    
 * @author Alan Barr
 */
template <class T=int>
class OutOfRangeError : public LogicError {
public: 
    /**
      Creates an Exception with message msg and no cause
     * @param the value of the paramter (templated)
     * @param the lower limit of the parameter (templated)
     * @param the upper limit of the parameter (templated)      
      */
    OutOfRangeError(const string& msg, const string& file, int line, T val, T lo, T hi) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
     * @param the value of the paramter (templated)
     * @param the lower limit of the parameter (templated)
     * @param the upper limit of the parameter (templated)      
      */
    OutOfRangeError(Throwable& cause, const string& file, int line, T val, T lo, T hi) throw();
    
    /**
      Creates an Exception with a message and a cause
     * @param the value of the paramter (templated)
     * @param the lower limit of the parameter (templated)
     * @param the upper limit of the parameter (templated)      
      */
    OutOfRangeError(const string& msg, Throwable& cause, const string& file, int line, T val, T lo, T hi) throw();    
    
    ///Get the value that was requested
    T getValue() const throw();
    ///Get the lower limit for the parameter
    T getLowLimit() const throw();
    ///Get the upper limit for the parameter
    T getHighLimit() const throw();
    
    virtual shared_ptr<Throwable> clone() const throw();
    
protected:
    OutOfRangeError(T val, T lo, T hi) throw() {}
    virtual string getMessage() const throw();
   
    T val, lo, hi;
    
};


template <class T>
OutOfRangeError<T>::OutOfRangeError(const string& msg, const string& file, int line, T val, T lo, T hi) throw()
        : val(val), lo(lo), hi(hi) {
    initialize("BAD_RANGE", "Sct::OutOfRangeError", msg, 0, file, line);
}

template <class T>
OutOfRangeError<T>::OutOfRangeError(Throwable& cause, const string& file, int line, T val, T lo, T hi) throw() 
        : val(val), lo(lo), hi(hi) {
    initialize("BAD_RANGE", "Sct::OutOfRangeError", "", &cause, file, line);
}
    
template <class T>   
OutOfRangeError<T>::OutOfRangeError(const string& msg, Throwable& cause, const string& file, int line, T val, T lo, T hi) throw()
       : val(val), lo(lo), hi(hi) {
    initialize("BAD_RANGE", "Sct::OutOfRangeError", msg, &cause, file, line);
}

template <class T>
string OutOfRangeError<T>::getMessage() const throw() {
    std::ostringstream output;
    output << msg << " [ value=" << val
    << " allowed range: "
    << lo << " -> " << hi << " ]";

    return output.str();
}

template <class T>
T OutOfRangeError<T>::getValue() const throw() {
    return val;
}

template <class T>
T OutOfRangeError<T>::getLowLimit() const throw() {
    return lo;
}

template <class T>
T OutOfRangeError<T>::getHighLimit() const throw() {
    return high;
}

template <class T>
shared_ptr<Throwable> OutOfRangeError<T>::clone() const throw() {
    return shared_ptr<Throwable> ( new OutOfRangeError<T>(*this) );
}

}
#endif //SCT_OUTOFRANGEERROR_H
