#ifndef SCT_LOGICERRORS_H
#define SCT_LOGICERRORS_H

#include "Exception.h"

namespace Sct {

/**
 * Logic errors represent internal problems in the logic of the program. 
 * These are (in theory) preventable, and even detectable before the program runs.
 * @ingroup Exceptions     
 * @author Alan Barr
 */
class LogicError : public Error {
public:   
    /**
      Creates an Exception with message msg and no cause
      */
    LogicError(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    LogicError(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    LogicError(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    LogicError() throw() {}    
};


/**
 * Thrown to report invalid arguments to functions.
 * @ingroup Exceptions     
 * @author Alan Barr
 */
class InvalidArgumentError : public LogicError {
public:    
    /**
      Creates an Exception with message msg and no cause
      */
    InvalidArgumentError(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    InvalidArgumentError(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    InvalidArgumentError(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    InvalidArgumentError() throw() {}        
};

///For backward compatibility
typedef InvalidArgumentError InvalidArgument;


/**
  This error indicates that a class' invariants have been violated somehow.
  This indicates a serious bug.
  @ingroup Exceptions
  @author Matthew Palmer
  */
class InvariantViolatedError : public LogicError {
public:    
    /**
      Creates an Exception with message msg and no cause
      */
    InvariantViolatedError(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    InvariantViolatedError(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    InvariantViolatedError(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    InvariantViolatedError() throw() {}   
};



/**
  This error indicates that a class is not in an appropriate state for the request that was made of it.
  This indicates a serious bug.
  @ingroup Exceptions  
  @author Matthew Palmer
  */
class IllegalStateError : public LogicError {
public:    
    /**
      Creates an Exception with message msg and no cause
      */
    IllegalStateError(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    IllegalStateError(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    IllegalStateError(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    IllegalStateError() throw() {}   
};

}

#endif //SCT_LOGICERRORS_H
