#include "IOManagerIS.h"
#include "IStreamIS.h"
#include "OStreamIS.h"
#include "IONameIS.h"
#include "../Serializable.h"
#include "../LogicErrors.h"
#include "../SctNames.h"

#include <is/isinfo.h>

namespace Sct {
namespace IS {
 
namespace detail {
    
class ISAdapter : public ISInfo {
public:
    ISAdapter(const string& className, const IOManagerIS& manager) : 
	ISInfo(className.c_str()), manager(manager), writeOb(0), className(className) {}
    
    ISAdapter(const Serializable& writeOb, const IOManagerIS& manager) : 
	ISInfo(writeOb.getClassName().c_str()), manager(manager), writeOb(&writeOb) {}
    
    virtual void refreshGuts(ISistream& inIS) {
	IStreamIS in(inIS);
	readOb = boost::dynamic_pointer_cast<Serializable>(manager.readImpl(in));
    }
    
    virtual void publishGuts(ISostream& outIS) {
	//Break IS type checking
	if (dynamic_cast<ISType*>(&outIS)!=0 ) return; 
	if (!writeOb) {
	    throw InvariantViolatedError("IOManagerIS::ISAdapter Tried to write a null object", __FILE__, __LINE__);
	}
	OStreamIS out(outIS);
	manager.writeImpl(out, *writeOb, true);
    }
    
    shared_ptr<Serializable> getOb() {
	return readOb;
    }
    
private:
    const IOManagerIS& manager;
    const Serializable* writeOb;
    shared_ptr<Serializable> readOb;
    string className;
};
}	
  
using namespace detail;

IOManagerIS::IOManagerIS() throw() {}

IOManagerIS& IOManagerIS::instance() throw() {
    static IOManagerIS bertha;
    return bertha;
}

void IOManagerIS::write(const Serializable& ob, const IOParams* params) const throw(LogicError, IoError) {   
    IONameIS IOName(ob.getUniqueID(), ob.getClassName());
    if (params) {
	const IOParamsIS* p = dynamic_cast<const IOParamsIS*>(params);
	if (p) IOName.setServer(p->serverName);
    }
    
    ISAdapter adapter(ob, *this);
    
    ISInfoDictionary& id = SctNames::getISDictionary();
    ISInfo::Status result = id.insert(IOName.getIOName().c_str(), adapter);
    
    if (result != ISInfo::Success) {        
	throw IsException(result, "Error writing to IS server.  Couldn't write: " + IOName.getIOName(), __FILE__, __LINE__);
    }

}

shared_ptr<Serializable> IOManagerIS::read(const string& name, const IOParams* params) const throw(LogicError, IoError) {
    IONameIS IOName(name);
    ISAdapter adapter(IOName.getClassName(), *this);
    
    ISInfoDictionary& id = SctNames::getISDictionary();
    ISInfo::Status result = id.findValue(name.c_str(), adapter);

    if (result != ISInfo::Success) {
	throw IsException(result, "Error reading from IS server.  Couldn't read: " + name, __FILE__, __LINE__);	
    }

    return adapter.getOb();
}

shared_ptr<Serializable> IOManagerIS::read(const ISCallbackInfo& info) const throw(LogicError, IoError) {
    IONameIS IOName(info.name());
    ISAdapter adapter(IOName.getClassName(), *this);
    
    ISInfo::Status result = info.value(adapter);

    if (result != ISInfo::Success) {
        throw IsException(result, string("Error reading from IS server.  Couldn't read: ") + info.name(), __FILE__, __LINE__);	
    }

    return adapter.getOb();    
}

shared_ptr<Serializable> IOManagerIS::read(ISInfoIterator& it) const throw(LogicError, IoError) {
    IONameIS IOName(it.name());
    ISAdapter adapter(IOName.getClassName(), *this);
    
    ISInfo::Status result = it.value(adapter);

    if (result != ISInfo::Success) {
        throw IsException(result, string("Error reading from IS server.  Couldn't read: ") + it.name(), __FILE__, __LINE__);	
    }

    return adapter.getOb();    
}
    
}
}
