#ifndef EXCEPTION_H
#define EXCEPTION_H

#include <sstream>
#include <string>
#include <exception>

#include <mrs/message.h>
#include <boost/shared_ptr.hpp>

using std::string;
using boost::shared_ptr;

namespace Sct {
    
/**
 * The base class for all of the exceptions which we throw in the SCT code. 
 * Adds a method that reports the exception to MRS.
 * Also adds support for a cause.  This enables exceptions to be chained
 * and wrapped.
 *
 * @ingroup Exceptions
 * @author Alan Barr and refactored by Matthew Palmer
 */
class Throwable : public std::exception {
public:
    /**Send the message to the message reporting service.
     * @param s the SEVERITY of the exception (see mrs/message.h)
     * @return the MRS status information. (See mrs/message.h)
     */
    virtual long sendToMrs(SEVERITY s) throw() = 0;
    
    /**Send the message to the message reporting service.
     * @return the MRS status information. (See mrs/message.h)
     */
    virtual long sendToMrs() const throw() = 0;
         
    /**
      * Set the severity of this exception
      * @param s the SEVERITY of the exception (see mrs/message.h)
      */
    virtual void setSeverity(SEVERITY s) throw() = 0;
    
    /**
      * Get the severity
      */
    virtual SEVERITY getSeverity() const throw() = 0;
    
    /**
      Get the cause
      @return a pointer to the cause of 0 if no cause
      */
    virtual const Throwable* getCause() const throw() = 0;
    
    /**
      Force virtual destructors
      */
    virtual ~Throwable() throw() {}
    
    /**
      Necessary for chaining of exceptions unfortunately.
      Should return a copy of this allocated on the heap.
      It is necessary for sub-classes to override this if 
      they add new data members or if they do and special
      processing in what() or sentToMrs()      
      */
    virtual shared_ptr<Throwable> clone() const throw() = 0;
};

/** Main programs should call this to the Sct defaults
  * @param giving the process name as an argument makes
  * the terminate message easier to understand!
  *
  * @ingroup Exceptions
  */
void setExceptionHandlers(const char* process_name)  throw();

    
/**
  * A base implementation for Throwable
  * All important information should be included in the string 
  * returned by getMessage().
  * An exception should not rely on adding extra parameters to an MRS message
  * as if it is nested, it won't get this opportunity.
  *
  * If a sub-class has its own data members or overrides any methods
  * then it should override the clone() method also.
  * Most sub-classes will probably just provide constuctors in the same
  * pattern as shown for the Error and Exception classes.
  *
  * @ingroup Exceptions  
  */
class AbstractThrowable : public Throwable {
public:       
    virtual ~AbstractThrowable() throw() {}

    /**Send the message to the message reporting service.
     * If you override this you should also override the clone method.      
     * @param s the SEVERITY of the exception (see mrs/message.h)
     * @return the MRS status information. (See mrs/message.h)
     */
    virtual long sendToMrs(SEVERITY s) throw();
    
    virtual long sendToMrs() const throw();
         
    virtual void setSeverity(SEVERITY s) throw();
    
    virtual SEVERITY getSeverity() const throw();
    
    
    /**
      Override of std::exception::what().
      Returns hasCause() ? msg + "\ncaused by:\n" + cause.msg : msg; 
     * If you override this you should also override the clone method.      
      */
    virtual const char* what() const throw(); 
    
    virtual const Throwable* getCause() const throw();
    
    virtual shared_ptr<Throwable> clone() const throw();    
    
protected:
    /**
      To ease implementation in long exception hierarchies,
      sub-classes can provide a default constructor then must
      use the initialize method in all substansive constructors.
      */
    AbstractThrowable() throw();
    
    /**
      Gets the detail message for this.
      */
    virtual string getMessage() const throw();
    
    void initialize(const string& id, const string& name, const string& msg, Throwable* cause, const string& file, int line) throw() ;
    
    string id;
    string name;
    string msg;
    string file;
    int line;
    SEVERITY severity;
    shared_ptr<Throwable> cause;
    
    mutable string text;	//Holds output of what()
};

/**
 * Represents genuinely exceptional events (such as resource acquisition failure)
 * @ingroup Exceptions   
 * @author Alan Barr and refactored by Matthew Palmer
 */
class Exception : public AbstractThrowable {
public:  
    /**
      Creates an Exception with message msg and no cause
      */
    Exception(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    Exception(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    Exception(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    Exception() throw() {}
};

/**
  Represents problems that might be more reasonably described as bugs.
  Should probably not be caught except perhaps at top level.
  * @ingroup Exceptions    
  */
class Error : public AbstractThrowable {
public:   
    /**
      Creates an Exception with message msg and no cause
      */
    Error(const string& msg, const string& file, int line) throw();
    
    /**
      Creates an Exception with a cause.  Useful for simply wrapping
      exceptions.
      */
    Error(Throwable& cause, const string& file, int line) throw();
    
    /**
      Creates an Exception with a message and a cause
      */
    Error(const string& msg, Throwable& cause, const string& file, int line) throw();
    
protected:
    Error() throw() {}
};


} // end of namespace Sct

#endif // #ifndef EXCEPTION_H
