#include "Exception.h"
#include "SctNames.h"
#include "StdExceptionWrapper.h"
#include <mrs/message.h>
#include <sstream>
#include <cstdlib>
#include <iostream>

using namespace std;

namespace Sct {
    
AbstractThrowable::AbstractThrowable() throw() {
}
    
long AbstractThrowable::sendToMrs(SEVERITY s) throw() {
    setSeverity(s);
    return sendToMrs();
}

long AbstractThrowable::sendToMrs() const throw() {
    MRSStream& msg = SctNames::Mrs();
    msg << id << MRS_TEXT(what());
    msg << getSeverity() << ENDM;
    return msg.getState();
}

void AbstractThrowable::setSeverity(SEVERITY s) throw() {
    severity = s;
}

SEVERITY AbstractThrowable::getSeverity() const throw() {
    return severity;
}

const char* AbstractThrowable::what() const throw() {
    if (text.length() > 0) return text.c_str();
    
    ostringstream oss;
    oss << name;
    if (file.length() > 0) {
	oss << " created at: " << file << ":" << line;
    } 
    oss << endl << getMessage();
    if (cause.get()) {
	oss << endl << "Caused by: " << cause->what();
    }
    text = oss.str();
    return text.c_str();
}

string AbstractThrowable::getMessage() const throw() {
    return msg;
}
    
const Throwable* AbstractThrowable::getCause() const throw() {
    return cause.get();
}
    
shared_ptr<Throwable> AbstractThrowable::clone() const throw() {
    return shared_ptr<Throwable>(new AbstractThrowable(*this));
}
     
void AbstractThrowable::initialize(const string& id, const string& name, 
				   const string& msg, Throwable* cause, 
				   const string& file, int line) throw() {
    this->id = id;
    this->name = name;
    if (msg.length() > 0) 
	this->msg = msg;
    else 
	this->msg = "Wrapped around";
    
    if (cause) this->cause = cause->clone();
    this->file = file;
    this->line = line;
    severity = MRS_ERROR; 	//Default severity
}

//Exception methods    
Exception::Exception(const string& msg, const string& file, int line) throw() {
    initialize("EXCEPTION", "Sct::Exception", msg, 0, file, line);
}
    
Exception::Exception(Throwable& cause, const string& file, int line) throw() {
    initialize("EXCEPTION", "Sct::Exception", "", &cause, file, line);
}
    
Exception::Exception(const string& msg, Throwable& cause, const string& file, int line) throw() {
    initialize("EXCEPTION", "Sct::Exception", msg, &cause, file, line);
}
    

//Error methods   
Error::Error(const string& msg, const string& file, int line) throw() {
    initialize("ERROR", "Sct::Error", msg, 0, file, line);
}
    
Error::Error(Throwable& cause, const string& file, int line) throw() {
    initialize("ERROR", "Sct::Error", "", &cause, file, line);
}
    
Error::Error(const string& msg, Throwable& cause, const string& file, int line) throw() {
    initialize("ERROR", "Sct::Error", msg, &cause, file, line);    
}
    


//Stuff to deal with system-wide handling of exceptions
    
//Prototypes
void Terminate() throw();
void Unexpected() throw();
    
const char* process_name = 0;

void setExceptionHandlers(const char* name) throw() {
    process_name=name;
    std::set_unexpected ( &Unexpected );
    std::set_terminate  ( &Terminate  );
}


/**
* This function is called in case of exception violations
* It should NOT be called by the user.
* <P> It sends a message to MRS and calls terminate();
*/

void Terminate() throw() {
    static int callCount = 0;
    Throwable* mrs = 0;
    try {
        if (!callCount++) {
            throw;   // rethrow the last exception
        } else {
            cerr << "Damn annoying exception throw bug happened again" << endl;
        }
    } catch(Throwable &e) {
        e.sendToMrs(MRS_ERROR);
    } catch(std::exception &e) {
	mrs = new StdExceptionWrapper(e);
    } catch(std::string &s) {          // Just in case	
        std::ostringstream os;
        os<<"uncaught string expection "<<s;
        mrs = new Error(os.str(), __FILE__, __LINE__);
    } catch(...) {
        mrs = new Error("uncaught unknown exception", __FILE__, __LINE__);
    }
    if (mrs) {
        mrs->sendToMrs(MRS_ERROR);
    } else {
	std::ostringstream os;
	os<<"Exception::Terminate() " << string(process_name);
	Error die(os.str(), __FILE__, __LINE__);
	die.sendToMrs(MRS_ERROR);
    }
    exit(EXIT_FAILURE);
}


/**
* This function is called in case of uncaught exceptions
* It should NOT be called by the user.
* <P> It sends the last created exception to MRS, and calls abort();
*/
void Unexpected() throw() {
    Error mrs("Exception::Unexpected()", __FILE__, __LINE__);
    mrs.sendToMrs(MRS_ERROR);
    terminate();
}




}  // end of namespace Sct
