#include "IOManagerStreamerVersioned.h"
#include "../Streamer.h"
#include "../Serializable.h"
#include <cstdlib>
#include "../SctNames.h"

using std::string;

namespace Sct {
  namespace Archive {
    
    IOManagerStreamerVersioned::IOManagerStreamerVersioned() {
    }
    
    IOManagerStreamerVersioned::VersionMap& IOManagerStreamerVersioned::getReadVersionMap(){
      static VersionMap read_version_map;
      return read_version_map;
    }
    
    void IOManagerStreamerVersioned::writeClassName(OStream& os, const string& classname) const{
      setReadMode(false);
      //std::cout << "writing classname " << classname << std::endl;
      unsigned version = IOManager::getStreamer(classname).getVersion();
      string mapname = getMapName(classname, version);
      IOManager::writeClassName(os, mapname);
    }
    
    string IOManagerStreamerVersioned::readClassName(IStream& is) const{
      setReadMode(true);
      string tmp = IOManager::readClassName(is);
      //std::cout << tmp << std::endl;
      unsigned slash=tmp.rfind("/");
      if (slash==string::npos){
	throw StreamCorruptedException(string("Not a classname/version pair: `")+tmp+ "'",__FILE__,__LINE__);
      }
      std::istringstream iss(tmp.substr(slash+1));
      unsigned version; 
      iss >> version;
      string classname = tmp.substr(0,slash);
      getReadVersionMap()[classname]=version;
      return classname;
    }

    void IOManagerStreamerVersioned::readImpl(IStream& in, Streamable& ob, const std::string& className) const{
      setReadMode(true);
      string readclassname = readClassName(in);
      if (readclassname != className){
	std::ostringstream oss;
	oss << "Expected a " << className << "but got a " << readclassname;
	throw StreamCorruptedException(oss.str(), __FILE__, __LINE__);
      }
      getStreamer(className).read(in, ob, *this);
    }
    
    void IOManagerStreamerVersioned::writeImpl(OStream& out, const Streamable& ob, const std::string& className) const{
      setReadMode(false);
      writeClassName(out, className);
      getStreamer(className).write(out, ob, *this);
    }
    
    void IOManagerStreamerVersioned::readImpl(IStream& in, Streamable& ob, bool bReadClassName) const {
      setReadMode(true);
      string checkname = readClassName(in);
      if (checkname != ob.getClassName()){
	std::ostringstream oss;
	oss << "Was expecting a `" << ob.getClassName() 
	    << "' but got a `" << checkname << "'";
	throw StreamCorruptedException(oss.str(), __FILE__, __LINE__);
      }
      Streamer& s = getStreamer(ob.getClassName());
      s.read(in, ob, *this);
     }
    
    shared_ptr<Streamable> IOManagerStreamerVersioned::readImpl(IStream& in, const string& className) const {
      setReadMode(true);
      string checkname = readClassName(in);
      if (checkname != className){
	std::ostringstream oss;
	oss << "Was expecting a `" << className 
	    << "' but got a `" << checkname << "'";
	throw StreamCorruptedException(oss.str(), __FILE__, __LINE__);
      }
      Streamer& s = getStreamer(className);
      return s.read(in, *this);
    }
    
    shared_ptr<Streamable> IOManagerStreamerVersioned::readImpl(IStream& in) const {
      setReadMode(true);
      string className = readClassName(in);
      Streamer& s = getStreamer(className);
      return s.read(in, *this);
     }

    void IOManagerStreamerVersioned::writeImpl(OStream& out, const Streamable& ob, bool bWriteClassName) const {
      setReadMode(false);
      Streamer& s = getStreamer(ob.getClassName());
      writeClassName(out, ob.getClassName());
      s.write(out, ob, *this);
    }

    Streamer& IOManagerStreamerVersioned::getStreamer(const string& className) const{
      if (readMode()){
	// hope we know it already!
	VersionMap version_map = getReadVersionMap();
	if (version_map.find(className)==version_map.end()){
	  std::ostringstream oss;
	  oss << "I know about the following " << version_map.size() << "streamer/version pairs :" << std::endl;
	  for (VersionMap::const_iterator it=version_map.begin();
	       it!=version_map.end();
	       ++it){
	    oss << (*it).first <<  "/" << (*it).second << std::endl;
	  }
	  string diag = oss.str();
	  SctNames::Mrs() << "SCT_IO" << MRS_TEXT(diag) << MRS_DIAGNOSTIC << ENDM;
	  throw NoSuchStreamerException(className, "IOManagerStreamerVersioned dosen't know what version to use for Streamer", __FILE__, __LINE__);
	} else {
	  // good- found the read streamer.
	  return IOManager::getStreamer(className, (*version_map.find(className)).second );
	} 
      } else {
	// in write mode write streamer
	return IOManager::getStreamer(className);
      }
    }

    void IOManagerStreamerVersioned::setReadMode(bool mode) const{
      boost::recursive_mutex::scoped_lock lock (getMutex());
      m_read_mode=mode;
    }
    
    bool IOManagerStreamerVersioned::readMode() const{
      boost::recursive_mutex::scoped_lock lock (getMutex());
      return m_read_mode;
    }
    boost::recursive_mutex& IOManagerStreamerVersioned::getMutex() const{
      return m_access;
    }

  }
}
