//! A quick utility to read a ROD's serial number register and do a sanity check.

#include <iostream>
using namespace std;

#include <ctype.h>

#include "RodModule.h"

#include "primParams.h"
#include "RCCVmeInterface.h"
#include "parameters.h"

int main(int argc, char *argv[]) {

using namespace SctPixelRod;

  unsigned long serialNumber, masked1, masked2, rodRevision;
  VmePort *myVmePort;
  
  std::string fileName(""), option;
  bool verbose = false;
  int slot = -1;
  unsigned long baseAddress;

  if (argc > 1) {
    for (int i=1; i<argc; i++) {
      option = argv[i];
      if (option[0] != '-') break;
      switch (option[1]) {
        case 'v': {
          verbose = true;
          break;
        }
        case 's': {
          slot = atoi(option.substr(2).c_str());
          break;
        }
        default: {
          break;
        }
      }
    }
  }
  
// Prompt for slot number
  if (slot < 0 ) {
    cout << "Enter slot number (decimal):"; 
    cin >> slot;
    while ((slot < 1) || (slot > 21)) {
      cout << "Slot number out or range [1:21], re-enter: ";
      cin >> slot;
    }
  }
  baseAddress = slot << 24;

// Create VME interface
  RCCVmeInterface *vme1 = new RCCVmeInterface();
  
// Create RodModule and initialize it
  RodModule* rod0 = new RodModule(baseAddress, mapSize, *vme1, numSlaves);
  myVmePort = rod0->getVmePort();

  try {
// Read the FPGA Status Register 6 (counting from 0)
    serialNumber = myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[6]);
    cout << "FPGA Status Reg[6] = " << hex << serialNumber << endl;
    masked1 = serialNumber&0xFF000000;
    masked2 = serialNumber&0x000000FF;
    if (0xad000000 == masked1) {
      cout << "Sanity check passed" << endl;
      rodRevision = (serialNumber&0x00FF0000)>>16;
      cout << "Board revision = " << rodRevision << endl;
      serialNumber = serialNumber&0x3FF;
      cout << "board serial number = " << serialNumber << endl;
    }
    else if (0x000000ad == masked2) {
      cout << "Byte order wrong." << endl;
    }
    else {
      cout << "Sanity check failed!!!" << endl;
    }

    if(verbose) {
      cout << "HPIC: 0x" << hex << rod0->hpiFetch(HPIC) << dec << endl;

      cerr << "Load HWOB to HPIC\n";
      unsigned long hpicValue = 0x00010001;
      rod0->hpiLoad(HPIC, hpicValue);
      cerr << "...done\n";

      rod0->sleep(1000);

      cerr << "Read from EMIF\n";
      // mdspSingleRead/Write use HPID not HPID++ (not allowed)
      for(unsigned long addr = 0x01800000; addr < 0x01800020; addr+=4) {
        unsigned long val = rod0->mdspSingleRead(addr);
        cerr << "0x" << hex << addr << ": 0x" << val << dec << endl;
      } 
      cerr << "...done\n";
    }
  } catch(BaseException &b) {
    cout << "Exception touching ROD:\n" << b << endl;
  }

// Delete the ROD and VME objects before exiting
  delete rod0;
  delete vme1;

  return 0;  
}


