//------------------------------FlashLoad------------------------------ 

/*! 
 *  @brief This is a utility program for loading the flash memories which
 *  hold the program for the FPGAs. Do not use this to load the MDSP flash 
 *  memory; use MdspFlashLoad. 
 *
 *  It prompts the user for the VME slot number of the ROD, for the flash
 *  memory to load, and for the name of the binary file to load, including 
 *  the path to it. The entire contents of the file are written to the FPGA
 *  flash memory.
 *
 *  @author Tom Meyer (meyer@iastate.edu)
 */

#include <iostream>
using namespace std;

#include <ctype.h>
#include <string>
#include <fstream>

#include "RodModule.h"
#include "RCCVmeInterface.h"

int main(int argc, char *argv[]) {

using namespace SctPixelRod;

  const unsigned long mapSize=0xc00040;         // Map size
  const long numSlaves=4;                       // Number of slaves
  std::string binFileName;                      // Name of binary file to load
  ifstream binFile;                             // Pointer to binary frile
  unsigned long selectAddr;                     // Start address of seleced flash
  unsigned long flashAddr;                      // Location in target flash
  int fileSize;                                 // Size of binary file in bytes
  int selectSize;                               // Size of selected flash
  int iselect;
  
  const unsigned long flashStart[5]={0xe00000, 0xe01000, 0xe80000, 0xed3000, 0xf26000};
  const long flashSize[5] = {24, 495204, 336688, 336680, 234456};
  std::string flashName[5] = {"Location", "ROD Controller", "Formatter", 
       "Event Fragment Builder", "Router"};
  
  std::string fileName(""), option;
  int slot = -1;
  unsigned long baseAddress;

  if (argc > 1) {
    for (int i=1; i<argc; i++) {
      option = argv[i];
      if (option[0] != '-') break;
      switch (option[1]) {
        case 's': {
          slot = atoi(option.substr(2).c_str());
          break;
        }
        default: {
          break;
        }
      }
    }
  }

// Prompt for slot number
  if (slot < 0 ) {
    cout << "Enter slot number (decimal):"; 
    cin >> slot;
    while ((slot < 1) || (slot > 21)) {
      cout << "Slot number out or range [1:21], re-enter: ";
      cin >> slot;
    }
  }
  baseAddress = slot << 24;
  
// Create VME interface
  RCCVmeInterface *vme1 = new RCCVmeInterface();
  
// Create RodModule and initialize it
  RodModule* rod0 = new RodModule(baseAddress, mapSize, *vme1, numSlaves);
  try{
    rod0->initialize();
  }
  catch (HpiException *h) {
    hex(cout);
    cout << h->getDescriptor() << '\n';
    cout << "calcAddr: " << h->getCalcAddr() << ", readAddr: " << 
            h->getReadAddr() << '\n';
    dec(cout);
  };

  cout << "Enter Flash to program:" << endl;
  cout << "    1 = Location file" << endl;
  cout << "    2 = ROD Controller" << endl;
  cout << "    3 = Formatter" << endl;
  cout << "    4 = Event Fragment Builder" << endl;
  cout << "    5 = Router" << endl;
  cout << "Your choice (1-5)? ";
  cin >> dec >> iselect;
  selectAddr = flashStart[iselect-1];
  selectSize = flashSize[iselect-1];
  
  cout << "Enter binary file name, including extension (""q"" to quit): ";
  cin >>  binFileName;

  if (binFileName.c_str() == "q") exit(0);
  binFile.open(binFileName.c_str(), ios::binary);
  if (!binFile.is_open()) {
    cout << "Unable to open binary file." << endl;
    exit(1);
  }
  
// Get size of file
  binFile.seekg(0, ios::end);           // go to end of file
  fileSize = binFile.tellg();          // file size given by current location
  binFile.seekg(0, ios::beg);          // go back to beginning of file
  if (fileSize != selectSize) {
    cout << "File size is incorrect. Expected: " << dec << selectSize << " Found: " 
    << dec << fileSize << endl;
    exit(3);
  }
    
// Create a buffer and read file into it
  UINT8 * buffer;
  try {
    buffer = new UINT8[fileSize];
  }
  catch (std::bad_alloc & ba) {
    cout << "Unable to allocate buffer for binary file." << endl;
    exit(2);
  }
  binFile.read((char *)buffer, fileSize);
  
// write buffer to flash
  flashAddr = selectAddr;

  try {
    rod0->writeBlockToFlash(flashAddr, buffer, fileSize);

    cout << dec << selectSize << " bytes written to the "<< flashName[iselect-1]
         << " flash memory" << endl;
  } catch(BaseException &b) {
    cout << "******* Exception during FLASH upload!********\n";
    cout << b << endl;
  }
        
// Clean up before exiting
  delete [] buffer;
  delete rod0;
  delete vme1;

  return 0;  
}


