//File: TimModule.h

#ifndef SCTPIXELROD_TIMMODULE_H
#define SCTPIXELROD_TIMMODULE_H

/*! \file
 * \brief TimModule: A derived class for VME TIM modules.
 *
 * This file declares a TIM class derived from the VmeModule base class.
 *
 * Contributors: John Lane <jbl@hep.ucl.ac.uk> - originator
 *
 * $Id: TimModule.h,v 1.7 2003/12/04 19:10:49 jbl Exp $
 *
 * $Log: TimModule.h,v $
 * Revision 1.7  2003/12/04 19:10:49  jbl
 * TimModule uses BaseException
 *
 * Revision 1.6  2003/06/04 15:04:47  tmeyer
 * Removed explicit directory structure from includes
 *
 * Revision 1.5  2003/05/20 19:26:25  jbl
 * TimModule UINT8 & UINT16
 *
 * Revision 1.4  2002/12/11 21:30:50  jbl
 * TimModule major update
 *
 *
 *
 * NB define eg I_AM_LINUX_HOST for processor.h (eg typedef UINT32)
 */

#include "TimDefine.h"
#include "processor.h"
#include "VmeInterface.h"
#include "VmePort.h"
#include "VmeModule.h"
#include "BaseException.h"

namespace SctPixelRod {

// ------------------------- TimException ------------------------------------

//! TimException: a general exception class for TIM errors.
/*!
  This class is thrown if an error in a TIM operation is detected.

  Contributors: John Lane <jbl@hep.ucl.ac.uk> - originator
*/

class TimException : public BaseException {

public:

  TimException( std::string descriptor, int data1, int data2 );
  int         getData1()      { return m_data1; };
  int         getData2()      { return m_data2; };
  virtual void what( std::ostream & );

private:

  int         m_data1;       //!< First  data value returned
  int         m_data2;       //!< Second data value returned
};

// ------------------------- TimModule ---------------------------------------

//! TimModule: A derived class for VME TIM modules.
/*!
  This is the implementation of a TIM class derived from the VmeModule base
  class. It should be the sole interface for VME communication with TIM.

  Contributors: John Lane <jbl@hep.ucl.ac.uk> - originator
*/

class TimModule : public VmeModule {

public:

  // Constructor and destructor
  // There are no copy or assignment operators due to VME restrictions.

  TimModule( UINT32 baseAddr, UINT32 mapSize, VmeInterface & ourInterface );

  virtual ~TimModule();

  // Accessor functions

  UINT32   getFirmware()     { return m_firmware; };
  UINT32   getSerialNumber() { return m_serialNumber; };
  VmePort* getVmePort()      { return m_vmePort; };

  // Methods

  void initialize();
  void reset();
  void status ();

  UINT32 fetchL1ID();                     //!< Read trigger number
  UINT16 fetchTimID();                    //!< Read TIM ID register

  void intTrigStart( const TimMaskFrequency frequency ); //!< Internal Trigger
  void intTrigStop();                               //!< Stop Internal Trigger

  void issueCommand( const TimBitCommand mask );  //!<       Issue TIM command
  void issueVCAL( const UINT8 pipelineDelay );   //!< Issue VCAL + L1A command

  void loadBitClear( const TimRegister addr, const UINT16 mask );//!< Clear bit
  void loadBitSet( const TimRegister addr,   const UINT16 mask );//!<   Set bit
  void loadByteHi( const TimRegister addr, const UINT8 byte ); //!< Upper byte
  void loadByteLo( const TimRegister addr, const UINT8 byte ); //!< Lower byte

  UINT16 regFetch( const TimRegister addr );
                                          //!< Read from a 16-bit VME register

  void   regLoad( const TimRegister addr, const UINT16 data );
                                          //!<  Write to a 16-bit VME register

  void seqFetch( const UINT16 size,      UINT16 buffer[] ); //!< Read sequencer
  void seqLoad( const UINT16 size, const UINT16 buffer[] ); //!< Load sequencer
  void seqRun(  const UINT16 size );                        //!< Run  sequencer

  UINT16 vmeFetch( const UINT32 addr )
         throw (VmeException &);          //!< Read from a 16-bit VME register

  void   vmeLoad(  const UINT32 addr, const UINT16 data )
         throw (VmeException &);          //!<  Write to a 16-bit VME register

private:

  UINT32   m_firmware;     //!< Firmware version number
  UINT32   m_serialNumber; //!< Board serial number
  VmePort* m_vmePort;      //!< VME Port handle

}; // End class TimModule declaration

}  // End namespace SctPixelRod

// ------------------------- Overload operator<< -----------------------------

//! Overloaded operator to print TIM status

namespace std {

ostream& operator<<( ostream& os, SctPixelRod::TimModule& tim );

}  // End namespace std

#endif // SCTPIXELROD_TIMMODULE_H
