//File: RodModule.cxx
// $Header: /afs/cern.ch/user/s/sctpixel/private/cvsroot/RodDaq/RodCrate/RodModule.cxx,v 1.61 2004/03/09 15:53:21 tmeyer Exp $

#include "RodModule.h"
#include "BocCard.h"

//! Namespace for the common routines for SCT and PIXEL ROD software. 
namespace SctPixelRod {

// Flag to force endian-ness swap before calls to HPI registers.  This should be
// a temporary patch.

static long endianSwap = 0;
static const long rodInitTimeout = 10;

//******************************Class NoImageFile**************************
//
// Description:
//  This class is thrown if If we try to load a binary image for a master or 
//  slave DSP and the file is not found
//   
//  Author(s):
//    Tom Meyer (meyer@iastate.edu) - originator

//	Constructors. Use defaults for destructor, copy, and assignment.

NoImageFile::NoImageFile(std::string descriptor, std::string fileName) : BaseException(descriptor) {
  m_fileName = fileName;
  setType(NOIMAGEFILE);
  }
  
void NoImageFile::what(std::ostream& os) {
  os << "NoImageFile Exception. Descriptor, File name: " << getDescriptor() << ";" << m_fileName << std::endl;
}
  
//******************************Class HpiException**************************
//
// Description:
//  This class is thrown if an error in an HPI read/write operation is detected.
//   
//  Author(s):
//    Tom Meyer (meyer@iastate.edu) - originator

//	Constructors. Use defaults for destructor, copy, and assignment.

HpiException::HpiException( std::string descriptor, unsigned long calcAddr,
                            unsigned long readAddr) : BaseException(descriptor) {
  m_calcAddr = calcAddr;
  m_readAddr = readAddr;
  setType(HPI);
  }

void HpiException::what(std::ostream& os) {
    os << "HpiException: " << getDescriptor() << std::endl;
    os << "Calculated Address:" << std::hex << getCalcAddr() << std::endl;
    os << "Read Address:" << std::hex << getReadAddr() << std::endl;
}  
//***************************Class RodException**************************
//
// Description:
//  This class is thrown if an error in a ROD operation is detected.
//   
//  Author(s):
//    Tom Meyer (meyer@iastate.edu) - originator

//	Constructors. Use defaults for destructor, copy, and assignment.

RodException::RodException( std::string descriptor) : BaseException(descriptor) {
  m_numData = 0;
  m_data1 = m_data2 = 0;
  setType(ROD);
  }
RodException::RodException( std::string descriptor, unsigned long data1) :
                            BaseException(descriptor) {
  m_numData = 1;
  m_data1 = data1;
  m_data2 = 0;
  setType(ROD);
  }
RodException::RodException( std::string descriptor, unsigned long data1,
                            unsigned long data2) : BaseException(descriptor) {
  m_numData = 2;
  m_data1 = data1;
  m_data2 = data2;
  setType(ROD);
  }

void RodException::what(std::ostream& os) {
  unsigned long numData;
  numData = getNumData();
  os << "RodException: " << getDescriptor() << std::endl;
  if (0 == numData) return;
  os << "Data1:" << getData1() << std::endl;
  if (1 == numData) return;
  os << "Data2:" << getData2() << std::endl;
}  
//********************************Class RodModule***************************
//
// Description:
//  This is a derived class providing the software interface for VME ROD modules.
//   
//  Author(s):
//    Tom Meyer (meyer@iastate.edu) - originator

//---------------------------------Constructor------------------------------                                    
/*
This is the only constructor to use.
*/

RodModule::RodModule( unsigned long baseAddr, unsigned long mapSize, 
     VmeInterface & ourInterface, long numSlaves) throw (RodException&,
     VmeException&) :
     VmeModule(baseAddr, mapSize, ourInterface) {
  m_slot = baseAddr>>24;
  m_serialNumber = 0xFFFFFFFF;  // Will be overwritten during init
  m_revision = 0;               // Will be overwritten during init
  m_numSlaves = numSlaves; 
  m_finBufferSize = 4096; 
  m_masterImageName = std::string("");
  m_myOutList = 0;

  try {
    m_myVmePort = new VmePort(m_baseAddress, m_mapSize, VmeInterface::A32, 
                  m_ourInterface); 
  }
  catch (std::bad_alloc & ba) {
    throw RodException("Failed to get VME Port.");
  }
  for (long i=0; i<4; i++) {
    m_slaveIpramName[i]= std::string("");
    m_slaveIdramName[i]= std::string("");
    m_slaveExtName[i]= std::string("");
  }

  for(int i=0; i<2; i++) {
    m_vmeCommandReg[i] = 0;
  }
  for(int i=0; i<3; i++) {
    m_rodStatusReg[i] = 0;
  }

  m_myOutList = 0;

  for (long i=0; i<N_TXT_BUFFS; i++) {
    m_textBuff[i] = 0;
  };
  m_myTextState = TEXT_IDLE;
  m_myPrimState = PRIM_IDLE;
  m_textType = TEXT_UNDEF;

  // Make the vme library generate exceptions on bus errors
  getVmePort()->setExceptionTrapping(true);

  m_myBoc = new BocCard(*this);
}

//---------------------------------Destructor------------------------------                                    
/*
Be sure to delete all VME Ports.
*/

RodModule::~RodModule() {
  delete(m_myVmePort);
  delete m_myBoc;
  m_myVmePort = 0;
  return;
}

//  Member methods

// Some data accessor methods placed here instead of in RodModule.h to avoid
// potential problems with forward declaration of BocCard.

  void RodModule::setBoc(BocCard* myBoc) {m_myBoc=myBoc; return;};
  BocCard* RodModule::getBocCard() {return m_myBoc; }

//--------------------------------initialize---------------------------------                                 

/* This method performs all actions necessary to place a properly loaded ROD 
in its initialized state, except for initializing the slave DSPs. This must be done
separately using initSlaveDsp().

Activities:
    Reset the ROD
    Read ROD serial number
    Load 0x000l000l into Master DSP HPIC register. (Says low order 16-bit word
                                                    comes first)
    Read back HPIC to verify
    Retrieve pointers to text buffer structs
    Initialize PrimList and TextBuff state variables
*/
void RodModule::initialize() {initialize(false); return;};

void RodModule::initialize(bool resetFlag) throw (RodException &, VmeException &) {

  unsigned long hpicReadback;   // for debugging
  const unsigned long fpgaResetValue = 0x00000020;
  const unsigned long rodResetValue = 0x00000040;
  unsigned long rodReset, fpgaReset;
  unsigned long rodRegValue;
  clock_t startTime;
  
// If reset flag is set reset the ROD so it starts in a known state
  if (resetFlag) {
    if (endianSwap) {
      rodReset = endianReverse32(rodResetValue);
      fpgaReset = endianReverse32(fpgaResetValue);
    }
    else {
      fpgaReset = fpgaResetValue;
      rodReset = rodResetValue;
    }

// Reconfigure the FPGAs
    m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[0], fpgaReset);  //FPGA config

// Test that FPGA has finished its reset
    rodRegValue = 0;
    startTime = time(NULL);
    bool change = false;
    do {
      rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[0]);
      if ((time(NULL)-startTime) > DSP_RESET_TIMEOUT) {
        throw RodException("FPGA reset timeout in initialize(), DSP_RESET_TIMEOUT=30", rodRegValue);
      }
      // Check for configuration active
      if ((rodRegValue&0x20) == 0x20) change = true;
    } while (((rodRegValue&0x3f) != 0x1f) || change==false);

// Wait a bit in between reconfigure and MDSP reset
    sleep(500);

    rodRegValue = readRodStatusReg(0);
    m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[2], rodReset);   // MDSP reset

// Test that MDSP has finished its reset
    rodRegValue = 0;
    startTime = clock();
    do {
      rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[2]);
      if ((clock()-startTime)/CLOCKS_PER_SEC > DSP_RESET_TIMEOUT) {
        throw RodException("DSP reset timeout in initialize(), DSP_RESET_TIMEOUT=",
        (long)DSP_RESET_TIMEOUT);
      }
    }
    while ((rodRegValue&0x3e) != 0x3e);
    rodRegValue = readRodStatusReg(0);

    // Wait an extra half a second
    sleep(500);
  }

// Read ROD serial number 
  m_serialNumber = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[6]);
  if (endianSwap) m_serialNumber = endianReverse32(m_serialNumber); // Temporary (I hope)
  m_revision = (m_serialNumber&0x00ff0000)>>16;
  m_serialNumber = m_serialNumber & 0x3ff;   // Mask to get 10 least significant bits
   
//	Initialize Master HPIC register to set half-word order

  unsigned long hpicValue = 0x00010001;

  hpiLoad(HPIC, hpicValue);
  hpicReadback = hpiFetch(HPIC);         // for debugging
  sleep(100);

// Wait for MasterDSP init
  for (long i=0; ; ++i) {
    if (readRodStatusBit(0, SR_RUNNING)) {
      break;
    }
    if (i>rodInitTimeout*10) throw RodException("Master DSP init timed out. Tries, Limit:", i,
        rodInitTimeout);
    sleep(100);
  }

  for (int i=0; i< N_TXT_BUFFS; i++) {
    m_textBuff[i] = (unsigned long *)mdspSingleRead(REPLY_BUFF_BASE + 4*i);
  }

//  Reinitialize state variables
  m_myTextState = TEXT_IDLE;
  m_myPrimState = PRIM_IDLE;

return;
}

//---------------------------------reset------------------------------------                                

/* This method resets the ROD.
*/

void RodModule::reset() throw (RodException&, VmeException &) {
  // Reset all FPGAs and DSPs (bit 7 = 0x80)
  const unsigned long rodResetValue = 0x00000040;
  unsigned long rodReset;
  unsigned long rodRegValue;
  clock_t startTime;
  
  if (endianSwap) {
    rodReset = endianReverse32(rodResetValue);
  }
  else {
    rodReset = rodResetValue;
  }
  m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[2], rodReset);
  sleep(10000);
  rodRegValue = 0;
  startTime = clock();
  do {
    rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[2]);
    if ((clock()-startTime)/CLOCKS_PER_SEC > DSP_RESET_TIMEOUT) {
      throw RodException("DSP reset timeout in reset() (first call), DSP_RESET_TIMEOUT=",
      (long)DSP_RESET_TIMEOUT);
    }
  }
  while (rodRegValue != 0x3e);
   
  // Clear IDRAM
  unsigned long * buff;
  try {
    // Allocate a buffer of zeros
    buff = new unsigned long[IDRAM_SZ/sizeof(long)];

    for(unsigned int i=0; i<IDRAM_SZ/sizeof(long); i++) {
      buff[i] = 0;
    }
  }
  catch (std::bad_alloc & ba) {
    throw RodException("Reset() failed to allocate buffer. Size=",
                                  IDRAM_SZ/sizeof(long));
  }
  mdspBlockWrite(IDRAM_BASE, buff, IDRAM_SZ/sizeof(long));
  delete [] buff;
  
  // Reset all DSPs
  resetMasterDsp();
  sleep(50);
  for (long i=0; i<m_numSlaves; i++) {
    resetSlaveDsp(i);
    sleep(50);
    rodRegValue = 0;
    startTime = clock();
    do {
      rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[2]);
      if ((clock()-startTime)/CLOCKS_PER_SEC > DSP_RESET_TIMEOUT) {
        throw RodException("DSP reset timeout in reset() (second call), DSP_RESET_TIMEOUT=",
        (long)DSP_RESET_TIMEOUT);
      }
    }
  while (rodRegValue != 0x3e);
  };
  
  //  Reinitialize state variables
  m_myTextState = TEXT_IDLE;
  m_myPrimState = PRIM_IDLE;
  
return;
}

//----------------------------------verify----------------------------------                                

/* This method checks the value of the h.o. byte of the 
*/

bool RodModule::verify() throw() {
  unsigned long regVal;
  
// Read status register 6 
  regVal = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[6]);
  if (endianSwap) regVal = endianReverse32(regVal); 
  regVal = regVal & 0xff000000;            // Mask to get most significant byte
  if (regVal!=0xad000000) return false;
  return true;
}

//---------------------------------status-----------------------------------                                

/* This method reports the status of the ROD.
For now, it simply prints to standard output.
*/
void RodModule::status() throw() {
    std::cout << "Slot: " << m_slot;
    std::cout << "Serial Number:" << m_serialNumber;
    std::cout << "Number of slave DSPs: " << m_numSlaves;
    std::cout << std::endl;
    std::hex(std::cout);
    std::cout << "Status registers[0-2]: ";
    try {
      for (int i=0; i<3; i++) {
        std::cout << readRodStatusReg(i) << " " ;
      }
    }
    catch (VmeException &) {
      std::cout << "status() can't read ROD status registers." << std::flush;
    }
    std::cout << std::endl;
    
    std::cout << "Command registers[0-1]: ";
    try {
      for (int i=0; i<2; i++) {
        std::cout << readRodCommandReg(i) << " " ;
      }
    }
    catch (VmeException &) {
      std::cout << "status() can't read ROD command registers.\n" << std::flush;
    }

    std::dec(std::cout);
    std::cout << "Primitive state: " << getPrimState() << " Text State: " << 
          getTextState();
    std::cout << std::endl;
    return;
}

//--------------------------------initSlaveDsp-------------------------------                       
void RodModule::initSlaveDsp(const std::string & ipramFile,
     const std::string & idramFile,const std::string & extFile,
     const long slaveNumber, char opt) throw (RodException&, NoImageFile&, VmeException &) {

  unsigned long readBack;
  unsigned long hpicValue = 0x00010001;
  unsigned long rodRegValue;
  clock_t startTime;
 
  if ((slaveNumber < 0) || (slaveNumber >= m_numSlaves)) throw RodException(
       "Slave Number out of Range. slaveNumber, m_numSlaves:", slaveNumber, 
       m_numSlaves);

// Set HPIC registers to make first halfword the most significant
  slvHpiLoad(SLAVE_HPIC_BASE, hpicValue, slaveNumber);
  readBack = slvHpiFetch(SLAVE_HPIC_BASE, slaveNumber);
  
// Load the memory images and remember the file names
  loadSlaveImage(ipramFile, SLAVE_IPRAM_ADDR, slaveNumber, opt);
  m_slaveIpramName[slaveNumber] = ipramFile;
  loadSlaveImage(idramFile, SLAVE_IDRAM_ADDR, slaveNumber, opt);
  m_slaveIdramName[slaveNumber] = idramFile;
  loadSlaveImage(extFile, SLAVE_CE2_ADDR, slaveNumber, opt);
  m_slaveExtName[slaveNumber] = extFile;
  
// Send the START_SLAVE_EXECUTING primitive. Do the primList dialog synchronously
  startSlave(slaveNumber);

// Check that slave has finished its reset
  rodRegValue = 0;
  startTime = clock();
  do {
    rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[2]);
    if ((clock()-startTime)/CLOCKS_PER_SEC > DSP_RESET_TIMEOUT) {
      throw RodException("DSP reset timeout in initSlaveDsp(), DSP_RESET_TIMEOUT=",
      (long)DSP_RESET_TIMEOUT);
    }
  }
  while (rodRegValue != 0x3e);
  return;
}


//--------------------------------loadSlaveImage-------------------------------                       
// This version writes to cout. This needs to be changed.
void RodModule::loadSlaveImage(const std::string & fileName, const unsigned long address,
     const long slaveNum, char opt) throw (NoImageFile&, RodException&, 
                                               VmeException &) {
  
  long numWords;
  unsigned long readBack;   
  int fileSize;
  const unsigned long DSP_BUSY_MASK=0x00000002;
  std::ifstream fin;
// Cannot load image if slave is running.  Check it and throw a rodException if it is
  readBack = slvHpiFetch(SLAVE_HPIC_BASE, slaveNum);
  if (readBack & DSP_BUSY_MASK) throw RodException(
     "loadSlaveImage tried to load to a running SDSP. slavenum = ", slaveNum);
     
  fin.open(fileName.c_str(),std::ios::binary);
  if (fin.is_open()) {

// Get size of file
    fin.seekg(0, std::ios::end);  // go to end of file
    fileSize = fin.tellg();  // file size is current location
    fin.seekg(0, std::ios::beg);  // go back to beginning of file

// Create buffer and fill it with file contents
    char * buffer; 
    try {
      buffer = new char[fileSize];
    }
    catch (std::bad_alloc & ba) {
      throw RodException(
                  "loadSlaveImage buffer failed to allocate. buffer, fileSize:",
                  (unsigned long)buffer, (unsigned long)fileSize);
    }
    fin.read(buffer, fileSize);

// Use slvBlockWrite to transfer buffer into memory
    numWords = (fileSize+3)/4;
    slvBlockWrite(address, (unsigned long*) buffer, numWords, slaveNum);
    
// Read back buffer for verification if 'v' option was given
    if (opt=='v') {
      char* buffer2;
      try {
        buffer2 = new char[fileSize];
      }
      catch (std::bad_alloc & ba) {
        throw RodException(
                  "loadSlaveImage buffer2 failed to allocate. buffer, fileSize:",
                  (unsigned long)buffer, (unsigned long)fileSize);
      }
      slvBlockRead(address, (unsigned long*)buffer2, numWords, slaveNum);
   
      for (int i=0; i<fileSize; i++) {
        if (buffer[i] != buffer2[i]) {
          delete [] buffer;
          delete [] buffer2;
          fin.close();
          char message[] = "loadSlaveImage buffer readback invalid for file, slavenum, char index: ";
          throw RodException( message+fileName, slaveNum, i);
        }
      } 
      delete [] buffer2;
      std::cout << "Image verification check complete\n";
    }
    
// Close the file and delete the buffer    
    fin.close();
    delete [] buffer;
  }
  else {
    throw NoImageFile("NoImageFile thrown from loadSlaveImage", fileName);
  }
  return;
}

//--------------------------------startSlave-------------------------------                       
// Starts slave executing
// The default way is to have the master DSP run the START_SLAVE_EXECUTING
// primitive. This is what should normally be used.
// A quick "back door" way is to set the slave's DSPINT bit directly.

void RodModule::startSlave(const long slaveNumber, char mode) 
                                            throw(RodException&, VmeException &) {
  const unsigned long dspint = 0x2;

  if (mode == 'q') {             // The quick way: just set DSPINT directly
    slvHpiLoad(SLAVE_HPIC_BASE, dspint, slaveNumber);
  }

  else {                         // The slow way, a master primitive (default)
  long primData[4]={slaveNumber, 1, 2, 0x200000};
  RodPrimitive * startSlaveExec;
  try {
    startSlaveExec = new RodPrimitive(8, 1, START_SLAVE_EXECUTING, 
       R_START_SLAVE_EXECUTING, primData);
  }
  catch (std::bad_alloc & ba) {
    throw RodException(
      "startSlave unable to allocate startSlaveExec. slaveNumber:", slaveNumber); 
  }
  synchSendPrim(*startSlaveExec); 
  delete startSlaveExec;
  }
  
  return;
}

/*---------------------------------synchSendPrim------------------------------                           
This version writes to cout. This needs to be changed.
*/
void RodModule::synchSendPrim(RodPrimitive & prim) throw (RodException&, 
                                                         VmeException &) {
  RodPrimList primList(1);  // Private primList (local to this routine)
  long myTextLength;                            // Actual length of text message
  TEXT_BUFFER_TYPE myTextType;                  // Buffer type for latest message
  PrimState returnPState;
  TextBuffState returnTState;

// Create a text buffer in case of error message
    char * myTextBuffer;
    try {
      myTextBuffer = new char[TEXT_BUFF_SIZE];
    }
    catch (std::bad_alloc & ba) {
      throw RodException(
          "synchSendPrim failed to allocate text. Size=", TEXT_BUFF_SIZE);
    }
    primList.insert(primList.begin(), prim);
    try {
      primList.bufferBuild();
    }
    catch (PrimListException &p) {
//      throw (RodException(p.getDescriptor,p.getData1(), p.getData2());
      std::cout << p.getDescriptor() << " ";
      std::cout << p.getData1() << ", " << p.getData2() << "\n";
    };
    try {
      this->sendPrimList(&primList);
    }
    catch (BaseException &h) {
      std::cout << h;
    };

// Wait for ROD to begin executing and then wait for it to finish executing
// Check for error messages in text buffer and read them out if they exist.
// Note: this is the synchronous way to do it. In general, it is better to do it
// asynchronously.
    do {
      returnPState = this->primHandler();
      returnTState = this->textHandler();

      // Use while to print out multiple buffers
      while (returnTState == TEXT_RQ_SET) {
        do {
          returnTState = this->textHandler();
        } while (returnTState != TEXT_READOUT);
        this->getTextBuffer(myTextBuffer, myTextLength, myTextType);
        this->clearTextBuffer();
        for (int i=0; i<myTextLength; i++) {
          std::cout << myTextBuffer[i];
          if (0==(i+1)%64) std::cout << std::endl;
        }
        std::cout << std::endl; 
        returnTState = this->textHandler();
      }
    } while (returnPState != PRIM_EXECUTING); 
    do {
       try {
         returnPState = this->primHandler();
       }
       catch (RodException &r) {
       std::cout << r.getDescriptor() <<", " << r.getData1() << ", " << r.getData2()
               << '\n';
       }
    } while (returnPState != PRIM_WAITING && returnPState != PRIM_IDLE); 
    delete [] myTextBuffer;    // Found by valgrind

  return;
}

//---------------------------------sendPrimList-------------------------------                           

void RodModule::sendPrimList(RodPrimList *l) throw(PrimListException &, VmeException &) {

  unsigned long * buffAddr;
  long buffCount;

  if (!l->getBuffer()) {
    l->bufferBuild();
  };

  buffCount = l->getBufferLength();

  if(abs(buffCount) > PRIM_BUFF_SIZE) {
    throw PrimListException("PrimList is bigger than DSP buffer", buffCount, PRIM_BUFF_SIZE);
  }

  buffAddr = l->getBuffer();
  mdspBlockWrite(PRIM_BUFF_BASE, buffAddr, buffCount);
  m_myPrimState = PRIM_LOADED;

  return;  
}  

//---------------------------------primHandler-------------------------------                           

PrimState RodModule::primHandler() throw(RodException &, VmeException &) {
long wordCount;
unsigned long status;
unsigned long* buffptr;
unsigned long replyBuffer;
  switch (m_myPrimState) {
    case PRIM_LOADED: {
      if (0==getDspAck()) {
        setInListReady();
        m_myPrimState = PRIM_EXECUTING;
      };
    break;
    };
    case PRIM_EXECUTING: {
      status = readRodStatusReg(0);
      if (1==getDspAck()) {
        if (status >> TEXT_BUFF_NOT_EMPTY[0]) {
//        throw RodException ("Error buffer not empty in primHandler(PRIM_EXECUTING)");
        }
        if (1==readRodStatusBit(0,OUTLIST_READY)) {
          // read buffer size
          hpiLoad(HPIA, REPLY_BUFF_BASE);
          wordCount = hpiFetch(HPID_NOAUTO);

	  // create new OutList object.  Note that this object persists until it is deleted
	  // in deleteOutList.  The pointer in outList dies when this routine exists but
	  // the object itself persists in memory until deleted.  By copying the pointer
	  // value in the address "outList" to m_myOutList, we keep knowledge of where the
	  // object is an may continue to use it until it is deleted.
	  
          RodOutList* outList;
          try {
            outList = new RodOutList(wordCount);
          }
          catch (std::bad_alloc & ba) {
            throw RodException("Outlist creation failed. WordCount=", 
                                              wordCount);
          }
          if (0==outList->getBody()) throw RodException(
                 "Outlist failed to allocate body array.WordCount=", wordCount);
		 
	  // This is where the value of m_myOutList gets set to the value of outList.
          this->setOutList(outList);
	  
	  // fill OutList object
          buffptr = m_myOutList->getBody();
          replyBuffer = REPLY_BUFF_BASE;
	  mdspBlockRead(replyBuffer, buffptr, wordCount);
          m_myPrimState = PRIM_WAITING;
        }
        else m_myPrimState = PRIM_IDLE;
        clearVmeCommandRegBit(INLISTRDY);
      }
    break;
    };
    default: {
    break;
    };
  };
  return m_myPrimState;
};

//--------------------------------deleteOutList--------------------------------

  void RodModule::deleteOutList() throw() {
  delete m_myOutList;
  m_myOutList = 0;
  return;
  };

//---------------------------------textHandler-------------------------------                           

TextBuffState RodModule::textHandler() throw(VmeException &) {
  unsigned long regValue;
  unsigned long dataPtr;
  long txtLength, txtLength1, txtLength2, txtWords;
  switch (m_myTextState) {
    case TEXT_IDLE: {
      regValue = readRodStatusReg(0) & SR_TEXT_MASK;
      if (0==regValue) return m_myTextState;         // Quick return if no text
      for (long i=0; i<4; i++) {
        if (regValue & SR_TEXT_BIT_MASK[i]) {
          m_textType = (TEXT_BUFFER_TYPE)i;
          setVmeCommandRegBit(TEXT_BUFF_READ_REQ[m_textType]);
          sleep(10);
          m_myTextState = TEXT_RQ_SET;
          return m_myTextState;
        };
      }
      break;
    };
    case TEXT_RQ_SET: {
      if (readRodStatusReg(0) & SR_TEXT_BIT_MASK[m_textType]) {
        return m_myTextState;
      };
      hpiLoad( HPIA, (unsigned long)m_textBuff[m_textType]);
      m_txtBuffer.dataEnd = hpiFetch(HPID_AUTO);
      m_txtBuffer.readIndx = hpiFetch(HPID_AUTO);
      m_txtBuffer.writeIndx = hpiFetch(HPID_AUTO);
      m_txtBuffer.mode = hpiFetch(HPID_AUTO);
      m_txtBuffer.overwrite = hpiFetch(HPID_AUTO);
      m_txtBuffer.overflow = hpiFetch(HPID_AUTO);
      m_txtBuffer.wrap = hpiFetch(HPID_AUTO);
      m_txtBuffer.state = hpiFetch(HPID_AUTO);
      m_txtBuffer.id = hpiFetch(HPID_AUTO);
      m_txtBuffer.data = (char *)hpiFetch(HPID_AUTO);
      dataPtr = (unsigned long)m_txtBuffer.data;

// If there was a wrap-around we need to read two blocks, otherwise only one
      if (m_txtBuffer.readIndx < m_txtBuffer.writeIndx) {           // No wrap
        txtLength = m_txtBuffer.writeIndx - m_txtBuffer.readIndx;
        txtWords = (txtLength+3)/4;     // Convert char count to words
	mdspBlockRead(dataPtr, (unsigned long*)m_textData, txtWords);
      }
      else {                                                         // Wrap
        txtLength1 = TEXT_BUFF_SIZE - m_txtBuffer.readIndx;
        txtWords = (txtLength1+3)/4;
	mdspBlockRead(dataPtr+4*m_txtBuffer.readIndx, (unsigned long*)m_textData, txtWords);
        txtLength2 = m_txtBuffer.writeIndx;
        txtWords = (txtLength2+3)/4;
	mdspBlockRead(dataPtr, (unsigned long*)(m_textData+4*txtLength1), txtWords);
        txtLength = txtLength1 + txtLength2;
      };
      clearVmeCommandRegBit(TEXT_BUFF_READ_REQ[m_textType]);
      sleep(10);
      m_myTextState = TEXT_READOUT;
      m_txtBuffer.state = txtLength;
      return m_myTextState;
      break;
    };
    default: {
      return m_myTextState;
      break;
    };
  };

  return m_myTextState;
};


//-----------------------------getTextBuffer-------------------------------                                

  void RodModule::getTextBuffer(char * buffer, long & length, 
         TEXT_BUFFER_TYPE & type) throw() {
    length = m_txtBuffer.state;
    type = (TEXT_BUFFER_TYPE)m_textType;
    for (int i=0; i<m_txtBuffer.state; i++) {
      buffer[i] = m_textData[i];
    };
    return;
  };
  
//--------------------------------clearTextBuffer------------------------------

void RodModule::clearTextBuffer() throw() {
  m_textType = TEXT_UNDEF;
  m_txtBuffer.state = 0;
  m_myTextState = TEXT_IDLE;
  return;
};

//----------------------------------hpiLoad------------------------------------                                

void RodModule::hpiLoad(unsigned long hpiReg, unsigned long hpiValue) 
     throw(VmeException &) {

  if (endianSwap) hpiValue = endianReverse32(hpiValue); 
  m_myVmePort->write32(hpiReg, hpiValue);
};

//--------------------------------hpiFetch-------------------------------------                                

unsigned long RodModule::hpiFetch(unsigned long hpiReg) 
              throw(VmeException &){

unsigned long hpiValue;

  hpiValue=m_myVmePort->read32(hpiReg);
  if (endianSwap) hpiValue = endianReverse32(hpiValue);
  
  return hpiValue;
};

//-----------------------------mdspSingleRead---------------------------------                                

unsigned long RodModule::mdspSingleRead(const unsigned long dspAddr) 
              throw(VmeException &) {
  unsigned long value; 

//  Load the DSP address into the HPIA register 
  hpiLoad(HPIA, dspAddr);

//	Do the read 
  value = m_myVmePort->read32(HPID_NOAUTO);
  if (endianSwap) {
    value = endianReverse32(value);
  }
  return value;
};

//------------------------------mdspSingleWrite------------------------------------                                   

void RodModule::mdspSingleWrite(unsigned long dspAddr, unsigned long buffer) 
     throw(VmeException &) { 

//  Load the HPID address into the HPIA register 
  hpiLoad(HPIA, dspAddr);

//	Do the write 
  if (endianSwap) {
      buffer = endianReverse32(buffer);
  }

  m_myVmePort->write32(HPID_NOAUTO, buffer);

  return;
};

//-----------------------------mdspBlockRead---------------------------------                                


void RodModule::mdspBlockRead(const unsigned long dspAddr, unsigned long buffer[],
       const long wordCount, HpidMode mode) throw (HpiException &, 
       VmeException &) {
  unsigned long hpidAddr; 
  long myCount, localCount, blockWordCount, wordIncr;

// Determine the HPI mode to use.
  switch (mode) {
    case AUTO:
      hpidAddr= HPID_AUTO;
      break;
    case NO_AUTO:
      hpidAddr = HPID_NOAUTO; 
      break;
    case DYNAMIC:
    default:
      if (wordCount == 1){
        hpidAddr = HPID_NOAUTO;
      }	
      else { 
        hpidAddr = HPID_AUTO;
      };
      break;
  };

//  Load the DSP address into the HPIA register 
  hpiLoad(HPIA, dspAddr);  
  
// Check if wordcount is odd and, if so, first do a single word read.

  localCount = wordCount;
  wordIncr = 0;
  if (wordCount%2 !=0) {
    buffer[0] = m_myVmePort->read32(hpidAddr);
    wordIncr = 1;
    localCount -= 1;
  }

// Check to see that we don't access too large a block. 
// MAX_HPID_WORD_ELEMENTS is defined in RodVmeAddresses.h

   blockWordCount = std::min(localCount, MAX_HPID_WORD_ELEMENTS);
   
//	Set up the transfer as a series of block transfers 
  for (myCount=localCount; myCount>0;
                    myCount-=blockWordCount, wordIncr+=blockWordCount) {

//	Recheck each block 
    blockWordCount = std::min(myCount, MAX_HPID_WORD_ELEMENTS);
    
//	Do the transfer for this block 
// Block reads are being done in hardware as from 10 June 03.

    m_myVmePort->blockRead32(hpidAddr, buffer+wordIncr, blockWordCount*4);

// Use this code if you have problems with the hardware block transfers    
/*    for (int i=0; i<blockWordCount; i++) {
      buffer[wordIncr+i] = m_myVmePort->read32(hpidAddr);
    }

    if (endianSwap) {
      for (int i=0; i<blockWordCount; i++) {
        buffer[wordIncr+i] = endianReverse32(buffer[wordIncr+i]);
      }
    }
*/
  }
  return;
};

//------------------------------mdspBlockWrite-----------------------------------  

void RodModule::mdspBlockWrite(unsigned long dspAddr, unsigned long buffer[], 
       long wordCount, HpidMode mode) throw (HpiException &, 
       VmeException &) { 

  unsigned long hpidAddr;
  long myCount, localCount, blockWordCount, wordIncr;

//  Load the initial dsp address into the HPIA register 
  hpiLoad(HPIA, dspAddr);

// Determine the HPI mode to use.
  switch (mode) {
    case AUTO:
      hpidAddr= HPID_AUTO;
      break;
    case NO_AUTO:
      hpidAddr = HPID_NOAUTO; 
      break;
    case DYNAMIC:
    default:
      if (wordCount == 1){
        hpidAddr = HPID_NOAUTO;
      }	
      else { 
        hpidAddr = HPID_AUTO;
      };
      break;
  };

// Check if wordcount is odd and, if so, first do a single word write.

  localCount = wordCount;
  wordIncr = 0;
  if (wordCount%2 !=0) {
    m_myVmePort->write32(hpidAddr, buffer[0]);
    wordIncr = 1;
    localCount -= 1;
  }

// Check to see that we don't access too large a block.
// MAX_HPID_WORD_ELEMENTS is defined in vmeAddressMap.h

  blockWordCount = std::min(wordCount, MAX_HPID_WORD_ELEMENTS);

//	Set up the transfer as a series of block transfers 
  for (myCount=localCount; myCount>0;
                    myCount-=blockWordCount, wordIncr+=blockWordCount) {

//	Recheck each block 
    blockWordCount = std::min(myCount, MAX_HPID_WORD_ELEMENTS);
  
//	Do the transfer for this block 
  if (endianSwap) {
    for (int i=0; i<blockWordCount; i++) {
      buffer[wordIncr+i] = endianReverse32(buffer[wordIncr+i]);
    }
  }

// Block writes are being done in hardware as from 10 June 03.

    m_myVmePort->blockWrite32(hpidAddr, &buffer[wordIncr], blockWordCount*4);
    
// Use this code if harfdware block transfers are a problem.
/*
    for (int i=0; i<blockWordCount; i++) {
      m_myVmePort->write32(hpidAddr, buffer[wordIncr+i]);
    }
*/
  }
  return;
};

//-----------------------------mdspBlockDump--------------------------------                                

void RodModule::mdspBlockDump(const unsigned long firstAddress, 
       const unsigned long lastAddress,
       const std::string & fileName) throw(RodException &, VmeException &) {
       
  long numBytes;
  std::ofstream fout;
  numBytes = lastAddress - firstAddress + 1;
  fout.open(fileName.c_str(), std::ios::binary);
  if (fout.is_open()) {
    unsigned long * buffer;
    try {
      buffer = new unsigned long[numBytes];
    }
    catch (std::bad_alloc & ba) {
      throw RodException(
       "mdspBlockDump failed to allocate buffer; buffer size in bytes=", numBytes);
    }   
    mdspBlockRead(firstAddress, buffer, numBytes/4);
    fout.write((char*)buffer, numBytes);
    fout.close();
    delete [] buffer;
  }
  else {
    throw RodException("mdspBlockDump failed to open file");
  }
  return;                    
}

//--------------------------------slvHpiLoad-----------------------------------                                

void RodModule::slvHpiLoad(unsigned long hpiReg, unsigned long hpiValue,
                           long slaveNum) throw(VmeException &) {

  unsigned long address;
  address = hpiReg + slaveNum*SLAVE_HPI_OFFSET;
  
  mdspSingleWrite(address, hpiValue);
};

//------------------------------slvHpiFetch------------------------------------

unsigned long RodModule::slvHpiFetch(unsigned long hpiReg, long slaveNum) 
              throw(VmeException &) {

  unsigned long hpiValue;
  unsigned long address;
  address = hpiReg + slaveNum*SLAVE_HPI_OFFSET;
  
    hpiValue=mdspSingleRead(address);

  return hpiValue;
};

//-------------------------------slvSingleRead---------------------------------
unsigned long RodModule::slvSingleRead(unsigned long dspAddr, long slaveNum ) 
              throw(VmeException &) {

  unsigned long slvHpia, slvHpid;
  unsigned long value;
  slvHpia = SLAVE_HPIA_BASE + slaveNum*SLAVE_HPI_OFFSET;
  slvHpid = SLAVE_HPID_NOAUTO_BASE + slaveNum*SLAVE_HPI_OFFSET;

//  Load the DSP address into the HPIA register 
  mdspSingleWrite(slvHpia, dspAddr);

//	Do the read 
  value = mdspSingleRead(slvHpid);

  return value;
};

//------------------------------slvSingleWrite------------------------------------                                   

void RodModule::slvSingleWrite(unsigned long dspAddr, unsigned long buffer, 
                 long slaveNum) throw(VmeException &) { 

  unsigned long slvHpia, slvHpid;
  slvHpia = SLAVE_HPIA_BASE + slaveNum*SLAVE_HPI_OFFSET;
  slvHpid = SLAVE_HPID_NOAUTO_BASE + slaveNum*SLAVE_HPI_OFFSET;

  mdspSingleWrite(slvHpia, dspAddr);

//	Do the write 
  mdspSingleWrite(slvHpid, buffer);

  return;
};

//-----------------------------slvBlockRead---------------------------------                                

void RodModule::slvBlockRead(const unsigned long dspAddr, unsigned long buffer[],
                const long wordCount, long slaveNum, HpidMode mode) 
                throw (HpiException &, VmeException &) {
                
  unsigned long hpidAddr, dspAddrLocal, hpiaAfterFetch, hpiaAfterCalc; 
  long myCount, blockWordCount, wordIncr;
  unsigned long slvHpia, slvHpidAuto, slvHpidNoAuto;
  slvHpia = SLAVE_HPIA_BASE + slaveNum*SLAVE_HPI_OFFSET;
  slvHpidAuto = SLAVE_HPID_AUTO_BASE + slaveNum*SLAVE_HPI_OFFSET;
  slvHpidNoAuto = SLAVE_HPID_NOAUTO_BASE + slaveNum*SLAVE_HPI_OFFSET;


// Determine the HPI mode to use.
  switch (mode) {
    case AUTO:
      hpidAddr= slvHpidAuto;
      break;
    case NO_AUTO:
      hpidAddr = slvHpidNoAuto; 
      break;
    case DYNAMIC:
    default:
      if (wordCount == 1){
        hpidAddr = slvHpidNoAuto;
      }	
      else { 
        hpidAddr = slvHpidAuto;
      };
      break;
  };

//  Load the DSP address into the HPIA register 
  mdspSingleWrite(slvHpia, dspAddr);

// Check to see that we don't access too large a block. HPID has 20 bits of 
// address space available. MAX_HPID_WORD_ELEMENTS is defined in RodVmeAddresses.h

  blockWordCount = std::min(wordCount, MAX_HPID_WORD_ELEMENTS);

//	Set up the transfer as a series of block transfers 
  for (myCount=wordCount, wordIncr=0; myCount>0;
                    myCount-=blockWordCount, wordIncr+=blockWordCount) {

//	Recheck each block 
    blockWordCount = std::min(myCount, MAX_HPID_WORD_ELEMENTS);

//	Do the transfer for this block 
    dspAddrLocal = dspAddr+ (4*wordIncr);
    mdspBlockRead(hpidAddr, &buffer[wordIncr], blockWordCount, NO_AUTO);
  }

//  Check HPID after transfer 
  hpiaAfterFetch = mdspSingleRead(slvHpia);
  if (wordCount == 1) {
    hpiaAfterCalc = dspAddr;
  }	else  {
    hpiaAfterCalc = dspAddr+wordCount*4;
  }
/*  if (hpiaAfterFetch != hpiaAfterCalc) {
    throw HpiException("mslvBlockRead address check failed", hpiaAfterCalc, 
                       hpiaAfterFetch);
  }
*/
  return;
};

//------------------------------slvBlockWrite-----------------------------------  

void RodModule::slvBlockWrite(unsigned long dspAddr, unsigned long buffer[], 
         long wordCount, long slaveNum, HpidMode mode) 
         throw (HpiException &, VmeException &) { 

  unsigned long hpidAddr, dspAddrLocal, hpiaAfterFetch, hpiaAfterCalc;
  long myCount, blockWordCount, wordIncr;
  unsigned long slvHpia, slvHpidAuto, slvHpidNoAuto;
  slvHpia = SLAVE_HPIA_BASE + slaveNum*SLAVE_HPI_OFFSET;
  slvHpidAuto = SLAVE_HPID_AUTO_BASE + slaveNum*SLAVE_HPI_OFFSET;
  slvHpidNoAuto = SLAVE_HPID_NOAUTO_BASE + slaveNum*SLAVE_HPI_OFFSET;

//  Load the initial dsp address into the HPIA register 
  mdspSingleWrite(slvHpia, dspAddr);

// Determine the HPI mode to use.
  switch (mode) {
    case AUTO:
      hpidAddr= slvHpidAuto;
      break;
    case NO_AUTO:
      hpidAddr = slvHpidNoAuto; 
      break;
    case DYNAMIC:
    default:
      if (wordCount == 1){
        hpidAddr = slvHpidNoAuto;
      }	
      else { 
        hpidAddr = slvHpidAuto;
      };
      break;
  };

// Check to see that we don't access too large a block. HPID has 20 bits of 
// address space available. MAX_HPID_WORD_ELEMENTS is defined in vmeAddressMap.h

  blockWordCount = std::min(wordCount, MAX_HPID_WORD_ELEMENTS);

//	Set up the transfer as a series of block transfers 
  for (myCount=wordCount, wordIncr=0; myCount>0;
                    myCount-=blockWordCount, wordIncr+=blockWordCount) {

//	Recheck each block 
    blockWordCount = std::min(myCount, MAX_HPID_WORD_ELEMENTS);
  
//	Do the transfer for this block 
  dspAddrLocal = dspAddr+ (4*wordIncr);
    mdspBlockWrite(hpidAddr, &buffer[wordIncr], blockWordCount, NO_AUTO);
  }
  
//  Check HPID after transfer 
  hpiaAfterFetch = mdspSingleRead(slvHpia);
	
  if (wordCount == 1) {
    hpiaAfterCalc = dspAddr;
  }	
  else  {
    hpiaAfterCalc = dspAddr+wordCount*4-4;
  }
  
/*  if (hpiaAfterFetch != hpiaAfterCalc) {
    throw HpiException("mdspBlockWrite address check failed", hpiaAfterCalc, 
                        hpiaAfterFetch);
  }
*/	
  return;
};

//------------------------------resetMasterDsp--------------------------------                                

void RodModule::resetMasterDsp() throw(RodException&, VmeException &) {
  unsigned long value=0;
  unsigned long rodRegValue;
  clock_t startTime;
  
  rodRegValue = readRodStatusReg(0);
  setBit(&value, 1);
  if (endianSwap) {
    value = endianReverse32(value);
  }
  m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[2], value);
  sleep(4000);
  rodRegValue = 0;
  startTime = clock();
  do {
    rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[2]);
    if ((clock()-startTime)/CLOCKS_PER_SEC > DSP_RESET_TIMEOUT) {
      throw RodException("DSP reset timeout in resetMasterDsp(), DSP_RESET_TIMEOUT=",
      (long)DSP_RESET_TIMEOUT);
    }
  }
  while (rodRegValue != 0x3e);
  rodRegValue = readRodStatusReg(0);
  
  return;
};

//------------------------------resetSlaveDsp--------------------------------                                

void RodModule::resetSlaveDsp(long slaveNumber) throw(RodException&, VmeException &) {
  unsigned long value=0;
  unsigned long rodRegValue;
  clock_t startTime;
  
  setBit(&value, 2+slaveNumber);
  if (endianSwap) {
    value = endianReverse32(value);
  }
  m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[2], value);
  sleep(4000);
  rodRegValue = 0;
  startTime = clock();
  do {
    rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[2]);
    if ((clock()-startTime)/CLOCKS_PER_SEC > DSP_RESET_TIMEOUT) {
      throw RodException("DSP reset timeout in resetSlaveDsp(), DSP_RESET_TIMEOUT=",
      (long)DSP_RESET_TIMEOUT);
    }
  }
  while (rodRegValue != 0x3e);
  
  return;
};


//------------------------------resetAllDsps--------------------------------                                

void RodModule::resetAllDsps() throw(RodException &, VmeException &) {
  unsigned long value=0;
  unsigned long rodRegValue;
  clock_t startTime;
  
  setBit(&value, 6);
  if (endianSwap) {
    value = endianReverse32(value);
  }
  m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[2], value);
  sleep(4000);
  rodRegValue = 0;
  startTime = clock();
  do {
    rodRegValue = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[2]);
    if ((clock()-startTime)/CLOCKS_PER_SEC > DSP_RESET_TIMEOUT) {
      throw RodException("DSP reset timeout in resetAllDsps(), DSP_RESET_TIMEOUT=",
      (long)DSP_RESET_TIMEOUT);
    }
  }
  while (rodRegValue != 0x3e);
  
  return;
};

//-------------------------------chipEraseHpi---------------------------------

void RodModule::chipEraseHpi() throw(VmeException &) {
  unsigned long value = 0x10;
  
// Commands 1 to 5
  commonEraseCommandsHpi(MDSP_FLASH_BOTTOM);
  
// Command 6
  mdspSingleWrite(MDSP_FLASH_BOTTOM+(0x5555<<2), value);
  
// Wait for operation to complete
  sleep(CHIP_ERASE_TIME_MS);
  return;
};
  
//--------------------------------sectorErase---------------------------------

void RodModule::sectorErase(unsigned long sectorBaseAddress) 
                            throw(RodException &, VmeException &) {
  unsigned long flashBaseAddress, valueD32;
  bool busyBit;

// Get flash base address
  switch (m_revision) {
    case 0x0e: 
      flashBaseAddress = FPGA_FLASH_REL_ADDR_REVE;
      break;
    case 0x0b:
    case 0x0c:
    default: 
      if ((sectorBaseAddress<FPGA_FLASH_0_BOTTOM)||
            (sectorBaseAddress>FPGA_FLASH_2_BOTTOM+FLASH_MEMORY_SIZE)) {
        throw RodException("Flash sector base addr out of range, sectorBaseAddress=",
             sectorBaseAddress);
      }
      if (sectorBaseAddress<FPGA_FLASH_1_BOTTOM) {
        flashBaseAddress = FPGA_FLASH_0_BOTTOM;
      }
      else if (sectorBaseAddress<FPGA_FLASH_2_BOTTOM) {
        flashBaseAddress = FPGA_FLASH_1_BOTTOM;
      }
      else flashBaseAddress = FPGA_FLASH_2_BOTTOM;
      break;
  }
// Commands 1 to 5
  commonEraseCommands(flashBaseAddress);
  
// Set write bit
  vmeWriteElementFlash(0x30, sectorBaseAddress, WRITE_COMMAND_HANDSHAKE_BIT);
  
// Wait for operation to complete
  switch (m_revision ) {
    case 0x0e:
      do {                           /* Poll busy */ 
        valueD32 = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[4]); 
        busyBit = readBit(valueD32, 3);                
      } while (busyBit);
      break;
    case 0x0b:
    case 0x0c: 
    default:
      sleep(SECTOR_ERASE_TIME_MS);
      break;
  }

  return;
};
  
//---------------_-------------writeByteToFlash-------------------------------

void RodModule::writeByteToFlash(unsigned long address, UINT8 data) 
                            throw (RodException &, VmeException &) {
  UINT8 readData;
  clock_t startTime;   // start time for DQ7 data polling, in seconds
  long updateAddress;
  
  if (data!=0xFF) {   // if erased, don't rewrite. 0xFF is default after erasing
// write data
    vmeWriteElementFlash(data, address, WRITE_DATA_HANDSHAKE_BIT);
    updateAddress = 0;
  }
  else {
    updateAddress = 1;
  }
  
// Wait for operation to complete - data polling
  startTime = clock();
  while(1) {
    readData = readByteFromFlash(address, updateAddress);
    if (readData == data) return;
    else if((clock()-startTime)/CLOCKS_PER_SEC > FLASH_TIMEOUT) {
      throw RodException("Flash timeout in writeByteToFlash, FLASH_TIMEOUT=",
      (long)FLASH_TIMEOUT);
    }
  }
  return;
};

//----------------------------writeBlockToFlash-------------------------------

void RodModule::writeBlockToFlash(unsigned long address, UINT8 *data, 
                               unsigned long numBytes) throw(RodException &, VmeException &) {
  unsigned long index, sectorSize;
  switch (m_revision) {
    case 0x0e:
      sectorSize = FLASH_SECTOR_SIZE_REVE;
      break;
    case 0x0b:
    case 0x0c:
    default:
      sectorSize = FLASH_SECTOR_SIZE;
      break;
  }
  for (index=0; index<numBytes; ++index) {
    if ((index%sectorSize)==0) {
      sectorErase(address+index);
    }
    writeByteToFlash(address+index, *(data+index));
  } 
  return;
};
  
//---------------------------writeBlockToFlashHpi-----------------------------

void RodModule::writeBlockToFlashHpi(unsigned long address, UINT8 *data, 
     unsigned long numBytes) throw (RodException &, VmeException &) {
  unsigned long index, sectorAddr;
  const unsigned long eraseData=0x30;
  const unsigned long data1=0xAA;
  const unsigned long data2=0x55;
  const unsigned long data3=0xA0;
  const unsigned long addr1=MDSP_FLASH_BOTTOM+(0x5555<<2);
  const unsigned long addr2=MDSP_FLASH_BOTTOM+(0x2AAA<<2);
  unsigned long byteRelAddr;
  long words;
  UINT8 dataVal, verifyVal;
  unsigned long longVal, busyVal;
  
  for (index=0; index<numBytes; ++index) {
    byteRelAddr = address+index-MDSP_FLASH_BOTTOM;
    if ((index%FLASH_SECTOR_SIZE)==0) {
// implement sectorEraseHpi(address+index) inline.
//    commands 1 to 5
      commonEraseCommandsHpi(MDSP_FLASH_BOTTOM);
// 6th command
      sectorAddr = MDSP_FLASH_BOTTOM+
                  (byteRelAddr<<2);
      mdspSingleWrite(sectorAddr, eraseData);
// Wait for operation to complete
      sleep(SECTOR_ERASE_TIME_MS);
    };
 
// implement writeByteToFlashHpi(address+index, *(data+index)) inline.
    dataVal = *(data+index);
    longVal = dataVal&0x000000FF;
    if (dataVal != 0xFF) {
      mdspSingleWrite(addr1, data1);                      // 1st command
      mdspSingleWrite(addr2, data2);                      // 2nd command
      mdspSingleWrite(addr1, data3);                      // 3rd command
      mdspSingleWrite(MDSP_FLASH_BOTTOM+(byteRelAddr<<2), longVal);
// Verify data by reading it back before continuing. Flash memory may need up to
// 20 microseconds to complete the write sequence.
      for (int i=0; i<4000; i++) {
        busyVal = mdspSingleRead(MDSP_FLASH_BOTTOM+byteRelAddr);
        switch (byteRelAddr%4) {
          case 1: busyVal = busyVal >> 8;
                  break;
          case 2: busyVal = busyVal >> 16;
                  break;
          case 3: busyVal = busyVal >> 24;
                  break;
          default: break;
        }
        busyVal = busyVal & 0x000000FF;
        if (busyVal == longVal) break;
      }
    };
  };
    
// Verification
  words = (numBytes+3)/4;       // round up
  UINT8 *buffer;
  try {
    buffer = new UINT8[words*4];
  }
  catch (std::bad_alloc & ba) {
    throw RodException("writeBlockToFlashHpi unable to get buffer.");
  }
  mdspBlockRead(address, (unsigned long*)buffer, words);
  for (index=0; index<numBytes; ++index) {
    dataVal = *(data+index);
    verifyVal = *(buffer+index);
    if (dataVal != verifyVal) {
      delete [] buffer;
      throw RodException("writeBlockToFlashHpi verify failed. index, data:", index,
                 dataVal);
    };
  }
  delete [] buffer;
  return;
};
  
//-----------------------------readByteFromFlash------------------------------

UINT8 RodModule::readByteFromFlash(unsigned long address, long updateAddress) 
                 throw (RodException &, VmeException &){
  UINT8 dataByte;
  unsigned long commandReg;
  clock_t startTime;
  unsigned long handshakeBitValue = 0;
  unsigned long valueD32;
  
  if (updateAddress) {
    m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[4], address);
  }
// Set rd bit
    setBit(&handshakeBitValue, READ_HANDSHAKE_BIT);
    m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[3], handshakeBitValue);
    
// Wait for valid data
    startTime = clock();
    while (1) {
      commandReg = m_myVmePort->read32(FPGA_CONTROL_REG_REL_ADDR[3]);
      if (0==readBit(commandReg, READ_HANDSHAKE_BIT)) break;
      if ((clock()-startTime)>FLASH_TIMEOUT) throw RodException(
                          "Timeout in readByteFromFlash. Address=", address);
    }
    
// Read valid data
  valueD32 = m_myVmePort->read32(FPGA_STATUS_REG_REL_ADDR[7]); 
  dataByte = (UINT8)(valueD32&0xFF);
  return dataByte;
};
  
//---------------------------vmeWriteElementFlash------------------------------

void RodModule::vmeWriteElementFlash(UINT8 value, unsigned long address, 
           long handshakeBit) throw (RodException &, VmeException &) {
  unsigned long ctrlReg4Val;  // address(23:0) + data(31:24)
  unsigned long commandReg;
  clock_t startTime;
  unsigned long handshakeBitValue=0;
  
  ctrlReg4Val = (value<<24)| address;
  m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[4], ctrlReg4Val);

// Set wr bit
  setBit(&handshakeBitValue, handshakeBit);
  m_myVmePort->write32(FPGA_CONTROL_REG_REL_ADDR[3], handshakeBitValue);

// Wait for ack
  startTime = clock();
  while(1) {
      commandReg = m_myVmePort->read32(FPGA_CONTROL_REG_REL_ADDR[3]);
      if (0==readBit(commandReg, handshakeBit)) break;
      if ((clock()-startTime)>FLASH_TIMEOUT) throw RodException(
                   "Timeout in vmeWriteElementFlash. Address, value=", 
                   address, (unsigned long)value);
  }
  return;   
};

//-----------------------------readBlockFromFlash------------------------------

void RodModule::readBlockFromFlash(unsigned long address, UINT8 *buffer, 
                                   unsigned long numBytes) throw(RodException &, VmeException &) {
  unsigned long index;
  for (index=0; index<numBytes; ++index) {
    buffer[index] = readByteFromFlash(address+index, 1);
  }
  return;
};
  
//-----------------------------commonEraseCommands----------------------------

void RodModule::commonEraseCommands(unsigned long flashBaseAddr) 
                                           throw(RodException &, VmeException &) {

  const unsigned long addr1 = flashBaseAddr+0x5555;
  const unsigned long addr2 = flashBaseAddr+0x2AAA;
  
// 1st command
  vmeWriteElementFlash(0xAA, addr1, WRITE_COMMAND_HANDSHAKE_BIT);
  
// 2nd command
  vmeWriteElementFlash(0x55, addr2, WRITE_COMMAND_HANDSHAKE_BIT);

// 3rd command
  vmeWriteElementFlash(0x80, addr1, WRITE_COMMAND_HANDSHAKE_BIT);
  
// 4th command
  vmeWriteElementFlash(0xAA, addr1, WRITE_COMMAND_HANDSHAKE_BIT);

// 5th command
  vmeWriteElementFlash(0x55, addr2, WRITE_COMMAND_HANDSHAKE_BIT);

  return;
};

//----------------------------commonEraseCommandsHpi--------------------------

void RodModule::commonEraseCommandsHpi(unsigned long flashBaseAddr) 
                                              throw(VmeException &) {

  unsigned long buffer;
  const unsigned long addr1 = flashBaseAddr+(0x5555<<2);
  const unsigned long addr2 = flashBaseAddr+(0x2AAA<<2);
  
// 1st command
  buffer = 0xAA;
  mdspSingleWrite(addr1, buffer);
  
// 2nd command
  buffer = 0x55;
  mdspSingleWrite(addr2, buffer);

// 3rd command
  buffer = 0x80;
  mdspSingleWrite(addr1, buffer);
  
// 4th command
  buffer = 0xAA;
  mdspSingleWrite(addr1, buffer);

// 5th command
  buffer = 0x55;
  mdspSingleWrite(addr2, buffer);

  return;
};

//---------------------------getFlashSectorSize---------------------------                                

  unsigned long RodModule::getFlashSectorSize() {
  unsigned long sectorSize;
  switch (m_revision) {
    case 0x0e:
      sectorSize = FLASH_SECTOR_SIZE_REVE;
      break;
    case 0x0b:
    case 0x0c:
    default:
      sectorSize = FLASH_SECTOR_SIZE;
      break;
  }
  return sectorSize;
};

//----------------------------------sleep-------------------------------------                                

void RodModule::sleep(const double milliSecs) {
  clock_t start, delay;
  delay = CLOCKS_PER_SEC;
  delay = clock_t(milliSecs * CLOCKS_PER_SEC / 1000 ); //need explicit type cast
                                                       //to avoid warning.
  start = clock();
  while (clock()-start< delay)   // wait until time elapses
     ;
  return;
};
  
//--------------------------------checkSum------------------------------------                                

unsigned long RodModule::checkSum(const unsigned long *sourceArray, const long wordCount) {
  unsigned long result=0;
  long i;
  
  for (i=0; i<wordCount; ++i) {
    result ^= sourceArray[i];
  }
  return result;
};
 
//-----------------------------endianReverse32----------------------------------                                

unsigned long RodModule::endianReverse32(const unsigned long inVal) {
  unsigned long outVal;
  outVal  = (inVal & 0x000000ff) << 24;
  outVal |= (inVal & 0x0000ff00) << 8;
  outVal |= (inVal & 0x00ff0000) >> 8;
  outVal |= (inVal & 0xff000000) >> 24;
  return outVal;
};
  
} //  End namespace SctPixelRod

//------------------------- Overload insertion operators "<<" -------------------------
/* This overloaded operator lets us use cout to print the status of the ROD
*/
  std::ostream& operator<<(std::ostream& os, SctPixelRod::RodModule& rod) {
    os << "Slot: " << rod.getSlot() << std::endl;
    os << "Serial Number:" << rod.getSerialNumber() << std::endl;
    os << "Number of slave DSPs: " << rod.getNumSlaves() << std::endl;
    os << "Status registers[0-2]: ";
    try {
      for (int i=0; i<3; i++) {
        os << std::hex << rod.readRodStatusReg(i) << " ";
      }
      os << std::endl;
    }
    catch (SctPixelRod::VmeException &) {
      os << "VmeException while reading ROD status registers. " << std::endl;
    }
    
    os << "Command registers[0-1]: ";
    try {
      for (int i=0; i<2; i++) {
        os << std::hex << rod.readRodCommandReg(i) << " ";
      }
      os << std::endl;
    }
    catch (SctPixelRod::VmeException &) {
      os << "VmeException while reading ROD command registers. " << std::endl;
    }

    os << "Primitive state: " << rod.getPrimState() << " Text State: " << 
          rod.getTextState() << std::dec << std::endl;
    return os;
  }

/* This overloaded operator lets us use cout to print the RodException information. We use the virtual
 * what() method to get the information from the derived class instead of BaseException.
*/
/*  std::ostream& operator<<(std::ostream& os, SctPixelRod::RodException& rodEx) {
    return rodEx.what(os, rodEx);
  }
*/
/* This overloaded operator lets us use cout to print the HpiException information
*/
/*  std::ostream& operator<<(std::ostream& os, SctPixelRod::HpiException& hpiEx) {
    return hpiEx.what(os,hpiEx);
  }
*/
/* This overloaded operator lets us use cout to print the NoImagerFile exception information
*/
/*  std::ostream& operator<<(std::ostream& os, SctPixelRod::NoImageFile& imaEx) {
    return imaEx.what(os, imaEx);
  }
*/

