/************************************************************************************
 * utilities.h
 *
 *   synopsis: Function prototypes and #defines for general purpose utility routines
 *             and macros.
 ************************************************************************************/
#ifndef UTILITIES
#define UTILITIES

#include "processor.h"

/* timing macros: delta t & delay in tenths of microseconds */
#ifdef CCS2
	#if ( (defined(I_AM_MASTER_DSP))||((defined(REV_B))||(defined(REV_C))) )
		#define delta_t(t0_) ((TIMER_getCount(timer1) - (t0_)) / 4)

		#define delay(x) { \
		    UINT32 delay_t0; \
		    delay_t0 = TIMER_getCount(timer1); \
		    while( ((TIMER_getCount(timer1) - delay_t0) / 4) < (x)); \
		}

	/* Rev. E RODs have a bug in the silicon when using the timers: if the period
	   is set to the maximum (0xffffffff), and the timer count is greater than
	   ~0x90000000, then the DSP lapses into a state where it stalls for up to
	   600 micro-seconds on certain function calls. Using a smaller period in the
	   timer prevents this, but means that we must use an inlined function for
	   delta_t instead of a macro since if it wraps the subtraction will no longer
	   automatically handle it (delay is made into an inlined function too).
	   The inlined functions need access to timer1; this is done using pointers
	   to maintain transparency in the rest of the program. */
	#elif (defined(REV_E))

		static inline UINT32 delta_t(UINT32 t0);
		static inline void   delay(UINT32 x);
		
		static inline UINT32 delta_t(UINT32 t0) {
			UINT32 t1, p1;
			t1= *((UINT32 *) 0x01980008);  //timer 1 count
			p1= *((UINT32 *) 0x01980004);  //timer 1 period

			if (t1 < t0) return ((t1 -t0 +p1)/5);
			else         return ((t1 -t0)/5);
		}

		
		static inline void delay(UINT32 x) {
			UINT32 t0, t1, delta, p1;
			t0= *((UINT32 *) 0x01980008);  //timer 1 count
			p1= *((UINT32 *) 0x01980004);  //timer 1 period

			do {
				t1= *((UINT32 *) 0x01980008);
				if (t1 < t0) delta= ((t1 -t0 +p1)/5);
				else         delta= ((t1 -t0)/5);
			} while (delta < x);
		}

	#endif
	
	/* local macro variable must not conflict with others? The backslash MUST be the
	   last character on the line, or an odd compile error will result */
#else
	#define delta_t(t0_) ((TIMER_GetCount(timer1) - (t0_)) / 4)
	
	/* local macro variable must not conflict with others? The backslash MUST be the
	   last character on the line, or an odd compile error will result */
	#define delay(x) { \
	    UINT32 delay_t0; \
	    delay_t0 = TIMER_GetCount(timer1); \
	    while( ((TIMER_GetCount(timer1) - delay_t0) / 4) < (x)); \
	}
#endif

/* led macros for quick timing calls */
#if defined(I_AM_MASTER_DSP)
	#define yellowLed_on     (*((UINT32 *) 0x018c0024)|= 1);
	#define greenLed_on      (*((UINT32 *) 0x01900024)|= 1);
	#define redLed_on        (*((UINT32 *) 0x018c0024)|= 4);
	
	#define yellowLed_off    (*((UINT32 *) 0x018c0024)&= ~1);
	#define greenLed_off     (*((UINT32 *) 0x01900024)&= ~1);
	#define redLed_off       (*((UINT32 *) 0x018c0024)&= ~4);
	
	#define yellowLed_toggle (*((UINT32 *) 0x018c0024)^= 1);
	#define greenLed_toggle  (*((UINT32 *) 0x01900024)^= 1);
	#define redLed_toggle    (*((UINT32 *) 0x018c0024)^= 4);

#elif defined(I_AM_SLAVE_DSP)
	#define fsrp0_on         (*((UINT32 *) 0x018c0024)|=  4);
	#define fsxp0_on         (*((UINT32 *) 0x018c0024)|=  8);

	#define fsrp0_off        (*((UINT32 *) 0x018c0024)&= ~4);
	#define fsxp0_off        (*((UINT32 *) 0x018c0024)&= ~8);

	#define fsrp0_toggle     (*((UINT32 *) 0x018c0024)^=  4);
	#define fsxp0_toggle     (*((UINT32 *) 0x018c0024)^=  8);

	//Rev. B & C RODs have the yellow LED attached to CLKXP1, while Rev. E RODs
	//use external pin #7 as a GPI/O output to drive the LED. However, this is
	//not modelled in the simulation, so the SP is used instead.
	#if (defined(REV_B)||defined(REV_C)||defined(SIM))
		#define yellowLed_on     (*((UINT32 *) 0x01900024)&= ~2);
		#define yellowLed_off    (*((UINT32 *) 0x01900024)|=  2);
		#define yellowLed_toggle (*((UINT32 *) 0x01900024)^=  2);
	#elif (defined(REV_E))
		#define yellowLed_on     (*((UINT32 *) 0x01b00008)&= ~0x80);
		#define yellowLed_off    (*((UINT32 *) 0x01b00008)|=  0x80);
		#define yellowLed_toggle (*((UINT32 *) 0x01b00008)^=  0x80);
	#endif
#endif

/* register access width (data bus width of EMIF) */
#define BYTE_W    8
#define HWORD_W  16
#define WORD_W   32

/* default value for primitive parameters, function arguments, etc */
#define DEFAULT (0xFFFFFFFF)

/* z is returned as lesser of x or y */
#define MINI(x,y,z)   z=((x)<(y))?(x):(y);

/* macros to set, reset and read fields of variables and registers */
#define N_BIT_MASK(wid)           (~(0xFFFFFFFF<<(wid)))
#define FIELD_MASK(bit,wid)       (N_BIT_MASK(wid) << (bit))
#define FIELD_OFF(bit,wid)        (~(FIELD_MASK(bit,wid)))
#define FIELD_VAL(bit,wid,val)    ((val&(N_BIT_MASK(wid))) << bit)

#define SET_VBIT(var,bit)        (var |= (1 << bit))
#define RST_VBIT(var,bit)        (var &= ~(1 << bit))
#define ASGN_VBIT(var,bit,val)   (var = ((RST_VBIT(var,bit)) | (val<<bit)))
#define GET_VBIT(var,bit)        ((var >> bit) & 1)
#define RST_VFLD(var,bit,wid)    (var &= (FIELD_OFF(bit,wid)))
#define ASGN_VFLD(var,bit,wid,val)  \
             (var = (RST_VFLD(var,bit,wid)) | (FIELD_VAL(bit,wid,val)))
#define GET_VFLD(var,bit,wid)    ((FIELD_MASK(bit,wid) & var) >> bit)

/*  Some of the macros below are essentially the same as those in the Texas
 * Instruments provided regs.h.  The ones in this file should be used in DSP code
 * authored by the ROD group.  If Texas Instruments changes their macro names for
 * some reason, the DSP code will not be affected.                  dpf             */
#define CONTENTS(adr) (*((volatile UINT32 *)(adr)))

#define READ_REG(adr)             (CONTENTS(adr))
#define WRITE_REG(adr,val)        (CONTENTS(adr) = (val))
#define SET_RBIT(adr,bit)         (CONTENTS(adr) |= (1 << (bit)))
#define RST_RBIT(adr,bit)         (CONTENTS(adr) &= (~(1 << (bit))))
#define ASGN_RBIT(adr,bit,val)    (CONTENTS(adr) = (RST_RBIT(adr,bit)) | (val<<bit))
#define GET_RBIT(adr,bit)         ((CONTENTS(adr) >> bit) & 1)
#define RST_RFLD(adr,bit,wid)     (CONTENTS(adr) &= (FIELD_OFF(bit,wid)))
#define GET_RFLD(adr,bit,wid)     ((CONTENTS(adr) & FIELD_MASK(bit,wid)) >> bit)
#define ASGN_RFLD(adr,bit,wid,val) (CONTENTS(adr) = \
                                   (RST_RFLD(adr,bit,wid)) | (FIELD_VAL(bit,wid,val)))

/* Contexts for waitRegister: */
#define  CONTEXT_NULL   0
#define  CONTEXT_TASK   1
#define  CONTEXT_PRIM   2
#define  CONTEXT_ALL    3


/* The copymem & setmem EDMA transfer complete codes. The only available method
   of monitoring the progress of an EDMA transfer is to allow an interrupt upon
   completion, and then in the interrupt check the TCC to decide what to do. */
//#define EDMA_SETMEM_ID     14  dpsf not true...
//#define EDMA_COPYMEM_ID    15

#endif
