/************************************************************************************
 * Title:   scanControl.h
 * Version: 4th September 2003
 *
 * Description:   ROD Master DSP scan configuration structure
 *                and constant definitions. See sctScanStructures.h,
 *    ABCDscans.h & pixelScanStructures.h for scan parameter definitions
 *    and detector-specific scan structures. 
 ************************************************************************************/
#ifndef SCAN_CONTROL_H	 /* multiple inclusion protection */
#define SCAN_CONTROL_H
	#include "processor.h"
	#if   defined(SCT_ROD)
		#include "ABCD/ABCDscans.h"
	#elif defined(PIXEL_ROD)
		#include "pixelScanStructures.h"
	#endif

#define N_CMD_LIST_CMDS   6
struct CmdList {
	UINT16  cmd[N_CMD_LIST_CMDS];
	UINT32 data[N_CMD_LIST_CMDS];
};

/* Scan control structures. For SCT the dual parameters in ScanGeneral (scanParameter,
   scanStart, scanEnd, nBins, dataPtr) and ScanDspDistribution (groupRangeMap) are used
   to implement power & load balancing on the SDSPs: 4 groups of modules are scanned
   upward together (1/SDSP), while a second set of 4 groups is scanned downward. For
   Pixels, the dual parameters in ScanGeneral describe the inner and outer loops
   ([0] & [1] respectively) for a two dimensional scan. The groupRangeMap parameter
   is unused; all module groups use the same mapping. */
typedef struct  {
	UINT8  configSet;         /* The configuration set to use (INIT/WORK/SPARE) */
	UINT8  reportErrors;      /* Toggle to enable data integrity checks on SDSPs */         
	UINT8  globalCtrl;        /* Determines whether MDSP or TIM sends triggers */         
	UINT8  moduleScanMode;    /* Concurrent or FE by FE */

	UINT8  scanParameter[2];  /* Scan parameter specifier */
	UINT8  uniformPoints[2];     /* If TRUE scanStart, scanEnd & nBins determine points */

	MDAT32 scanStart[2];      /* Start value for uniform scan */
	MDAT32 scanEnd[2];        /* End value for uniform scan */

	#if   defined(SCT_ROD)
		UINT16 nBins[2];          /* # of bins in inner & outer loop */
		//UINT16 nBins, unused;          /* # of bins */ 
	#elif defined(PIXEL_ROD)
		UINT16 nBins[2];          /* # of bins in inner & outer loop */
	#endif
	UINT32 repetitions;       /* # of events per bin */
	MDAT32 *dataPtr[2];       /* pointers to data for custom scans; 0xffffffff
	                             (DEFAULT) indicates that data is in the primitive list
	                             right after the scan structure. */

	UINT8  maskMode;          /* Staged or static */
	UINT8  stageAdvanceFirst; /* Indicates whether the innermost loop is the mask stage
	                             or bin[0] parameter */
	UINT16 maskStages;        /* Number of mask stages to actually execute in scan */

	#if   defined(SCT_ROD)
	UINT8 stage0;             /* Initial Mask stage. */
	UINT8 unused1;
	#elif defined(PIXEL_ROD)
	UINT8  maskStageMode;     /* Mask staging option i.e. which ctrl bits are staged */
	UINT8  maskStagePattern;  /* Mask stage pattern option e.g. 32-stage, 160-stage etc. */
	#endif
	UINT16 unused2;

	UINT8  unused3;
	UINT8  currentChip;       /* Current chip being scanned if FE-by-FE mode */
	UINT16 currentMaskStage;  /* Current mask stage */    
	UINT16 currentBin[2];     /* Current scanning point */
	UINT32 currentDelay;           

} ScanGeneral; /* General scan parameters */

typedef struct {
	UINT32 *base;    /* Base of histograms in SDSP memory; 0xffffffff=> default */
	UINT8  opt[4];   /* histogramming options: [0] TRUE => do occupancy histo.
	                                           [1] TRUE => do time-slice spectrum
	                                           [2] bit 0 set: do TOT spectrum,
	                                               bit 3 set: do TOT mean calculation
	                                           [3] will be filled in by MDSP (indicates
	                                               #evt./L1A for SDSP). */
	UINT8 extSetup;  /* Indicates whether MDSP will set up the router & SDSPs or not */
	UINT8 errors;    /* Toggles histogramming of errors  (error task) */
	#if   defined(SCT_ROD)
		UINT8 unused[2];		
	#elif defined(PIXEL_ROD)
		UINT8 errorsMCC; /* Toggles histogramming of MCC errors (error task) */
		UINT8 unused;
	#endif

} ScanRODSetup; /* Histogramming options */


typedef struct {
	UINT8 definedGroups;    /* bitfield indicating which groups are included in scan */
	UINT8 nDspPairs;        /* The # of SDSP pairs. A DSP pair use complementary SP
	                            and receive their trigger streams simultaneously; used
	                            for interference/ cross-rtalk checks. */
	UINT8 slvBits;          /* Indicates which SDSPs participate in scan. */
	UINT8 unused0;

	UINT8 dspPair[2][2];   /* The DSP pairs. */
	#if   defined(SCT_ROD)
		UINT8 groupRangeMap[2]; /* SCT: Bitfield indicating which mapping groups use. */
	#elif defined(PIXEL_ROD)
		UINT8 unused[2];
	#endif
	UINT8 groupSPMap[2];    /* Bitfield indicating which serial port groups use. */
	UINT8 groupDSPMap[4];   /* Bitfields indicating to which SDSP groups send events.*/

} ScanDspDistribution; /* Histogramming options */

typedef struct {
	struct  CmdList triggerSequence[2];
	#if   defined(SCT_ROD)
		UINT32 incData[2]; /* SCT: amount to increment command data by between bins. */
		UINT8  incCmd[2];  /* SCT: index of command (typ. delay cmd) to increment */
		UINT8  port;           /* Serial ports to use: 0, 1 or both (2) */
		UINT8  unused;
	#elif defined(PIXEL_ROD)
		UINT8 calcFromStruct; /* Pixel: indicates that the serial stream (cal. pulse/
		                         delay/L1A) should be calculated from the scan control
		                         structure. */
		UINT8 port;           /* Serial ports to use: 0, 1 or both (2) */
		UINT8 unused[2];
	#endif
} ScanSerialData;


#if   defined(SCT_ROD)
	typedef struct {
		ScanGeneral          general;
		SCTScanReset         reset;
		ScanRODSetup         rodSetup;
		ScanDspDistribution  dspDist;
		ScanSerialData       serial;
	
	} ScanControl;

#elif defined(PIXEL_ROD)
	typedef struct {
		ScanGeneral          general;
		PixelScanStrobe      strobe;
		PixelScanTrigger     trigger;
		PixelScanReset       reset;
		PixelScanFE          fe;
		PixelScanMCC         mcc;
		PixelScanFitting     fitting;
		ScanRODSetup         rodSetup;
		ScanDspDistribution  dspDist;
		ScanSerialData       serial;
	
	} ScanControl;
#endif

#endif /* Multiple inclusion protection */
