/*************************************************************************************
 * memoryPartitions.h
 *
 *  synopsis: Defines the memory partitions in the DSP memory space.
 *            Declares routine for initializing memory.
 *            Should be included in the DSP and the VME host code.
 *
 *   In Project->Options->Compiler it is POSSIBLE to define
 *    TI_EVM: EVM DSP has smaller SDRAMs than the ROD DSPs. This option partitions
 *            memory accordingly.
 *    MAP_0: probably never used, ROD DSPs are wired for MAP_1, see peripherals 
 *           manual 10-5
 *    SIM:   If defined, handles some aspects of the DSP's environment (for SDSPs,
 *           this is the MDSP's responses to their communication register settings;
 *           for the MDSP this is the host's & SDSPs' responses. 
 *
 *  Damon Fasching, UW Madison                            fasching@wisconsin.cern.ch
 *  Douglas Ferguson, UW Madison   (510) 486-5230         dpferguson@lbl.gov
 *
 *  modifications/bugs    
 *   - Updated to reflect changes in the project organization. When BIOS was
 *     introduced, the program memory sections must be defined inside the config
 *     database; the MEMORY section of the linker command file is no longer used &
 *     is taken out. External code & data are combined and put the XPROG memory
 *     section (FARDATA gone). The default positions and sizes of the 4 text, and
 *     the primitive & reply buffers, have changed. 6701s now have 32M of external
 *     memory; 6201 still has 16M.                                   26.04.02 dpsf
 *
 *   - Inter-DSP communication buffers put in SDRAM & communication registers moved
 *     to the beginning of internal memory (IDREG segment in the configuration
 *     database file) The 2 handshake registers have been left in internal memory
 *     with the other registers. A new section of memory, IDATA, has been created
 *     for important program variables; they should now remain relatively fixed in
 *     one spot instead of floating about in the program IDRAM while code is added
 *     & modified, as they did before. This is intended to be used with the
 *     DATA_SECTION pragma (see Optimizing C Compiler User's Guide, 7.6.4). Use of
 *     the pragma makes variables inaccessible to routines in other files, so
 *     use with care. (They can still be accessed by calling a function to return a
 *     pointer to them).                                                05.06.02 dpsf
 *
 *   - Added EXPERT, HCMD, HSTAT, TRAPREQ & TRAPSTAT registers for general purpose
 *     use, histogramming, and event trapping respectively. New defined locations in
 *     memory were added for the (suggested) base in memory of histograms & the
 *     trapping buffer as well.                                         20.06.02 dpsf
 *
 *   - New memory section MODULE_DATA was added for storage of new module
 *     configuration structures on the master.                          03.07.02 dpsf
 * 
 *  modifications/bugs    
 *   - Master & slave DSPs now have different size primitive & reply buffers. The
 *     compilation constants HOST_(PRM/REP)_BFR_(BASE/SZ) in memoryPartitions.h have
 *     been replaced with different sets for the two DSP types.        10.12.02 dpsf
 *************************************************************************************
 * NOTE: The segment sizes defined in this file for use in positioning registers and
 *       other program items (which are not explicitly declared as variables) must
 *   agree with the corresponding memory segment length and base, set in the
 *   configuration database file (.cdb). **Otherwise overwrites are possible** The
 *   memory order of the segments defined there should be maintained as well. These
 *   segment sizes are:
 *
 *   IDREGS_SZ, IDATA_SZ, IDRAM_MEM_SZ, BURST_SZ, XPROG_SZ, and 
 *   (master only) MODULE_DATA_SZ
 *
 *   IDRAM_SZ is the total size of the DSP IDRAM.
 *   (IDRAM_MEM is used here for the memory section IDRAM in the .cdb file, which
 *    cannot be renamed.)
 ************************************************************************************/
#ifndef MEMORY_PARTITIONS
#define MEMORY_PARTITIONS

#include "processor.h"
/*
 * Each DSP has 0x10000 bytes of internal program memory, IPRAM, starting at 0x0
 *              0x10000 bytes of internal data memory, IDRAM, starting at 0x80000000
 *              0x1000000 bytes SDRAM starting at 0x2000000 or 0x3000000
 *
 * Some of this is used for buffers and communication mailboxes whose addresses and
 * sizes are defined below.
 *
 * Memory sections are defined in the configuration database file so that it is
 * available for the linker to allocate code, data, stack, heap, etc... The BIOS is
 * set up so that it uses an external user linker command file (.cmd) for the
 * placement of non-BIOS related memory sections; this file then calls the BIOS .cmd
 * file at its finish. 
 *
 *                      table of available memory spaces
 *
 * mem space    base address             what's there for each device
 *
 *               MAP 0      MAP 1        EVM       master DSP      slave DSPs
 * EMIF CE0  0x00000000  0x00400000  256KB SBSRAM   RRI FPGA   external registers
 * EMIF CE1  0x01000000  0x01400000                 boot ROM         router
 * EMIF CE2  0x02000000  0x02000000   4MB SDRAM    16MB SDRAM      16MB SDRAM
 * EMIF CE3  0x03000000  0x03000000   4MB SDRAM                    16MB SDRAM
 *  IPRAM    0x01400000  0x00000000          64KB internal program memory
 *  IDRAM    0x80000000  0x80000000          64KB intermal data memory   
 *
 *************************************************************************************
 *                   sizes of the available memories.
 ************************************************************************************/
#define IPRAM_SZ       (0x10000)   /* 64KB of internal program memory */
#define IDRAM_SZ       (0x10000)   /* 64KB of internal data memory */
#if (defined(I_AM_SLAVE_DSP)&&defined(REV_E))
	#define SRAM_SZ_NO_CACHE        (0x40000)
	#define SRAM_SZ_16K_CACHE       (0x3c000)
	#define SRAM_SZ_32K_CACHE       (0x38000)
	#define SRAM_SZ_48K_CACHE       (0x34000)
	#define SRAM_SZ_64K_CACHE       (0x30000)
#endif

#ifdef TI_EVM
	#define SDRAM0_SZ  (0x400000)  /* 4MB of SDRAM available at CE2 on the EVM */
	#define SDRAM1_SZ  (0x400000)  /* 4MB of SDRAM available at CE3 on the EVM */

#else                             
	#if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
		#define SDRAM0_SZ  (0x800000)  /*  16 MB of SDRAM available on the ROD */
		#define SDRAM1_SZ  (0x800000)  /*  divided into two halves both on CE2 */
	#elif (defined(I_AM_SLAVE_DSP))
		#if (defined(REV_B)||defined(REV_C))
			#define SDRAM0_SZ  (0x01000000)
			#define SDRAM1_SZ  (0x01000000)
		#elif defined(REV_E)
			#define SDRAM0_SZ  (0x08000000)
			#define SDRAM1_SZ  (0x08000000)
		#endif
	#endif
#endif

/************************************************************************************
 *                  base addresses the available memories
 ************************************************************************************/
/* internal program, internal data and 4 EMIF chip enables for off chip memories */
#if (   (defined(I_AM_MASTER_DSP)) || (defined(I_AM_NT_HOST)) || (defined(I_AM_LINUX_HOST)) \
     || (defined(I_AM_SLAVE_DSP)&&(defined(REV_B)||defined(REV_C)))  )
	#ifdef MAP_0
		#define CE0_BASE   (0x00000000)
		#define CE1_BASE   (0x01000000)
		#define IPRAM_BASE (0x01400000)
	#else
		#define CE0_BASE   (0x00400000)
		#define CE1_BASE   (0x01400000)
		#define IPRAM_BASE (0x00000000)
	#endif
	
	#define CE2_BASE   (0x02000000)
	#define CE3_BASE   (0x03000000)
	#define IDRAM_BASE (0x80000000)

#elif (defined(I_AM_SLAVE_DSP)&&defined(REV_E))
	#define CE0_BASE   (0x80000000)
	#define CE1_BASE   (0x90000000)
	#define SRAM_BASE  (0x00000000)
	#define IPRAM_BASE (0x00000000)
	
	#define CE2_BASE   (0xa0000000)
	#define CE3_BASE   (0xb0000000)
	#define IDRAM_BASE (0x00010000)
#endif


/* SDRAM: EVM has 2 4MB SDRAMs at CE2 and CE3; ROD has 16 MB SDRAM at CE2
   + another 16M at CE3 for 6701s */
#define SDRAM0_BASE  (CE2_BASE)
#ifdef TI_EVM
	#define SDRAM1_BASE  (CE3_BASE)
#else
	#if (defined(I_AM_MASTER_DSP)|| defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
		#define SDRAM1_BASE  (CE2_BASE) + (SDRAM0_SZ)
	#elif (defined(I_AM_SLAVE_DSP))
		#define SDRAM1_BASE  (CE3_BASE)
	#endif
#endif

/* The master DSP has a boot ROM at CE1; slaves each have a router FIFO */
#ifndef TI_EVM
	#if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
		#define BOOT_ROM_BASE   (CE1_BASE)
		#define BOOT_ROM_SZ     (0x80000)      /* boot ROM is 512K */
	#elif (defined(I_AM_SLAVE_DSP))
		#define ROUTER_FIFO_BASE   (CE1_BASE)
		#define ROUTER_FIFO_SZ     (0x1000)      /* 1K words */
    #endif
#endif


/************************************************************************************
 *                                IDRAM items
 ************************************************************************************
 * IDREGS_SZ bytes of IDRAM are reserved for communications registers.
 * IDATA_SZ  bytes are reserved for important variables for which it's desirable they
 *           be kept (for monitoring) in a fixed location. Note that this method has a
 *   minor drawback in that the variables become 'far' data objects, they cannot be
 *   declared 'external' and thus cannot be directly accessed by functions from files
 *   other than the one that they were declared in; they must be accessed through
 *   function calls to functions residing in the same file. See accessRegister.c and
 *   eventHandler.c for examples. 
 * IDRAM_MEM_SZ bytes are reserved for the linker to place general heap, stack and 
 *              'near' data objects (arrays and structures which are not declared
 *   'far'). This memory corresponds to the IDRAM memory segment in the .cdb file. In
 *   the slaves it also holds initialization data for variable values and constants.
 *
 * BURST_SZ bytes of IDRAM are reserved for the program to store data for immediate
 *          use. On the slaves it might be a temporary store for data retrieved from
 *   the router FPGA via DMA.  This data might be quickly checked for errors and dumped
 *   if none are found or analysed if errors are found.  Or the data might be processed
 *   and added to histograms being accumulated in the SDRAM.  On the master it might be
 *   a temporary store for data being transmitted to the front end electronics.
 *
 * **** THESE SIZES MUST BE THE SAME AS THE CORRESPONDING LENGTH PARAMETER OF MEMORY
 *      SECTION MANAGER IN THE CONFIGURATION DATABASE FILE ****
 *
 * NOTE: If the memory section sizes are mistaken, the error will sometimes be caught
 *       by the compiler at compile time. Another check is done at run time inside
 *   checkMemBounds.c. Most larger data objects like the primitive buffers, text
 *   buffers, and external burst buffers do not have variables and memory segments
 *   declared for them; for these the space is 'virtually' allocated here and checked
 *   in chekMemBounds.c
 *
 */
#define IDREGS_SZ      (0x0060)
#define IDATA_SZ       (0x0800)
#define IDRAM_MEM_SZ   (0x77a0)
#define BURST_SZ       (0x8000) 

#define IDRAM_ALLOCATED ((IDREGS_SZ)+(IDATA_SZ)+(IDRAM_MEM_SZ)+(BURST_SZ))

#define IDREGS_BASE     (IDRAM_BASE)
#define IDATA_BASE      ((IDREGS_BASE) + (IDREGS_SZ))
#define IDRAM_MEM_BASE  ((IDATA_BASE)  + (IDATA_SZ))  /* IDRAM_BASE is kept for the
                                                         physical memory base   */
#define BURST_BFR_BASE   ((IDRAM_MEM_BASE) + (IDRAM_MEM_SZ))

/* communication registers: used for the handshake for host to DSP primitive lists */
/* 0x00 */
#define STATUS_REG_0     (IDRAM_BASE)
#define STATUS_REG_1     ((STATUS_REG_0)   + sizeof(UINT32))
#define STATUS_REG_2     ((STATUS_REG_1)   + sizeof(UINT32))
#define COMMAND_REG_0    ((STATUS_REG_2)   + sizeof(UINT32))

/* 0x10 */
#define DIAGNOSTIC_REG   ((COMMAND_REG_0)  + sizeof(UINT32))
#define TRAPSTAT_REG_0   ((DIAGNOSTIC_REG) + sizeof(UINT32))
#define TRAPSTAT_REG_1   ((TRAPSTAT_REG_0) + sizeof(UINT32))
#define LOOP_REG         ((TRAPSTAT_REG_1) + sizeof(UINT32))

/* 0x20 */
#define HCMD_STAT_REG_0  ((LOOP_REG)        + sizeof(UINT32))
#define HCMD_STAT_REG_1  ((HCMD_STAT_REG_0) + sizeof(UINT32))
#define HSTAT_REG_0      ((HCMD_STAT_REG_1) + sizeof(UINT32))
#define HSTAT_REG_1      ((HSTAT_REG_0)     + sizeof(UINT32))

/* 0x30 */
#define RESERVED_REG_0   ((HSTAT_REG_1)    + sizeof(UINT32))
#define RESERVED_REG_1   ((RESERVED_REG_0) + sizeof(UINT32))
#define RESERVED_REG_2   ((RESERVED_REG_1) + sizeof(UINT32))
#define RESERVED_REG_3   ((RESERVED_REG_2) + sizeof(UINT32))

/* 0x40 */
#define RESERVED_REG_4   ((RESERVED_REG_3) + sizeof(UINT32))
#define RESERVED_REG_5   ((RESERVED_REG_4) + sizeof(UINT32))
#define MEMORY_MAP_REG   ((RESERVED_REG_5) + sizeof(UINT32))
#define TASK_STATE_REG   ((MEMORY_MAP_REG) + sizeof(UINT32))

/* 0x50 */
#define HCMD_REG         ((TASK_STATE_REG) + sizeof(UINT32))
#define TRAP_CMD_STAT    ((HCMD_REG)       + sizeof(UINT32))
  
/* inter-dsp communication registers: used for the handshake between master and
 * slave DSPs when one sends a list for the other to process.  These registers
 * reside in the IDRAM of each slave DSP.  The addresses are known to both the
 * master and the slaves.  Note that the write register for the slave is the read
 * register for the master and vice versa. 
 *
 * These are NOT used in COMPACT mode, but they are still defined for the
 * alignment of the text, primitive and reply buffers. */
#if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
	#define INTR_DSP_HSHK_WR  ((TRAP_CMD_STAT)    + sizeof(UINT32))
	#define INTR_DSP_HSHK_RD  ((INTR_DSP_HSHK_WR) + sizeof(UINT32))
#elif defined(I_AM_SLAVE_DSP)
	#define INTR_DSP_HSHK_RD  ((TRAP_CMD_STAT)    + sizeof(UINT32))
	#define INTR_DSP_HSHK_WR  ((INTR_DSP_HSHK_RD) + sizeof(UINT32))
#endif

/************************************************************************************
 *                                SDRAM0 items
 ************************************************************************************
 *** THIS MUST BE THE SAME AS THE LENGTH PARAMETER OF MEMORY SECTION XPROG IN
 *   THE LINKER COMMAND FILE *****/
#define XPROG_SZ   (0x40000)
  /* The ERR, INFO, DIAG and XFER buffers are the text buffers for sending messages
   * upstream to the host processor.  Slave text buffers are communicated to the host
   * via the master XFER buffer.  The XFER buffer is not used on the slaves.
   * The HOST_PRM and HOST_REP buffers are for primitive lists from the host and the
   * associated reply data.  Host to slave primitive and associated reply data are
   * transferred via the master DSP. The inter-DSP transfer buffers are small buffers
   * used for sending simple primitive lists from DSP to DSP (master to slave and
   * vice-versa). Note that in COMPACT mode the buffers are actually in IDRAM. */

#ifdef COMPACT
  #define TXT_BFR_SZ       (0x800)
  #define INTR_DSP_PRM_BFR_SZ  (0x800)
  #define INTR_DSP_REP_BFR_SZ  (0x800)

  #if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
    #define MDSP_PRM_BFR_SZ  (0x2000)
    #define MDSP_REP_BFR_SZ  (0x2000)
  #endif
  #define SDSP_PRM_BFR_SZ  (0x2000)
  #define SDSP_REP_BFR_SZ  (0x2000)

  #define ERR_BFR_BASE       (BURST_BFR_BASE)
  #define INFO_BFR_BASE      ((ERR_BFR_BASE)  + (TXT_BFR_SZ))
  #define DIAG_BFR_BASE      ((INFO_BFR_BASE) + (TXT_BFR_SZ))
  #define XFER_BFR_BASE      ((DIAG_BFR_BASE) + (TXT_BFR_SZ))

  #define INTR_DSP_PRM_BFR_BASE_SND ((XFER_BFR_BASE) +(TXT_BFR_SZ))
  #define INTR_DSP_REP_BFR_BASE_SND ((INTR_DSP_PRM_BFR_BASE_SND)+(INTR_DSP_PRM_BFR_SZ))
  #define INTR_DSP_PRM_BFR_BASE_PRC ((INTR_DSP_REP_BFR_BASE_SND)+(INTR_DSP_REP_BFR_SZ))
  #define INTR_DSP_REP_BFR_BASE_PRC ((INTR_DSP_PRM_BFR_BASE_PRC)+(INTR_DSP_PRM_BFR_SZ))

  #if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
    #define MDSP_PRM_BFR_BASE  ((INTR_DSP_REP_BFR_BASE_PRC) + (INTR_DSP_PRM_BFR_SZ))
    #define MDSP_REP_BFR_BASE  ((MDSP_PRM_BFR_BASE) + (MDSP_PRM_BFR_SZ)) 
  #endif
  #define SDSP_PRM_BFR_BASE  ((INTR_DSP_REP_BFR_BASE_PRC) + (INTR_DSP_PRM_BFR_SZ))
  #define SDSP_REP_BFR_BASE  ((SDSP_PRM_BFR_BASE) + (SDSP_PRM_BFR_SZ)) 

  #define XBURST_BFR_BASE  ((SDRAM0_BASE) + (XPROG_SZ))
  #define XBURST_BFR_SZ    (0x38000) 

  #define BURST_BFR_RESERVED_BASE ((XBURST_BFR_BASE) +(XBURST_BFR_SZ))
  #define BURST_BFR_RESERVED_SZ   (0x8000)

  #define HISTOGRAM_DEFAULT_BASE   ((BURST_BFR_RESERVED_BASE) +(BURST_BFR_RESERVED_SZ))
  #define SDRAM0_ALLOCATED ((HISTOGRAM_DEFAULT_BASE) -(SDRAM0_BASE))

#else

  #define TXT_BFR_SZ       (0x8000)
  #define INTR_DSP_PRM_BFR_SZ  (0x800)
  #define INTR_DSP_REP_BFR_SZ  (0x800)

  /* WARNING: In non-COMPACT compilations, the master & slave DSPs now have
              different size primitive & reply buffers. To accomodate this those
       buffers have different compilation constants defined. it also means that THE
       DSP_(PRM/REP)_BFR_BASE DEFINITIONS MUST BE THE LAST DEFINITIONS WHICH ARE
       COMMON TO BOTH DSP TYPES. THE TEXT BUFFERS AND INTER-DSP PRIM BUFFERS MUST
       BE DEFINED BEFORE THEM. */
  #if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
    #define MDSP_PRM_BFR_SZ  (0x50000)
    #define MDSP_REP_BFR_SZ  (0x50000)
  #endif
  #define SDSP_PRM_BFR_SZ  (0x20000)
  #define SDSP_REP_BFR_SZ  (0x20000)

#if (   (defined(I_AM_MASTER_DSP)) || (defined(I_AM_NT_HOST)) || (defined(I_AM_LINUX_HOST)) \
     || (defined(I_AM_SLAVE_DSP)&&(defined(REV_B)||defined(REV_C)))  )
  #define ERR_BFR_BASE       ((SDRAM0_BASE) + (XPROG_SZ))
  #define INFO_BFR_BASE      ((ERR_BFR_BASE)  + (TXT_BFR_SZ))
  #define DIAG_BFR_BASE      ((INFO_BFR_BASE) + (TXT_BFR_SZ))
  #define XFER_BFR_BASE      ((DIAG_BFR_BASE) + (TXT_BFR_SZ))

#elif (defined(I_AM_SLAVE_DSP)&&defined(REV_E))
  #define ERR_BFR_BASE1       0x20000
  #define INFO_BFR_BASE1      0x28000

  #define ERR_BFR_BASE       ((SDRAM0_BASE) + (XPROG_SZ))
  #define INFO_BFR_BASE      ((ERR_BFR_BASE)  + (TXT_BFR_SZ))
  #define DIAG_BFR_BASE      ((INFO_BFR_BASE) + (TXT_BFR_SZ))
  #define XFER_BFR_BASE      ((DIAG_BFR_BASE) + (TXT_BFR_SZ))
#endif


  #define INTR_DSP_PRM_BFR_BASE_SND ((XFER_BFR_BASE) +(TXT_BFR_SZ))
  #define INTR_DSP_REP_BFR_BASE_SND ((INTR_DSP_PRM_BFR_BASE_SND)+(INTR_DSP_PRM_BFR_SZ))
  #define INTR_DSP_PRM_BFR_BASE_PRC ((INTR_DSP_REP_BFR_BASE_SND)+(INTR_DSP_REP_BFR_SZ))
  #define INTR_DSP_REP_BFR_BASE_PRC ((INTR_DSP_PRM_BFR_BASE_PRC)+(INTR_DSP_PRM_BFR_SZ))

  #if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
    #define MDSP_PRM_BFR_BASE  ((INTR_DSP_REP_BFR_BASE_PRC) + (INTR_DSP_PRM_BFR_SZ))
    #define MDSP_REP_BFR_BASE  ((MDSP_PRM_BFR_BASE) + (MDSP_PRM_BFR_SZ)) 
    #define CMD_BUFFER_BASE    ((MDSP_REP_BFR_BASE) + (MDSP_REP_BFR_SZ))
    #define CMD_BUFFER_SZ      0x8000
    #define SDRAM0_FREE_BASE   ((CMD_BUFFER_BASE) +(CMD_BUFFER_SZ))

  #endif
  #define SDSP_PRM_BFR_BASE  ((INTR_DSP_REP_BFR_BASE_PRC) + (INTR_DSP_PRM_BFR_SZ))
  #define SDSP_REP_BFR_BASE  ((SDSP_PRM_BFR_BASE) + (SDSP_PRM_BFR_SZ)) 

  #if (defined(I_AM_SLAVE_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
    #define XBURST_BFR_BASE           ((SDSP_REP_BFR_BASE)+(SDSP_REP_BFR_SZ))
    #define XBURST_BFR_SZ             (0x38000) 
	
    #define BURST_BFR_RESERVED_BASE ((XBURST_BFR_BASE) +(XBURST_BFR_SZ))
    #define BURST_BFR_RESERVED_SZ   (0x8000)
	
    #define HISTOGRAM_DEFAULT_BASE  ((BURST_BFR_RESERVED_BASE) +(BURST_BFR_RESERVED_SZ))

  #endif

  #if   defined(I_AM_MASTER_DSP)
    #define SDRAM0_ALLOCATED ((SDRAM0_FREE_BASE) -(SDRAM0_BASE))
  #elif defined(I_AM_SLAVE_DSP)
    #define SDRAM0_ALLOCATED ((HISTOGRAM_DEFAULT_BASE) -(SDRAM0_BASE))
  #endif

  #define MODULE_DATA_BASE    0x02400000
  #if   defined(SCT_ROD)
    #define MODULE_DATA_SZ      0x00080000      /* .5 MB */
  #elif defined(PIXEL_ROD)
    #define MODULE_DATA_SZ      0x00b00000      /* 11 MB */
  #endif

#endif

/* default position for trapped events is in high memory: whereas the burst buffers
   are always needed for trapping events, this is potentially reserved memory for
   gathering trapped events; it is intended to be a suggestion when running the event
   trapping task. */
#if  (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))
	#define SLV_SDRAM1_BASE          (0x03000000)
	#define SLV_SDRAM1_SZ            (0x01000000)   /* needed by VME host */
	#define EVENT_TRAP_DEFAULT_SZ    (0x20000)
	#define EVENT_TRAP_DEFAULT_BASE  ((SLV_SDRAM1_BASE) +(SLV_SDRAM1_SZ) \
	                                  -(EVENT_TRAP_DEFAULT_SZ))
#else
	#define EVENT_TRAP_DEFAULT_SZ    (0x20000)
	#define EVENT_TRAP_DEFAULT_BASE  ((SDRAM1_BASE) +(SDRAM1_SZ) \
	                                  -(EVENT_TRAP_DEFAULT_SZ))
#endif

/* SDRAM1: Free space for application defined data on SCT RODs; for Pixel RODs
           the module database takes up the last 4 MB of SDRAM0 & the first 7
           MB of SDRAM1 (see above) & the upper 1 MB is free space. */
#if (defined(I_AM_MASTER_DSP) || defined(I_AM_NT_HOST) || defined(I_AM_LINUX_HOST))

	#define MEMORY_LIMIT  0x03000000
	/* if running on the TI evaluation module, the 6701 DSP communication registers
	   must be offset, and the RRIF must be faked. A small amount of slave memory is
	   faked as well. These are put right before the end of memory. */
	#define EVM_SLV_REG_BASE(slv)    (MEMORY_LIMIT - 0x18200 + 0x80*(slv))
	#define EVM_SLV_MEM_BASE(slv)    (MEMORY_LIMIT - 0x18000 + 0x2000*(slv))
	#define EVM_RRIF_REG_BASE        ((MEMORY_LIMIT) - 0x10000)

	#define SDRAM1_FREE_BASE   (SDRAM1_BASE)
#else 
	#define SDRAM1_FREE_BASE   (SDRAM1_BASE)
#endif

#define SDRAM1_ALLOCATED   0

#if defined(TI_EVM)  /* need a histogram base for SDRAM1 */
	#define HISTOGRAM_BASE1   ((SDRAM1_BASE) +(0x40000))
#endif

/************************************************************************************
 *    RRIF memory mapped items - CE0 on ROD; EVM_RRIF_REG_BASE on TI_EVM/SIM (no RRIF)
 ************************************************************************************/
#if defined(I_AM_MASTER_DSP) 
	/* base address for memory mapped registers and FIFOs */

	#if (defined(TI_EVM) || defined(SIM))
		#define REG_BASE (EVM_RRIF_REG_BASE)
	#else
		#define REG_BASE    (CE0_BASE)
	#endif
	
	/* offsets into each device  These are RRIF-centric offsets, i.e. from the DSP
	 * point of view they need to be left shifted 2 bits.  This is done where needed
	 * in the files containing the access routines. */
	#define FMT_BASE  (0x0)
	#define EFB_BASE  (0x800)
	#define RTR_BASE  (0x900)
	#define RRIF_BASE (0x1000)
	#define FIFO_BASE (0x1800)
	#define BOC_BASE  (0x2000)
	#define SLV_BASE  (0xE0000)
	
	#define LED_OFFSET 0x100
	#define HEARTBEAT_ADR ((UINT32 *)(((RRIF_BASE + LED_OFFSET) << 2) + REG_BASE))
	
#elif defined(I_AM_SLAVE_DSP)
	#define ROUTER_FIFO_ADDR       (CE1_BASE)

#endif

#endif
