/*
 * ISInterface.java
 *
 * Created on 06 November 2003, 17:16
 */

package GuiComponents.System;

import ipc.*;
import is.*;

/**
 * Provides useful methods for accessing IS.
 * Clients should use this rather than going directly so that the correct 
 * client-side IS code is used.
 *
 * This class and SystemInterface mirror some of the methods from SctNames in the C++ code
 *
 * @author Matthew Palmer
 */
public class ISInterface {
    
    /**
     * Get the singleton for accessing IS
     */ 
    public static ISInterface getInstance() {
        return instance;
    }
        
    /**
     * Re-obtain all connections to external processes.
     */
    public void refresh() {
	r = null;
    }
    
    /**
     * Returns a list of all ISServers currently available
     *Except that ServerIterator doesn't work, so just return Dave's server
     */
    public String[] getISServers() {
	try {
	    is.ServerIterator sit = new is.ServerIterator(si.getPartition());
	    String[] servers = new String[sit.entries()];
	    for (int i=0; i<sit.entries(); ++i) {
		servers[i] = sit.nextServerName();
	    }
	    return servers;
	} catch (NullPointerException npe) { //Caused by partition not available
	    return new String[0];
	}
    }
    
    /**
     * Returns a list of all objects on a server
     * @param String server: the name of the IS Server
     */
    public String[] getISObjects(String server) {
	try {
	    InfoList l = new InfoList(si.getPartition(), server);
	    String[] objects = new String[l.getSize()];
	    AnyInfo info;
	    for (int i=0; i<l.getSize(); i++) {
		info = l.getInfo(i);
		objects[i] = new String(info.getName()); 
	    }
	    return objects;
	} catch (NullPointerException npe) { //Caused by partition not available
	    return new String[0];
	}
    }
    
    public int getNumberOfISObjects(String server, String pattern) {
        if (si.getPartition()==null) return 0;
        InfoList l = new InfoList(si.getPartition(), server, pattern);
        return l.getSize();
    }
    
    public String[] getISObjects(String server, String pattern) {
        if (si.getPartition()==null) return new String[0];
        InfoList l = new InfoList(si.getPartition(), server, pattern);
        String[] objects = new String[l.getSize()];
        AnyInfo info;
        for (int i=0; i<l.getSize(); i++) {
            info = l.getInfo(i);
            objects[i] = new String(info.getName()); 
        }
        return objects;
    }
    
    public Repository getRepository() {
	if (r == null) {
	    try {
		r = new SctRepository(si.getPartition());
	    } catch (NullPointerException npe) {
		//Do nothing - this must have been caused by not being able to get the partition, which we've already logged
	    }
	}
        return r;
    }
    

    /**
     * @{ @name Convenience methods
     * All these methods return a String describing the outcome rather than
     * throwing errors if something bad happens.
     */      
    public String subscribe(String server, String regex, boolean isRegex, InfoListener l) {        
        try {
            getRepository().subscribe(server, regex, isRegex, l);
        } catch (RepositoryNotFoundException rnfe) {
            return "Internal error while subscribing to server " +server +" for " + regex + " : " + rnfe; 
        } catch (InvalidExpressionException iee) {
            return "Internal error while subscribing to server " +server +" for " + regex + " : " + iee; 
	} catch (NullPointerException npe) {
	    return "Couldn't get repository - see logs";
	}
        
        return "Subscribe to " + server + " successful";
    }

    public String subscribe(String server, InfoListener l) {
           return subscribe(server, ".*", true, l);
    }
    
    public String unsubscribe(String server) {
        return unsubscribe(server,".*");
    }

    public String unsubscribe(String server, String regex) {
        try {
            getRepository().unsubscribe(server, regex,  true);
        } catch (RepositoryNotFoundException rnfe) {
            return "Oops, internal error while unsubscribing: " + rnfe;
        } catch (SubscriptionNotFoundException snfe) {
            return "Ooop, internal error - subscription not found: " + snfe;
        } catch (NullPointerException npe) {
	    return "Couldn't get repository - see logs";
	}
        
        return "Unsubscribe from " + server + " for " + regex + " successful";
    }
    
    public String remove(String object) {
        try {
            getRepository().remove(object);
        } catch (RepositoryNotFoundException rnfe) {
            return "Oops, internal error while unsubscribing: " + rnfe;
        } catch (InfoNotFoundException infe) {
            return "Object not found in repository: " + infe;
        } catch (NullPointerException npe) {
	    return "Couldn't get repository - see logs";
	}
        
        return "Object" + object + " successfully removed";
    }
    
    /** Creates a new instance of ISInterface */
    private ISInterface() {        
        si = SystemInterface.getInstance();
        refresh();
    }
    
    private static ISInterface instance = new ISInterface();
    private Repository r;
    private SystemInterface si;
}
