#ifndef FITSTRATEGY_H
#define FITSTRATEGY_H
#include "Sct/LogicErrors.h"
#include "Sct/MathsError.h"
#include <boost/noncopyable.hpp>
#include <map>
#include <string>

using namespace Sct;
class TH1;
class TF1;
class TGraph;
class TGraphErrors;

namespace SctFitter{
    /**
       Motivated by the need to fitting rapidly, abstracted fit strategy allows different fit algorithms to be used. 
       Allows easy switch from say ROOT fitting to a custom fitter.
       Concrete strategies should add themselves to the map using the 
       addToMap function, at static initialization.
       It is assumed that: 
       <BR>o The histogram errors have been set
       <BR>o The function limits have been set
       <BR>o The histogram is truly a TH1 (not a silly ROOT TH2 which inherits from TH1!)
       @author Alan Barr
       @date 17/5/03
     */
    class FitStrategy : boost::noncopyable {
    public:
	/**
	   Fit Root TH1 using the current options.
	   @param hist input histogram to be fitted
	   @param fit output
	*/
	virtual void fitTH1(const TH1& hist, TF1& fit) const throw(Sct::LogicError, Sct::MathsError)=0;
	/**
	   Fit Root TGraph using the current options.
	   @param hist input histogram to be fitted
	   @param fit output
	*/
	virtual void fitTGraph(const TGraph& graph, TF1& fit) const throw(Sct::LogicError, Sct::MathsError)=0;
	/**
	   Fit Root TGraph using the current options.
	   @param hist input histogram to be fitted
	   @param fit output
	*/
	virtual void fitTGraphErrors(const TGraphErrors& graph, TF1& fit) const throw(Sct::LogicError, Sct::MathsError)=0;

	// set the options string
	void setOptions(string opt) throw();
	// get the options string
	const string& getOptions() const throw();
	// concrete strategies must impliment this function i.e. give a name.
	virtual const string& getName() const throw() =0 ;
    protected:
	/// Clients can make a FitStrategy Object.
	FitStrategy(string opt=string("")) throw();
	/// virtual destructor.
	virtual ~FitStrategy();
    private:
	/// a string containing fit options for this strategy
	string options;
    };

    /**
       Factory allows dynamic choice of FitStrategies which can be asked for by name.
     */
    class FitStrategyFactory{
    public:
	/// @return a reference to the singleton
	static FitStrategyFactory& instance() throw();
	
	/// Get a FitStrategy by name
	FitStrategy* getStrategy(string name) throw(LogicError);
	/** 
	    Concrete strategies should add themselves to the map, using the
	    same name as they give in getName();
	*/ 
	bool addToMap(const string& className, FitStrategy& s) throw();
	/// a map of the sub-classes and their names
	std::map <string, FitStrategy*> strategyMap;
    private:
	~FitStrategyFactory() throw() {} ///< singleton
	static FitStrategyFactory* bertha; ///< pointer to the singleton.
    };

} // end of namespace SctFitter;
#endif // #ifndef FITSTRATEGY_H
