#ifndef FITALGORITHM_H
#define FITALGORITHM_H

#include "FitterMode.h"
#include "Sct/Exception.h"
#include "Sct/MathsError.h"
#include "SctData/ModuleElement.h"
#include "SctData/DefectList.h"
#include <memory>
#include <boost/utility.hpp>
#include <boost/thread.hpp>

//Forward declarations
class TH1;
namespace SctData {
class FitObject;    
class FitScanResult;
class RawScanResult;
}

using SctData::FitScanResult;
using SctData::RawScanResult;
using SctData::ModuleElement;
using SctData::DefectList;
using SctData::FitObject;	
using Sct::LogicError;
using Sct::MathsError;
using std::auto_ptr;

namespace SctFitter {

/**
  This class is an abstract base class which describes how a fit is done.
  Specific sub-classes register themselves with the FitAlgorithmMap which is 
  used to determine which FitAlgorithm to use given a ConfigurationVariable.
  Each FitAlgorithm has its own mode that is used to determine whether to fit
  links, chip, channels (and indeed if that is even meaningful!)
  @author Matthew Palmer based on original code by Alan Barr (!)
  @date 7 June 2003
  */
class FitAlgorithm : boost::noncopyable {
public:
    virtual ~FitAlgorithm() throw() {}

    /**
      This is the entry point for the FittingService.
      The default implementation checks the mode then call doFit for each link/chip/channel
      as appropriate.  If any throw a ModuleDefect, they are added appropriately.  Finally, if
      the mode says, then createSummaryHistograms is called.
    */
    virtual auto_ptr<FitScanResult> doFit(const RawScanResult& raw) const throw(Sct::LogicError);

    const FitterMode& getMode() const throw();
    // not thread safe!    FitterMode& getMode() throw();
    void setMode(const FitterMode& mode) throw();

    ///{
    ///@name Utility functions
    
    /**
     *   Associated histogram utility routine. <P>
     *   If `last' is true then returns the value of
     *   x for which the bin content first is greater than `fraction' of its maximum
     *   <P> If `last' is false then returns the value of x for which  
     *   the cell content first is greater than `fraction' of its maximum.
     *  <P> Uses findBin function
     */
    static float findLevel(const TH1& h, const float fraction, const bool forward) throw(MathsError);

    /**
     *   Associated histogram utility routine. <P> 
     *   If `forward' is true then returns the bin
     *   for which the bin content first is greater than `fraction' of its maximum
     *   <P> If `forward' is false then returns bin for which
     *   the cell content is last greater than `fraction' of its maximum.
     *   <P>Throws an exception if the bin is not found.
     */
    static int findBin(const TH1& h, const float fraction, const bool forward) throw(MathsError);

    
    ///}
    
protected:
    FitAlgorithm() throw();

    /**
     *  Attempts to do an individual fit. Default implementation calls guessParameters
     *  and checkForDefects then uses the Fitter's FitStrategy to actually do the fit.
     */
    virtual void doFit(const TH1& hist, FitObject& fitObject,
		       const ModuleElement& element, 
		       DefectList& defects) const throw (LogicError);
    
    ///@{
    /**
      Returns a prototypical FitObject for this algorithm
      */
    virtual auto_ptr<FitObject> getPrototype() const throw() = 0;
    
    /**
     *  guess the initial fit parameters from histogram prior to doing a fit : put them in FitObject.
     */
    virtual void guessParameters(const TH1& hist, FitObject& fitOb) const throw (LogicError, MathsError) = 0;

    /**
     * checks the element for defects. Add them to the ModuleDefectList.
     */
    virtual void checkForDefects(const TH1& hist, const ModuleElement& element, 
				 DefectList& defects) const throw (LogicError) = 0;
    /**
     * checks fit for defects - add them to the defect list
     */
    virtual void checkForDefects(const FitObject& fo, const ModuleElement& element, 
				 DefectList& d) const =0;

    ///@}
    
    /**
    * Makes set of summary histograms from the fitObject results.
    * Default implementation does nothing for convenience.
    */
    virtual void createSummaryHistograms(FitScanResult& fits) const throw() = 0;
 protected:
  boost::recursive_mutex& getMutex() const {return m_access;} ///< access the mutex
private:
    mutable boost::recursive_mutex m_access;  ///< access to the mode can be locked for thread-safeness.    
    FitterMode mode;			///< The FitterMode used by this algorithm
};

}
#endif //#ifndef FITALGORITHM_H
