#ifndef TEST_H
#define TEST_H

#include <string>
#include <vector>
#include <list>

#include "IS/TestData.h"
#include "ipc/Sct_CalibrationController.hh"

using std::vector;
using std::list;
using std::string;

class Sct_SctApi_T_Scan;

namespace SctCalibrationController  {

/**
  This a record of each Test that is done.
  It contains references to the Scans that made it up
  and also the TestData object that is available in IS.
  */
class Test {    
public:
    /**
      Construct a Test.  Puts data into IS.
      @param data the TestData that defines this Test
      @param list the list of modules this Test is done on (for ease!)
      @throws InvalidArgumentError if data is not consistent.
      */
    Test(const TestData& data,  const list<string>& list);
    
    /**
      Removes the TestData object from IS.
      */
    ~Test();
    
    /**
      Adds a Scan to this Test.  You may only add the correct number of
      Scans.
      @param scan the Scan to add.
      @throws InvalidArgumentError if too many Scans are added
      */
    void addScan(Sct_SctApi_T_Scan* scan);
    
    /**
      Get access to the underlying TestData
      */
    const TestData& getData() const;


    /** 
      Get the module list
      @note Nothrow
      */
    list<string> getModuleList() const;
    
    /**
      Set the status of this Test - see TestData.
      A Test may only have the COMPLETED state if all the Scans have been added.
      Once a Test has entered the ABORTED or COMPLETED states, it cannot be changed.
      @throws InvalidArgumentError if not enough Scans have been added or if
      the status is currently ABORTED or COMPLETED and status is not ABORTED or COMPLETED 
      respectively.
      */
    void setStatus(TestData::status_E status);
    
    /**
      Returns a uniqueID for this Test - used for the name in IS
      */
    string getUniqueID() const;    
    
    /**
      Retrieve one of the constituent Scans
      @throws InvalidArgumentError if index is too large
      */
    Sct_SctApi_T_Scan* getScan(unsigned int index) const;
    
private:
    /**
      Publish or update the TestData object in IS.
      Won't throw - just reports errors
      */
    void publish();
    
    /**
      Remove the TestData object from IS.
      Won't throw - just reports errors
      */
    void withdraw();    
    
    TestData data;					///< Most of the underlying data
    typedef vector<Sct_SctApi_T_Scan*> ScanVector;	///< A vector of Scans!
    ScanVector scans;					///< The constituent Scans
};   
}

#endif //TEST_H
