#ifndef SEQUENCE_H
#define SEQUENCE_H

#include <vector>
#include <list>
#include <memory>
#include <boost/shared_ptr.hpp>
#include "IS/SequenceData.h"
#include "Test.h"

using std::vector;
using std::list;
using std::auto_ptr;
using boost::shared_ptr;

namespace SctCalibrationController {

/**
  This is a record of each Sequence that is done.
  It holds an underlying SequenceData that is put into IS and also
  references to the constituent Tests.
  */
class Sequence {
public:
    /**
      Force creation using shared_ptrs and new.  This is because each Sequence
      is automatically added to the SequenceMap.  Puts the SequenceData into IS.
      @param name the name of this Sequence
      @param runNumber the run number
      @param startScanNumber the scan number of the first Scan in this Sequence
      @note Problems can be caused if there is a significant gap between the creation of 
      the Sequence and the execution of the Scans as this would mean things get out of step/wrong!
      */
    static shared_ptr<Sequence> create(const string& name, unsigned long runNumber, unsigned long startScanNumber, const list<string>& list);
    
    /**
      Removes the SequenceData from IS.
      */
    virtual ~Sequence();
    
    /**
      Adds a Test to this Sequence.  Note transfer of ownership semantics.
      */
    void addTest(auto_ptr<Test> test);    
    
    /**
      Access the underlying SequenceData.
      */
    const SequenceData& getData() const;


    /**
      Get the list of modules in this Sequence
      @note Nothrow
      */
    list<string> getModuleList() const;
  
    /**
      Set the status of this Sequence - see SequenceData.      
      Once a Sequence has entered the ABORTED or COMPLETED states, it cannot be changed.
      @throws InvalidArgumentError if the status is currently ABORTED or COMPLETED 
      and status is not ABORTED or COMPLETED respectively.
      */
    void setStatus(SequenceData::status_E status);

    /**
      Get a unique ID for this Sequence - used by IS.
      */
    string getUniqueID() const;
    
    /**
      Get a Test. 
      @throws InvalidArgumentError if index is too large
      */
    shared_ptr<Test> Sequence::getTest(unsigned int index) const;
    
private:    
    /**
      Creates a Sequence - 
      @see create
      */
    Sequence(const string& name, unsigned long runNumber, unsigned long startScanNumber, const list<string>& list);
    SequenceData data;						///< The underlying data
    
    /**
      @implementation 
      For efficiency we allocate a slightly larger array in SequenceData than
      necessary for the Test names.  This is the real size of that array.
      */
    unsigned int realSize;					
    
    /**
      Put (create or update) the SequenceData into IS.
      Will not throw - just reports the problem.
      */
    void publish();
    
    /**
      Remove the SequenceData from IS
      Will not throw - just reports the problem.s      
      */
    void withdraw();    
    
    typedef vector<shared_ptr<Test> > TestVector;		///< A vector of Tests!
    TestVector tests;						///< The constituent Tests
};    
    
}

#endif //TESTSEQUENCEIMPL_H
