#include "Sequence.h"
#include "SequenceMap.h"
#include "Sct/IoExceptions.h"
#include "Sct/LogicErrors.h"
#include "Sct/SctNames.h"
#include <iostream>
#include <sstream>
#include <algorithm>
#include <boost/date_time/posix_time/posix_time.hpp>

using namespace std;
using namespace Sct;
using namespace boost::posix_time;

namespace SctCalibrationController {

Sequence::Sequence(const string& name, unsigned long runNumber, unsigned long startScanNumber, const list<string>& l) {
    
    data.sequenceName = name;
    data.runNumber = runNumber;
    data.startScanNumber = startScanNumber;
    data.nTests = 0;
    data.testNames_size = 0;
    realSize = 10;
    data.testNames = new string[realSize];
    string startTime = to_iso_string(second_clock::universal_time());
    data.startTime = startTime.substr(0, 15);
    
    data.modules = new string[l.size()];
    data.modules_size = l.size();
    unsigned int count = 0;
    for (list<string>::const_iterator i=l.begin(); i!=l.end(); ++i, ++count) {
        data.modules[count] = *i;
    }


    cout << "Create TestSequenceImpl: " << data.sequenceName << " Id: " << getUniqueID() << endl;
    
    publish();
}

Sequence::~Sequence() {
    cout << "Destroy TestSequenceImpl: " << data.sequenceName << " Id: " << getUniqueID() << endl;
    withdraw();
}

shared_ptr<Sequence> Sequence::create(const string& name, unsigned long runNumber, unsigned long startScanNumber, const list<string>& list) {
    shared_ptr<Sequence> seq(new Sequence(name, runNumber, startScanNumber, list));
    SequenceMap::instance().addSequence(seq);
    return seq;
}

list<string> Sequence::getModuleList() const{
    list<string> list;
    for (unsigned int i=0; i<data.modules_size; ++i) {
        list.push_back(data.modules[i]);
    }
    return list;
}

string Sequence::getUniqueID() const {
    ostringstream s;
    s << "SequenceData." << data.runNumber << "." << data.startScanNumber;
    return s.str();
}

void Sequence::addTest(auto_ptr<Test> test) {
    cout << test.get() << endl;
    tests.push_back(shared_ptr<Test>(test));
    cout << data.nTests << endl;
    ++data.nTests;
    cout << data.nTests << endl;
    
    if (data.nTests >= realSize) {
	realSize *= 2;
	string* testNames = new string[realSize];    
	for (unsigned int i=0; i<data.nTests; ++i) {
	    testNames[i] = tests[i]->getUniqueID();
	}
	delete [] data.testNames;
	data.testNames = testNames;	
    } else {
	data.testNames[data.nTests-1] = tests[data.nTests-1]->getUniqueID();
    }
    data.testNames_size = data.nTests;
    publish();
}

shared_ptr<Test> Sequence::getTest(unsigned int index) const {
    if (index >= data.nTests) {
	ostringstream s;
	s << "Tried to getTest with index: " << index << " but only " << data.nTests << " available in Sequence: " << getUniqueID();
	throw InvalidArgumentError(s.str(), __FILE__, __LINE__);
    }
    return tests[index];
}

const SequenceData& Sequence::getData() const {
    return data;
}

void Sequence::setStatus(SequenceData::status_E status) {
    //Check first
    if (data.status == SequenceData::ABORTED && status != SequenceData::ABORTED) {
	throw InvalidArgumentError("Can't change status from ABORTED in Sequence: " + getUniqueID(), __FILE__, __LINE__);
    }
    if (data.status == SequenceData::COMPLETED && status != SequenceData::COMPLETED) {
	throw InvalidArgumentError("Can't change status from COMPLETED in Sequence: " + getUniqueID(), __FILE__, __LINE__);
    }
    if (data.status != status && (status == SequenceData::COMPLETED || status == SequenceData::ABORTED)) {
	string endTime = to_iso_string(second_clock::universal_time());
	data.endTime = endTime.substr(0, 15);
    }
    
    data.status = status;
    publish();
}

void Sequence::publish() {
    string name = SctNames::getControlDataName();
    name += ".";
    name += getUniqueID();
    ISInfoDictionary& is = SctNames::getISDictionary();
    ISInfo::Status s;
    
    if (is.contains(name.c_str())) {
	s = is.update(name.c_str(), data);
    } else {
	s = is.insert(name.c_str(), data);
    }
    
    if (s != ISInfo::Success) {
	IsException error(s, "Sequence::publish failed", __FILE__, __LINE__);
	error.sendToMrs(MRS_DIAGNOSTIC);
    }
}

void Sequence::withdraw() {
    string name = SctNames::getControlDataName();
    name += ".";
    name += getUniqueID();
    ISInfoDictionary& is = SctNames::getISDictionary();
    ISInfo::Status s;
    
    s = is.remove(name.c_str());
    if (s == ISInfo::CommFailure) {
	IsException error(s, "Sequence::withdraw failed", __FILE__, __LINE__);
	error.sendToMrs(MRS_DIAGNOSTIC);
    }
}

}
    
