#ifndef CALIBRATIONCONTROLLERIMPL_H
#define CALIBRATIONCONTROLLERIMPL_H

#include "ipc/CalibrationController.h"
#include "IS/CalibrationControllerStatus.h"
#include <boost/noncopyable.hpp>
#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>

#include <memory>

using std::auto_ptr;

namespace SctCalibrationController {

class Test;
class ConfigUpdater;
class RunControl;
class SctApiAccessException;
class Sequence;

using namespace Ipc;

/**
   Implementation of the CalibrationController interface.
   A singleton, it controls the synchronisation of the various tasks required 
   during testing and calibration. 
   
   There is an abort method which allows the premature termination of a sequence or test after the end of the current scan.

   Maintains a reference to a SctApi, to which it forwards requests for scans.
   Requests for sequencees, tests and scans can be made only when in control.

   Publishes its status in IS as CalibrationControllerStatus.

   @author Matthew Palmer
*/
    
class CalibrationControllerImpl : public CalibrationController, private boost::noncopyable {
public:
    /**
    Initializes the singleton - should only be called by main or the RunController
    */
    static CalibrationControllerImpl& initialize(RunControl& rc);

    /**
    Gets the CC instance
    */
    static CalibrationControllerImpl& instance();

    /** @{ @name Overrides
    @Note all nothrow
    */

    virtual ScanLibrary& getScanLibrary() const;
    virtual TestLibrary& getTestLibrary() const;
    virtual SequenceLibrary& getSequenceLibrary() const;
    virtual void doScan( boost::shared_ptr<ScanRequest> r);
    virtual void doTest (boost::shared_ptr<TestRequest> t);
    virtual void doSequence (boost::shared_ptr<SequenceRequest> r);
    virtual void abort();
    virtual Sct_SctApi_T_Scan* getScan(unsigned long runNumber, unsigned long scanNumber) const;
    virtual void setUpdateOption( Sct_CalibrationController_T_CalibrationController_UpdateOption );
    virtual void updateWith (ilu_T_CString testResultInIs);
    ///@}

    /// @{ @name Internal for communication with RunControl:
    ///@note all nothrow
    /**
    Sets the SctApi reference
    */
    void setApi(Sct_SctApi_T_SctApi& api);

    /**
    The RunController calls this to tell the CC to take control
    The CC is then in charge until giveupControl is called
    */
    void takeControl(unsigned int runNumber);

    /**
    Called when the CC is no longer in charge
    */
    void giveupControl();

    /**
    Called when a reset is requested
    */
    void reset();
    ///@}



    ///@{ @name Internal for use by other classes
    ///@note all nothrow

    /**
    Get the handle to the SctApi
    */
    Sct_SctApi_T_SctApi* getApi() const;

    /**
    Get the current run number
    */
    unsigned int getRunNumber() const;

    /**
    Get the number of the next scan
    */
    unsigned int getNextScanNumber() const;
    ///@}

private:
    CalibrationControllerImpl(RunControl& rc);
    virtual ~CalibrationControllerImpl();
    void updateStatus();

    bool isInControl();
    bool isBusy();
    void setBusy(bool busy);
    Sct_SctApi_T_SctApi* api;

    /**
    Executes a sequence
    @note nothrow
    */
    void executeSequence(SequenceRequest& sr);

    /**
    Executes a Test
    @throws SctApiAccessException if there is some problem accessing the Api
    @throws IllegalArgumentError if the api pointer is null
    */
    auto_ptr<Test> executeTest(TestRequest& tr, boost::shared_ptr<const Sequence> s);

    /**
    Executes a Scan and waits for the Scan to be completed
    @throws SctApiAccessException if there is some problem accessing the Api
    @throws IllegalArgumentError if the api pointer is null
    */
    void executeScan (ScanRequest& s, unsigned int index);

    /**
    Applies changes from the TestResults for a given run and scan number to a list of modules
    @throws SctApiAccessException if there is some problem accessing the Api
    @throws IllegalArgumentError if the api pointer is null
    */
    void applyChanges(const unsigned long runNumber, const unsigned long scanNumber, const std::list<std::string>& list );

    /**
    Should be called by all nothrow methods when they catch an SctApiAccessException.
    Informs the RunController that an exception has occured and puts it into an error state
    */
    void sctApiError(SctApiAccessException& e);

    /**
    Sets the next scan number
    @nothrow
    */
    void setNextScanNumber(unsigned int scanNumber);

    /**
       Utility function: get all modules in all modulegroups for a particular scan.
    */
    static std::list<string> getAllModulesForScan(Sct_SctApi_T_Scan* scan);

    CalibrationControllerStatus status;                         ///< Our current status
    RunControl& rc;                                             ///< The RunController
    unsigned int runNumber;                                     ///< The current run number
    unsigned int nextScanNumber;                                ///< The number of the next scan
    volatile bool abortNow;                                     ///< Flag used to indicate the user has requested an abort
    volatile bool abortRightNow;                                ///< Flag used to indicate that the Api is no longer running and we should not wait for the end of the current scan
    static CalibrationControllerImpl* inst;                     ///< The singleton object.  Not valid until initialize has been called

    boost::thread_group m_sequence_request_thread_group;        ///< The worker group that handles executing sequences

    friend class SequenceRequestWorker;                         ///< The worker group class
};
}

#endif //#ifndef CALIBRATIONCONTROLLERIMPL_H
