#include "CalibrationControllerImpl.h"
#include "ScanLibraryImpl.h"
#include "TestLibraryImpl.h"
#include "SequenceLibraryImpl.h"
#include "SequenceRequestWorker.h"
#include "RunController.h"
#include "SctApiAccessException.h"
#include "ipc/SequenceRequest.h"
#include "scripts/DefaultSequence.h"
#include "scripts/DefaultTest.h"
#include "Sequence.h"
#include "Test.h"
#include "SctApiCall.h"
#include "ConfigUpdater/ConfigUpdaterManager.h"

#include "Sct/SctNames.h"
#include "Sct/LogicErrors.h"
#include "Sct/IoExceptions.h"
#include "Sct/IpcException.h"
#include "Sct/StdExceptionWrapper.h"
#include "Sct/IS/IOManagerIS.h"
#include "Sct/IS/IONameIS.h"
#include "Sct/ILUMarshalling.h"
#include "SctData/TestResult.h"
#include "SctData/UniqueID.h"
#include <ipc/server.h>
#include <is/isinfotmpl.h>
#include <ilu/iluxport.h>
#include <boost/utility.hpp>
#include <unistd.h>
#include <iostream>

#ifndef __Sct_SctApi_H_
#include "Sct_SctApi/Sct_SctApi.hh"
#endif


using namespace std;
using namespace Sct;
using namespace SctData;

namespace SctCalibrationController {

using namespace Scripts;

CalibrationControllerImpl::CalibrationControllerImpl(RunControl& rc) : rc(rc), runNumber(0), 
    nextScanNumber(0), abortNow(false), abortRightNow(false) {
    publish();
    status.currentScanIndex = 0;
    updateStatus();
}

CalibrationControllerImpl::~CalibrationControllerImpl() {
    withdraw();

    //Remove status from IS:
    string name = SctNames::getControlDataName();
    name += ".Status";
    ISInfoDictionary& is = SctNames::getISDictionary();
    is.remove(name.c_str());
}

Sct_SctApi_T_SctApi* CalibrationControllerImpl::getApi() const {
    return api;
}

void CalibrationControllerImpl::setApi(Sct_SctApi_T_SctApi& api) {
    this->api = &api;
}

void CalibrationControllerImpl::reset() {
    status.status = status.LOADED;
    updateStatus();
    api = 0;
}

void CalibrationControllerImpl::sctApiError(SctApiAccessException& e) {
    rc.sctApiError(rc.SCTAPI_ACCESS_FAILURE, e.what());
    status.status = status.ERROR;
    updateStatus();
}

void CalibrationControllerImpl::takeControl(unsigned int runNumber) {
    if (status.status != status.LOADED || api == 0)
        return;
    this->runNumber = runNumber;
    nextScanNumber = 0;
    status.status = status.INCONTROL;
    
    SctNames::Mrs() << "CC_START" << MRS_TEXT("Calibration Controller taking control") << MRS_INFORMATION << ENDM;
    updateStatus();
}

void CalibrationControllerImpl::giveupControl() {
    if (status.status == status.INCONTROL) {
	SctNames::Mrs() << "CC_END" << MRS_TEXT("Calibration Controller giving up control") << MRS_INFORMATION << ENDM;

        status.status = status.LOADED;
        updateStatus();
    }
}

bool CalibrationControllerImpl::isInControl() {
    return status.status == status.INCONTROL;
}

bool CalibrationControllerImpl::isBusy() {
    return status.status == status.BUSY;
}

void CalibrationControllerImpl::setBusy(bool busy) {
    if (busy && status.status == status.INCONTROL)
        status.status = status.BUSY;
    else if (!busy && status.status == status.BUSY)
        status.status = status.INCONTROL;
    updateStatus();
}

unsigned int CalibrationControllerImpl::getRunNumber() const {
    return runNumber;
}

unsigned int CalibrationControllerImpl::getNextScanNumber() const {
    return nextScanNumber;
}

void CalibrationControllerImpl::setNextScanNumber(unsigned int scanNumber) {
    nextScanNumber = scanNumber;

    APICALL(api, setScanNumber(&st, scanNumber), "setScanNumber failed in setNextScanNumber")
}

void CalibrationControllerImpl::abort() {
    if (status.status != status.BUSY) return;
    abortNow = true;
    //Notify MRS
    SctNames::Mrs() << "CC_ABORT" << MRS_TEXT("User requested abort Sequence") << MRS_INFORMATION << ENDM;

    //See if SctApi is responding, if not, then abortRightNow!
    ipcStatus status;
    ipc_T_Info info;
    api->get_info(&status, &info);
    if (status.returnCode != 0) abortRightNow = true;
}

void CalibrationControllerImpl::executeSequence(SequenceRequest& sr) {
    SctNames::Mrs() << "CC_SEQ_START" << MRS_TEXT("Calibration Controller starting sequence") << MRS_INFORMATION << ENDM;

    string name;
    try {
        //First up, synchronise our scanNumber with SctApi's
        APIRETCALL(api, nextScanNumber, getScanNumber(&st), "Couldn't synchronise scan numbers")
        //The call startSequence
	name = sr.startSequence(*api);
    } catch (SctApiAccessException& e) {
	sctApiError(e);
	throw;	//Drop through
    } catch (IpcException& e) {
	e.sendToMrs();
	return;
    }

    shared_ptr<Sequence> s;
    try {
        sctConf_T_ModuleList mlist = 0;
        APIRETCALL(api, mlist, getModuleList(&st), "Couldn't get module list")

        list<string> list = copyILUToList<_sctConf_T_ModuleList_sequence, string>(mlist);
        s = Sequence::create(name, runNumber, nextScanNumber, list);

        status.currentSequence = s->getUniqueID();
        updateStatus();

        TestRequest* tr = 0;
        unsigned long index = 0;

	do {
	    tr = sr.getNextTest(index);
	    if (tr) {
		unsigned firstScanInTest = getNextScanNumber();
		unsigned runNumber = getRunNumber();
		s->addTest(executeTest(*tr, s));

		//Check we didn't fail somewhere
		if (s->getTest(index)->getData().status != TestData::COMPLETED) {
		    SctNames::Mrs() << "CC_SEQ_END" << MRS_TEXT("Calibration Controller aborting sequence") << MRS_INFORMATION << ENDM;
		    s->setStatus(SequenceData::ABORTED);
		    return;	//All done!
		}

		if (tr->canFeedback()) {
		    std::list<std::string> moduleList = s->getModuleList();
		    applyChanges(runNumber, firstScanInTest, moduleList);
		}
	    }
	    ++index;
	} while (tr != 0);

        // Save any changes that have been made (into the ROD)
        APICALL(api, setABCDModules(&st, Sct_SctApi_T_BankType_CALIBRATION_CONFIG), "setABCDModules failed after update")

	SctNames::Mrs() << "CC_SEQ_END" << MRS_TEXT("Calibration Controller ending sequence") << MRS_INFORMATION << ENDM;
	sr.endSequence();
	s->setStatus(SequenceData::COMPLETED);

    } catch (SctApiAccessException& e) {
	sctApiError(e);
	throw;	//Drop through
    } catch (Throwable& e) {
	e.sendToMrs(MRS_INFORMATION);
	if (s) s->setStatus(SequenceData::ABORTED);
    }
}

void CalibrationControllerImpl::applyChanges(const unsigned long runNumber, const unsigned long scanNumber, const std::list<std::string>& modulelist ) {
    cout << "Update option = " << status.updateOption << endl;

    bool printed=false;
    while (status.updateOption=="WAIT") {
        if (!printed)
            cerr << "CalibrationController awaiting instructions as to whether to apply changes" << endl;
        printed=true;
    }

    //status.updateOption can be changed by other IPC method
    if (status.updateOption!="UPDATE")
        return;

    cout << "updating " << modulelist.size() << "" << " modules" << endl;

    std::ostringstream regexp;
    regexp << ".*TestResult\\." << runNumber << "\\." << scanNumber << "\\..*";
    cout << "Cal Controller looking for " << regexp.str() << endl;

    unsigned nprinted=0;

    // make another list which we can remove things from:
    std::list<std::string> mylist=modulelist;

    while (!mylist.empty() && status.updateOption=="UPDATE") {
        ISInfoIterator ii(SctNames::getPartition(),
                          SctNames::getTestDataName().c_str(),
                          regexp.str().c_str() );

        while (ii() && status.updateOption=="UPDATE") {
            if (nprinted<10)
                cout << "    checking " << ii.name() << endl;


            // make an IONameIS which allows us to interpret the IS name
            Sct::IS::IONameIS isname(ii.name());
            std::string moduleName=UniqueID(isname.getUniqueID()).getModule();

            // look for the module name in our list
            std::list<string>::iterator it = std::find( mylist.begin(), mylist.end(), moduleName);
            // if its there, then update that module, and remove it from our list.
            if ( it != mylist.end() ) {
                shared_ptr<const Serializable> ob ( Sct::IS::IOManagerIS::instance().read(ii.name()) );
                shared_ptr<const SctData::TestResult> testResult = boost::dynamic_pointer_cast<const SctData::TestResult>(ob);
                if (!testResult)
                    throw Sct::InvalidArgumentError("Fitter::applyChangesCallback not a SctData::TestResult.", __FILE__, __LINE__);

                cout << "about to update " << moduleName << endl;
		ConfigUpdaterManager::instance().update(*testResult, *getApi());
                it=mylist.erase(it);
            }
        } // end of loop over currently available TestResult

        sleep (1);

        if (nprinted<10) {
            cout << "Cal Controller waiting for TestResults (have "
            << mylist.size() << " still to go)" << endl;
        }
        nprinted ++;
    }
}

void CalibrationControllerImpl::updateWith(ilu_T_CString testResultInIs){
  setBusy(true);
  try{
    shared_ptr<const TestResult> tr = dynamic_pointer_cast<const TestResult> ( Sct::IS::IOManagerIS::instance().read(testResultInIs) );
    if (!tr.get()) throw Sct::IoException(string("Couldnt retrieve ")+testResultInIs , __FILE__, __LINE__);
    ConfigUpdaterManager::instance().update(*tr, *getApi());
  }catch(Sct::Throwable& e){
    e.sendToMrs();
  }catch(std::exception& e){
    Sct::StdExceptionWrapper(e, __FILE__, __LINE__).sendToMrs();
  }
  setBusy(false);
}

void CalibrationControllerImpl::setUpdateOption(Sct_CalibrationController_T_CalibrationController_UpdateOption opt) {
    cout << "CalibrationContrllerImpl: resetting update option to " << opt;
    switch (opt) {
    case Sct_CalibrationController_T_CalibrationController_UpdateOption_update : {
        status.updateOption="UPDATE";
        SctNames::Mrs() << "CC_UPDATE_CHANGE"
			<< MRS_TEXT("CalibrationControllerImpl setting updateOption=UPDATE") << MRS_DIAGNOSTIC << ENDM;
        break;
    }
    case Sct_CalibrationController_T_CalibrationController_UpdateOption_noupdate : {
	status.updateOption="NOUPDATE";
	SctNames::Mrs() << "CC_UPDATE_CHANGE"
			<< MRS_TEXT("CalibrationControllerImpl setting updateOption=NOUPDATE")  << MRS_DIAGNOSTIC << ENDM;
	break;
    }
    case Sct_CalibrationController_T_CalibrationController_UpdateOption_wait : {
	status.updateOption="WAIT";
	SctNames::Mrs() << "CC_UPDATE_CHANGE"
			<< MRS_TEXT("CalibrationControllerImpl setting updateOption=WAIT")  << MRS_DIAGNOSTIC << ENDM;
	break;
    }
    default : {
	IllegalStateError e("CalibrationControllerImpl updateOption UNKNOWN", __FILE__, __LINE__);
	e.sendToMrs(MRS_DIAGNOSTIC);
	break;
    }
    }
    updateStatus();
}

auto_ptr<Test> CalibrationControllerImpl::executeTest(TestRequest& tr, shared_ptr<const Sequence> s) {
    TestData data;
    vector<double> testPoints;

    SctNames::Mrs() << "CC_TEST_START" << MRS_TEXT("Calibration Controller starting test") << MRS_INFORMATION << ENDM;

    APICALL(api, setABCDModules(&st, Sct_SctApi_T_BankType_CALIBRATION_CONFIG), "setABCDModules failed")

    data.runNumber = runNumber;
    data.startScanNumber = nextScanNumber;
    tr.startTest(*api, data.testName, data.testVariable, data.nScans, testPoints);

    data.testPoints = new double[testPoints.size()];
    data.testPoints_size = testPoints.size();
    for (unsigned int i=0; i<testPoints.size(); ++i) {
        data.testPoints[i] = testPoints[i];
    }

    auto_ptr<Test> test (new Test(data, s->getModuleList()));

    status.currentTest = test->getUniqueID();
    updateStatus();

    try {
    	unsigned int i=0;
	for (; i<data.nScans; ++i) {
	    ScanRequest* scan = tr.getNextScan(i);
	    if (scan==0) throw IllegalStateError("getNextScan returned zero!", __FILE__, __LINE__);
	    if (scan->getScan()==0) throw IllegalStateError("ScanRequest.getScan returned zero!", __FILE__, __LINE__);
	    test->addScan(scan->getScan());
	    if (abortNow) {
		abortNow = false;
		abortRightNow = false;
		break;
	    }
#warning executeScan may call ROD doRawScan() which can raise an exception!
	    executeScan(*scan, i);
	}
	if (i == data.nScans) {
	    SctNames::Mrs() << "CC_TEST_END" << MRS_TEXT("Calibration Controller ending test") << MRS_INFORMATION << ENDM;
	    test->setStatus(TestData::COMPLETED);
	} else {
	    SctNames::Mrs() << "CC_TEST_ABORT" << MRS_TEXT("Calibration Controller aborting test") << MRS_INFORMATION << ENDM;
	    test->setStatus(TestData::ABORTED);

	    const TestData& testData = test->getData();
	    setNextScanNumber(testData.startScanNumber + testData.nScans);
	}

    } catch (SctApiAccessException& e) {
	sctApiError(e);
	throw;	//Drop through
    } catch (IpcException& e) {
	SctNames::Mrs() << "CC_TEST_ABORT" << MRS_TEXT("Calibration Controller aborting test") << MRS_INFORMATION << ENDM;
	test->setStatus(TestData::ABORTED);
	e.sendToMrs(MRS_INFORMATION);

	//Try an increment the scan number to the end of the test
	const TestData& testData = test->getData();
	setNextScanNumber(testData.startScanNumber + testData.nScans);
    }

    //End the test
    try {
    	tr.endTest();
    } catch (IpcException& e) {
	e.sendToMrs(MRS_DIAGNOSTIC);
    }

    //Reset the module configurations to the original
    try {
        APICALL(api, getABCDModules(&st, Sct_SctApi_T_BankType_CALIBRATION_CONFIG), "getABCDModules failed")
        APICALL(api, sendABCDModules(&st, Sct_SctApi_T_BankType_PHYSICS_CONFIG), "sendABCDModules failed")
    } catch (SctApiAccessException& e) {
        sctApiError(e);
        e.sendToMrs();
    }

    return test;
}

void CalibrationControllerImpl::executeScan(ScanRequest& scan, unsigned int index) {
    ++nextScanNumber;
    SctNames::Mrs() << "CC_SCAN_START" << MRS_TEXT("Calibration Controller starting scan") << MRS_INFORMATION << ENDM;

    status.currentScanIndex = index;
    updateStatus();

    if ( scan.isRaw() ){
      APICALL(api, doRawScan(&st, scan.getScan(), scan.delay(), scan.width(), scan.configureModules(), scan.clockByTwo()  ), "doRawScan failed");
    }else{
      APICALL(api, doScan(&st, scan.getScan() ), "doScan failed");
    }

    ISInfoInt i;
    i.setValue(-1);
    ISInfo::Status isStatus = ISInfo::Success;
    ISInfoDictionary& is = SctNames::getISDictionary();
    int nInitialSleeps = 0;

    while ((i.getValue() == -1 && nInitialSleeps < 20) || (i.getValue() == 1 && isStatus == ISInfo::Success)) {
        isStatus = is.findValue("SCTAPIServer.scanning", i);
        usleep(100);
	++nInitialSleeps;
	if (abortRightNow) {
	    abortRightNow = false;
	    abortNow = false;
	    throw SctApiAccessException("Abort called and SctApi not responding", "doScan failed", __FILE__, __LINE__);
	}
    }

    SctNames::Mrs() << "CC_SCAN_END" << MRS_TEXT("Calibration Controller ending scan") << MRS_INFORMATION << ENDM;
}

CalibrationControllerImpl& CalibrationControllerImpl::initialize(RunControl& rc) {
    inst = new CalibrationControllerImpl(rc);
    return *inst;
}

CalibrationControllerImpl& CalibrationControllerImpl::instance() {
    if (inst == 0)
	throw InvariantViolatedError("Need to call CalibrationControllerImpl::initialize first!", __FILE__, __LINE__);
    return *inst;
}

ScanLibrary& CalibrationControllerImpl::getScanLibrary() const {
    return ScanLibraryImpl::instance();
}

TestLibrary& CalibrationControllerImpl::getTestLibrary() const {
    return TestLibraryImpl::instance();
}

SequenceLibrary& CalibrationControllerImpl::getSequenceLibrary() const {
    cout << "Get Sequence Library" << endl;
    return SequenceLibraryImpl::instance();
}

void CalibrationControllerImpl::doScan(shared_ptr<ScanRequest> scan) {
    shared_ptr<DefaultTest> test(new DefaultTest(scan));
    doTest(test);
}

void CalibrationControllerImpl::doTest (shared_ptr<TestRequest> t) {
    shared_ptr<DefaultSequence> seq(new DefaultSequence(t));
    doSequence(seq);
}

void CalibrationControllerImpl::doSequence (shared_ptr<SequenceRequest> r) {
    if (!isInControl() || isBusy()) {
        cerr << "Can't execute SequenceRequest - bad status: " << status.status << endl;
        return;
    }

    cout << "do Sequence" << endl;

    status.status = status.BUSY;  // set back to INCONTROL by the SequenceRequestWorker
    updateStatus();


    m_sequence_request_thread_group.create_thread(SequenceRequestWorker(*this, r));
}

Sct_SctApi_T_Scan* CalibrationControllerImpl::getScan(unsigned long runNumber, unsigned long scanNumber) const {
  cerr << "CalibrationControllerImpl::getScan - Not implemented" << endl;
  return 0;
}

void CalibrationControllerImpl::updateStatus() {
    string name = SctNames::getControlDataName();
    name += ".Status";
    ISInfoDictionary& is = SctNames::getISDictionary();
    ISInfo::Status st;

    if (is.contains(name.c_str())) {
        st = is.update(name.c_str(), status);
    } else {
        st = is.insert(name.c_str(), status);
    }
    if (st != ISInfo::Success) {
        IsException error(st, "CalibrationControllerImpl can't update status", __FILE__, __LINE__);
        error.sendToMrs(MRS_DIAGNOSTIC);
    }
}

CalibrationControllerImpl* CalibrationControllerImpl::inst = 0;

}
