#ifndef SCTARCHIVING_ZLIBSTRINGCOMPRESSOR_H
#define SCTARCHIVING_ZLIBSTRINGCOMPRESSOR_H
#include <zlib.h>
#include <string>
using std::string;

namespace SctArchiving{
/** 
    Uses zlib to compress a string into another (hopefully shorter!) string.
    Can also extract a compressed string. The extraction step requires a buffer,
    so the class has an interal buffer, which it increases in size if necessary,
    to fit the uncompressed string.
    Sends mrs messages if it encounters problems with zlib.
    Designed so that a single instance of this class should be re-used many times,
    so clients should initiate one and then continue to re-use it for 
    best use of the buffering.
    Note that returned `strings' are actually references to data held in internal buffers.
    @note the buffer size required for extraction cannot be known in advanced.
    So the buffer can dynamically increase its size, to allow for bigger 
    `original sized' strings.
*/
class ZlibStringCompressor{
 public:
  /** 
      constructor, with optional bufferSizeArgument;
  */
  ZlibStringCompressor(unsigned long bufferSize=0);
  /** 
      Compress a string. 
      @param reference to string to compress 
      @param gzip compression level 1->9;
      @return compressed string (actual string is stored in buffer.)
   */
  const string compress(const string& toCompress, int level);
  /** 
      Uncompress a string. 
      @param reference to compressed string. 
      @return uncompressed equivalent (actual string is stored in buffer.)
  */
  const string inflate(const string& compressed);
  /** 
      Find the size of the buffer in bytes (for information);
   */
  unsigned long getBufferSize();
  /** 
      Level of 0 produces no output. Higher levels produce more output.
   */
  void setDebugLevel(int level);
  /**
     return the value of the debug level.
  */
  int getDebugLevel();
  /** 
      set the buffer size (in bytes) 
  */
  Bytef* setBufferSize(unsigned long size);
private:
  /** 
      Check return result number from zlib, throwing InvalidArgument if necessary 
  */
  static void checkZlibResult(const int result, const char* file, const int line);
  /** 
      size in bytes of the buffer 
  */
  long unsigned m_bufferSize;
  /*
    pointer to start of buffer
  */
  Bytef* m_buffer;
  /** 
      Find the length in bytes of a string
  */
  static unsigned long findSizeInBytes(const string& s);
  /** 
      Find the minimum buffer size for compresses string 
      @param uncompressed size
  */
  static unsigned long findRequiredBufferSize(const unsigned long uncompressedSize);
  int m_debugLevel;
};


inline unsigned long ZlibStringCompressor::findSizeInBytes(const string&s ){
  return sizeof(char)*s.size();
}

// buffer size myst be at least original size + 0.1% + 12 bytes
inline unsigned long ZlibStringCompressor::findRequiredBufferSize(long unsigned unbufferedSize){
  return unbufferedSize + unbufferedSize/1000 + 12;
}

inline void ZlibStringCompressor::setDebugLevel(int level){
  m_debugLevel=level;
}

inline int ZlibStringCompressor::getDebugLevel(){
  return m_debugLevel;
}

}

#endif //  SCTARCHIVING_ZLIBSTRINGCOMPRESSOR_H
