//#include "Sct/isstream_bugfix.h"
#include "Archiver.h"
#include "ArchivingWorkerGroup.h"
#include "Sct/Archive/IOManagerArchiveFile.h" 
#include "Sct/SctNames.h"
#include "Sct/IpcObjectException.h"
#include "Sct/UnsupportedOperationError.h"
#include "Sct/StdExceptionWrapper.h"
#include "Sct/ISStreamerWrapper.h"
#include "is/isinfo.h"
#include <pmg/pmg_initSync.h>

#include "unistd.h"
#include <memory>

#include "TransferCommand.h"
#include "IsGetCommand.h"
#include "IsPutCommand.h"
#include "ArchiveGetCommand.h"
#include "ArchivePutCommand.h"
#include "Sct/Archive/IOParamsArchive.h"

using namespace std;

using namespace Sct;
using namespace Sct::Archive;
using namespace Sct::IS;

using boost::mutex;

void pmgSynch(void *) {
    pmg_initSync();
}

int main(int argc, char** argv) {
    using namespace SctArchiving;
    setExceptionHandlers(argv[0]);

    // renice this job!
    nice(10);

    bool multiThreaded = true; //Choose single/multi-threaded
    IPCCore::init(multiThreaded);
    
    try {
      Archiver& s=Archiver::instance();
      if (!s.publish()){
	throw IpcObjectException("ArchivingService failed to publish", __FILE__, __LINE__);
      }
      s.go(1);
      s.getArchiverServer().doSoon(pmgSynch, NULL);
      s.getArchiverServer().run();
    } catch (Throwable& e) {
      e.sendToMrs(MRS_FATAL);
      terminate();
    }
}

namespace SctArchiving {

Archiver* Archiver::archiver = 0;

Archiver& Archiver::instance() {
  if(!archiver) archiver = new Archiver();
  return *archiver;
}

  Archiver::Archiver() throw(ConfigurationException) : IPCObject(ArchivingServiceI_C_ArchivingServiceInterface_instanceName, &getArchiverServer()),
     nArchived(), nRetrieved(0), nValidated(0), isTimeTaken(0.), fileTimeTaken(0.) {
  
  auto_ptr<ISInfoReceiver> ir(new ISInfoReceiver(SctNames::getPartition()));
  if (!ir.get()) throw ConfigurationException("Archiver::initialize can't make infoReceiver ", __FILE__, __LINE__) ;
  infoReceiver = ir;
  workergroup = new ArchivingWorkerGroup();
  cout << "constructed" << endl;
  cout << "Persistent dir:" << SctNames::getPersistentDir() << endl;
  cout << "Retrieval is server : " << SctNames::getRetrievedDataName() << endl;
  m_archive_manager = &IOManagerArchiveFile::instance();
  m_retrieval_is_server=Sct::SctNames::getRetrievedDataName();
  m_suspend_callback=false;
}

Archiver::~Archiver() throw() {
  cout << "ArchivingService waiting for queue to finish " << endl;
  while (workergroup->busy() || workergroup->queueSize() ) {
    sleep(1);
  }
  workergroup->join();
}

void Archiver::addCommand(boost::shared_ptr<ArchivingCommand> command)const {
  workergroup->push(command);
}

IPCServer& Archiver::getArchiverServer() throw() {
  static IPCServer archiverServer(ArchivingServiceI_C_ArchivingServiceInterface_serverName, SctNames::getPartition());
  return archiverServer;
}

inline bool isControl(std::string name){
  return ( name.find("TestData")!=string::npos || name.find("ControlData")!=string::npos);
};

void Archiver::archive_callback(ISCallbackInfo *isc){
  try{
    bool over_write=isControl(isc->name());
    if (isc->reason() == ISInfoUpdated && !over_write ) {
      throw IoException(string("That is odd ... ") + isc->name() +  " has been modified! I'm not going to overwrite the data."
			, __FILE__, __LINE__);
    }
    if (isc->reason() != ISInfoCreated && isc->reason() != ISInfoUpdated) return;
    if (Archiver::instance().m_suspend_callback){
	SctNames::Mrs() << MRS_TEXT("ArchivingService being suspended from callbacks") << "ARCHIVE_SUSPEND" << MRS_INFORMATION << ENDM;
        return;
    }
    cout << "Archiving " << isc->name() << endl;

    shared_ptr<IONameIS> name ( new IONameIS(isc->name()));
    shared_ptr<GetCommand> get (new IsGetCommand(name));
    shared_ptr<PutCommand> put (new ArchivePutCommand());

    if (over_write){
      shared_ptr<Sct::Archive::IOParamsArchive> params;
      params = shared_ptr<Sct::Archive::IOParamsArchive>(new IOParamsArchive());
      params->setOverWrite();
      put->setParams(params);
    }

    shared_ptr<TransferCommand> command( new TransferCommand(get, put));
    Archiver::instance().workergroup->push( command );
    
  } catch (Sct::Throwable& e){
    e.sendToMrs();
  }catch(std::exception& e){
    StdExceptionWrapper(e).sendToMrs();
  }
}

IOManagerArchive& Archiver::getIOManagerArchive() const {
  return *m_archive_manager;
}

string Archiver::getRetrieveIsServer() const throw(){
  return  m_retrieval_is_server;
}

void  Archiver::setRetrieveIsServer (ArchivingServiceIStatus *_status, ilu_T_CString serverName){
  _status->returnCode = ArchivingServiceIReply_Success;
  m_retrieval_is_server=serverName;
}

void Archiver::suspendCallbacks (ArchivingServiceIStatus *_status, ilu_Boolean doSuspend){
   _status->returnCode = ArchivingServiceIReply_Success;
   m_suspend_callback=doSuspend;
}


void Archiver::subscribe(const string& servername, const string& regexp, ISCallbackInfo::Callback callback){
  if (servername==SctNames::getRetrievedDataName()){
    ostringstream oss;
    oss << "ArchivingService is not allowed to subscribe to " << servername 
	<< " since this could set up a loop";
    IoException(oss.str(), __FILE__, __LINE__).sendToMrs();
  }
  ISInfo::Status s=infoReceiver->subscribe(servername.c_str(), regexp.c_str(), callback);
  if (s!=ISInfo::Success) {
    ostringstream os;
    os <<"Could not subscribe to "<<servername<< " to retrieve " << regexp;
    IsException(s, os.str(), __FILE__, __LINE__).sendToMrs();
  } else {
    cout << "Subscribed to " << servername << " to retrieve " << regexp << endl;
  }
}
  
void Archiver::go(unsigned nWorker) throw(IsException) {
  subscribe(SctNames::getEventDataName(), "SctData::RawScanResult.*", archive_callback);
  subscribe(SctNames::getFittedDataName(), "SctData::FitScanResult.*", archive_callback);
  subscribe(SctNames::getTestDataName(), "SctData::.*TestResult.*", archive_callback);
  subscribe(SctNames::getControlDataName(), "TestData.*", archive_callback);
  subscribe(SctNames::getControlDataName(), "SequenceData.*", archive_callback);
  workergroup->go(nWorker);
}

  void Archiver::incrimentNArchived() throw(){
    mutex::scoped_lock scoped_lock(counterMutex);
    ++nArchived;
  }
  
  void Archiver::incrimentNRetrieved() throw(){
    mutex::scoped_lock scoped_lock(counterMutex);
    ++nRetrieved;
  }
  
  void Archiver::incrimentNValidated() throw(){
    mutex::scoped_lock scoped_lock(counterMutex);
    ++nValidated;
  }
  
  void Archiver::addISTime(double time) throw(){
    mutex::scoped_lock scoped_lock(counterMutex);
    isTimeTaken += time;
  }

  void Archiver::addFileTime(double time) throw(){
    mutex::scoped_lock scoped_lock(counterMutex);
    fileTimeTaken += time;
  }

const char* Archiver::getStatus() const throw(){
  ostringstream os;
  os << "pid = " << getpid() << endl;
  os << "Workers=" << workergroup->nWorkers();
  os << ", (Busy=" << workergroup->busy() << ")" 
     << ". Queue = " << workergroup->queueSize() << endl
     << "# Archived = " << nArchived
     << " ; # Retrieved = " << nRetrieved  
      << " ; # Validated = " << nValidated << endl
     << " Approx average time / command / thread = " << getAverageTime(fileTimeTaken)
     << " (Archive) + " << getAverageTime(isTimeTaken) << " (IS/ISProxy)" << endl;
  os << "Compression level = " << getIOManagerArchive().getCompressionLevel() << endl;
  os << "IOManagerArchive Information: \n"
     << getIOManagerArchive().status();
  return os.str().c_str();
}

  double Archiver::getAverageTime(double time) const throw(){
    long ncommands =  nArchived + nRetrieved + nValidated;
    return (ncommands>0) ? time/ncommands : 0.;
  }

  char* Archiver::status(ArchivingServiceIStatus* status){
    status->returnCode = ArchivingServiceIReply_Success;
    char *statusMessage = new char[300];
    strcpy(statusMessage, getStatus());
    return statusMessage;
  }
  
  void Archiver::archiveISName(ArchivingServiceIStatus *status, char* ioNameIS){
    try {
      status->returnCode = ArchivingServiceIReply_Success;
      shared_ptr<IONameIS> name( new IONameIS(ioNameIS));
      shared_ptr<GetCommand> get (new IsGetCommand(name));
      shared_ptr<PutCommand> put (new ArchivePutCommand());

      if (isControl(ioNameIS)){
	shared_ptr<Sct::Archive::IOParamsArchive> params;
	params = shared_ptr<Sct::Archive::IOParamsArchive>(new IOParamsArchive());
	params->setOverWrite();
	put->setParams(params);
      }

      shared_ptr<TransferCommand> command( new TransferCommand(get, put));
      workergroup->push( command );
    }catch(Throwable& e){
      e.sendToMrs();
    }
  }

  void Archiver::retrieveISName (ArchivingServiceIStatus *_status, char* ioNameIS){
    try{ 
      IONameIS nameis(ioNameIS);
      IONameArchiveFile archfile( nameis.getUniqueID(), nameis.getClassName());
      /// need to do this cos IPC dosent guarentee constness
      char* blah = const_cast<char*> (archfile.getIOName().c_str());
      retrieveArchName (_status, blah);
    }catch(Throwable& e){
      e.sendToMrs();
    }
  }

  void Archiver::retrieve(ArchivingServiceIStatus *_status, char* runNumber, char* scanNumber, char* className, char* specifier){
    try{ 
      std::ostringstream oss;
      oss << SctNames::getPersistentDir() << "/" << className 
	  << "." << runNumber << "." << scanNumber 
	  << "." << specifier << ".gz";
      IONameArchiveFile arch(oss.str());
      char* blah = const_cast<char*>(arch.getIOName().c_str());
      retrieveArchName(_status, blah);
    }catch(Throwable& e){
      e.sendToMrs();
    }
  }

  void Archiver::retrieveArchName (ArchivingServiceIStatus *status, char* archivingName){
    try{
      status->returnCode = ArchivingServiceIReply_Success;
      shared_ptr<IONameArchiveFile> name(new IONameArchiveFile(archivingName));
      shared_ptr<GetCommand> get (new ArchiveGetCommand(name));
      shared_ptr<IsPutCommand> put (new IsPutCommand());
      put->setServer(getRetrieveIsServer());
      shared_ptr<TransferCommand> command( new TransferCommand(get, put));
      workergroup->push( command );
    }catch(Throwable& e){
      e.sendToMrs();
    }
  }

  
  ilu_ShortInteger Archiver::busy(ArchivingServiceIStatus* status){
    status->returnCode = ArchivingServiceIReply_Success;
    return workergroup->busy();
  }

  ilu_ShortInteger Archiver::queueLength(ArchivingServiceIStatus* status){
    status->returnCode = ArchivingServiceIReply_Success;
    return workergroup->queueSize();
  }

  ilu_ShortInteger Archiver::getCompressionLevel (ArchivingServiceIStatus *status) {
    status->returnCode = ArchivingServiceIReply_Success;
    return getIOManagerArchive().getCompressionLevel();
  }
  
  void Archiver::setCompressionLevel ( ArchivingServiceIStatus *status, ilu_ShortInteger level) {
    status->returnCode = ArchivingServiceIReply_Success;
    getIOManagerArchive().setCompressionLevel(level);
  }
} // end of namespace SctArchiving
