#include "TrimRangeAlgorithm.h"
#include "AnalysisAlgorithmMap.h"
#include "AnalysisService.h"
#include "Sct/SctParameters.h"
#include "Sct/UnsupportedOperationError.h"
#include "SctData/DefectPrototype.h"
#include "SctData/TrimRangeTestResult.h"
#include "SctData/FitScanResult.h"
#include "SctData/FitObject.h"
#include "SctData/StandardDefects.h"
#include "SctData/ModuleConfiguration.h"
#include "SctData/ChipConfiguration.h"
#include "SctData/ResponseCurve.h"
#include <boost/shared_ptr.hpp>
#include <algorithm>
#include <memory>
#include <vector>

using namespace Sct;
using namespace SctData;
using namespace boost;

namespace SctAnalysis {

bool debug = false;
    
bool TrimRangeAlgorithm::inMap = AnalysisAlgorithmMap::instance().setAlgorithm("TrimRangeTest", auto_ptr<AnalysisAlgorithm>(new TrimRangeAlgorithm()));

shared_ptr<AnalysisAlgorithm> TrimRangeAlgorithm::clone(const TestData& testData, const string& moduleName) const throw() {
    return shared_ptr<AnalysisAlgorithm>(new TrimRangeAlgorithm(testData, moduleName, *this));
}

bool TrimRangeAlgorithm::s_targetVariation = false;
bool TrimRangeAlgorithm::s_rangeVariation = true;

void TrimRangeAlgorithm::loadData() {
    loadAllFits();
}

bool TrimRangeAlgorithm::canAnalyze() const{
    return hasAllFits();
}

shared_ptr<TestResult> TrimRangeAlgorithm::createTestResult() const {
        return shared_ptr<TestResult>(new TrimRangeTestResult());    
}

void TrimRangeAlgorithm::analyze() {           
    shared_ptr<TrimRangeTestResult> result = dynamic_pointer_cast<TrimRangeTestResult> ( getTestResult() );
    result->setScanVariable(getFit(0)->getHeader().getVariable());
    const ModuleConfiguration& config = getFit(0)->getConfiguration();

    /*
       STAGE 1 : make sense of the incoming data.
       make a vector of all ranges, all points in those ranges and the 
       appropriate scan number.
    */
    std::vector<TrimRange> all_ranges = createRanges(*result);

    /*
       STAGE 2 : make graphs of trim against vt50 for each chip
       and each channel and each range
    */

    shared_ptr<SctData::TrimRangeTestResult::ChipTrimData> dataArray[all_ranges.size()][nChipModule];

    // loop over trim ranges
    for (unsigned irange=0; irange<all_ranges.size(); ++irange) {

        TrimRange& range = all_ranges[irange];

        // loop over chips
        for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
            dataArray[irange][ichip] =
                shared_ptr<TrimRangeTestResult::ChipTrimData>(new TrimRangeTestResult::ChipTrimData());

            // loop over trim-scan points
            for (unsigned ipoint=0; ipoint<range.points.size(); ++ipoint) {

                TrimPoint& this_point = range.points[ipoint];
                shared_ptr<const FitScanResult> fitted=getFit(this_point.scan);
                const DefectList& fitDefects=fitted->getDefects();

                // loop over channels in that chip.
                for (unsigned iChanInChip=0; iChanInChip<nChannelChip; ++iChanInChip) {
                    unsigned ichannel=iChanInChip + ichip*nChannelChip;

                    // get the relevant TrimData object
                    Stat<TrimRangeTestResult::TrimData>& trdt = dataArray[irange][ichip]
                            ->channelData.modifyAt(iChanInChip);

                    // channel is valid if there are no defects on it.
                    if(fitDefects.defectSeverityEncompassingElement(ModuleElement::Channel(ichannel)) >= SERIOUS) {
                        trdt.valid=false;
                        if (debug) {
                            cout << "Severe defect on Channel " << ichannel << endl;
                            //fitDefects.getDefectsEncompassingElement( ModuleElement::Channel(ichannel))->print(cout);
                        }
                    }
		    if (config.getChipConfiguration(ichip).isMasked(iChanInChip)) {
			trdt.valid=false;
			if (debug) cout << "Masked channel: " << ichannel << endl;
		    }
                    if (trdt.valid) {
                        float gr_mean = fitted->getChannelFit(ichannel).getParameter(1);
                        float gr_value = this_point.value;
                        trdt.value.graph.push_back(std::make_pair(gr_mean, gr_value));
                    } // if channel is valid
                } // loop over channels
            } // loop over scan-points
        } // loop over chips
    } // loop over ranges


    /*
      STAGE 3:
      create a vector of target s starting from firstTarget in uniform steps
      to a total of ntargets
    */
    std::vector<float> target;
    for (unsigned itarg=0; itarg<nTargets(); ++itarg) {
        target.push_back (firstTarget() + itarg * targetStep());
    }

    shared_ptr<const TrimRangeTestResult::ChipTrim> trimArray[all_ranges.size()][nChipModule][nTargets()];

    /*
      STAGE 4a: fit the graphs 
      STAGE 4b: find the number of trimmable channels for each in a series of targets
    */
    for (unsigned irange=0; irange<all_ranges.size(); ++irange) {
        for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
            // 4a:
            if (dataArray[irange][ichip].get()==0) {
                throw InvariantViolatedError("TrimRangeAlgorithm: dataArray pointer should not be zero!", __FILE__, __LINE__);
            }
            doTrimDataFit( *dataArray[irange][ichip] );
            for (unsigned itarget=0; itarget<nTargets(); ++itarget) {
                //4b:
		if (debug && target[itarget]> 85 && target[itarget]<95) 
		    cout << "Chip: " << ichip << " Range: " << all_ranges[irange].range << endl;
                trimArray[irange][ichip][itarget] = getChipTrim(*dataArray[irange][ichip],
                                                    target[itarget],
                                                    all_ranges[irange].range);
            }
        }
    }

    /*
      STAGE 5 select the best trim target which trims the maximum number of channels
    */

    unsigned best_target_for_chip[nChipModule];       // index of vector targets
    unsigned best_range_for_chip[nChipModule];        // index of vector all_ranges

    // 5a no chip-to-chip variation:
    if ( !allowTrimTargetVariation() && !allowTrimRangeVariation() ) {
	if (debug) cout << "stage 5a" << endl;
        unsigned max_trimmed=0;
        unsigned target_lo=0, target_hi=0;

        for (unsigned irange=0; irange<all_ranges.size(); ++irange) {
            for( unsigned itarget=0; itarget<nTargets(); ++itarget) {
                unsigned n_trimmed=0;
                /* find number of channels which are trimmable in the module */
                for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
                    n_trimmed+=trimArray[irange][ichip][itarget]->channelTrim.n();

                }
                if (n_trimmed>max_trimmed) {
                    target_lo=itarget;
                    best_range_for_chip[0] = irange;
                    max_trimmed=n_trimmed;
                }
                if (n_trimmed==max_trimmed && irange==best_range_for_chip[0]) {
                    target_hi=itarget;
                }
            }
        }
        for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
            best_range_for_chip[ichip]=best_range_for_chip[0];
            best_target_for_chip[ichip]=(target_hi+target_lo)/2;
        }
    } else if ( allowTrimTargetVariation() && !allowTrimRangeVariation() ) { // 5b chip-to-chip target variation:
	if (debug) cout << "stage 5b" << endl;
	unsigned target_lo[nChipModule];
	unsigned target_hi[nChipModule];
	unsigned max_trimmed[nChipModule];
	for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
	    target_lo[ichip]=target_hi[ichip]=max_trimmed[ichip]=0;
	}	

	for (unsigned irange=0; irange<all_ranges.size(); ++irange) {
	    for( unsigned itarget=0; itarget<nTargets(); ++itarget) {
		unsigned n_trimmed=0;
		/* find number of channels which are trimmable in the module */
		for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
		    n_trimmed = trimArray[irange][ichip][itarget]->channelTrim.n();
		    
		    if (n_trimmed>max_trimmed[ichip] ) {
			target_lo[ichip]=itarget;
			best_range_for_chip[ichip] = irange;
		    }
		    if (n_trimmed==max_trimmed[ichip] && irange==best_range_for_chip[ichip]) {
			target_hi[ichip]=itarget;
		    }
		}
	    }
	}
	for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
	    best_target_for_chip[ichip]=(target_hi[ichip]+target_lo[ichip])/2;
	}
    } else if ( !allowTrimTargetVariation() && allowTrimRangeVariation() ) {           // 5c chip-to-chip range variation:
	if (debug) cout << "stage 5c"<<endl;
	unsigned target_lo=0;
	unsigned target_hi=0;
	unsigned total_channels_trimmed_max=0;
	unsigned total_range_lo=0;

	for( unsigned itarget=0; itarget<nTargets(); ++itarget) {
	    if (debug)
		cout << "TARGET " << itarget << "  " << target[itarget] << "mV" << endl;
	    unsigned total_channels_trimmed=0;
	    unsigned range_total=0;
	    
	    for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
		unsigned best_range=0;
		unsigned max = 0;

		for (unsigned irange=0; irange<all_ranges.size(); ++irange) {
		    unsigned n_trimmed = trimArray[irange][ichip][itarget]->channelTrim.n();

		    // if this is the lowest range with the maximum number of channels:
		    if (n_trimmed > max ||
			( all_ranges[irange].range < all_ranges[best_range].range
			  && n_trimmed==max)) {
			best_range = irange;
			max = n_trimmed;
		    }
		}
		// find the total number of channels trimmed
		total_channels_trimmed+=max;
		range_total+=all_ranges[best_range].range ;
	    }
	    if (debug)
		cout << "  TOTAL = " <<  total_channels_trimmed << " for " << range_total << endl;
	    
	    if (total_channels_trimmed > total_channels_trimmed_max ||
		( total_channels_trimmed == total_channels_trimmed_max &&
		  range_total < total_range_lo ) ) {
		target_lo=itarget;
		total_range_lo=range_total;
		total_channels_trimmed_max=total_channels_trimmed;
		if (debug)
		    cout << " ...... modified lower target" << endl;
	    }

	    if (total_channels_trimmed == total_channels_trimmed_max &&
		total_range_lo==range_total ) {
		target_hi=itarget;
		if (debug)
		    cout << " ...... modified upper target" << endl;
	    }

	}
	for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
	    best_target_for_chip[ichip]=(target_hi+target_lo)/2;
	    // know the target, get the ranges again:
	    if (debug)
		cout << endl << " --> CHIP " << ichip << endl;
	    unsigned max=0;
	    best_range_for_chip[ichip]=0;

	    for (unsigned irange=0; irange<all_ranges.size(); ++irange) {
		unsigned n_trimmed = trimArray[irange][ichip][best_target_for_chip[ichip]]->channelTrim.n();
		if (debug)
		    cout << "R" << irange << " -> " << n_trimmed << endl;
		if (n_trimmed> max ||
		    ( all_ranges[irange].range < all_ranges[best_range_for_chip[ichip]].range
		      && n_trimmed==max)) {
		    best_range_for_chip[ichip] = irange;
		    max=n_trimmed;
		    if (debug)
			cout << "best range changed to #" << irange << " = " << all_ranges[irange].range << endl;
		}
	    }
	}
    } else {
	throw UnsupportedOperationError("Trim Range Algorithm not implemented for desired options!", __FILE__, __LINE__);
    }
    
    /* STAGE 6 fill output TestResult data */
    if (debug)
        cout << "stage 6"<<endl;

    {
        // get a chip configuration for an example chip:
        const ChipConfiguration& config=getFit(0)->getConfiguration().getChipConfiguration(0);
        auto_ptr<ResponseCurve> rc = ResponseCurveMap::getMap().get(config);
        float cal_charge_mV = getFit(0)->getConfiguration().getChipConfiguration(0).getCalCharge()*2.5 ;
        float cal_charge_fC = rc->getFunction()->Eval(cal_charge_mV);

        result->charge = cal_charge_fC;
        result->type = allowTrimRangeVariation() ? (unsigned)-1 : best_range_for_chip[0] ;
        result->algorithm = allowTrimTargetVariation() ? 1 : 0 ;
    }

    for (unsigned ichip=0; ichip<nChipModule; ++ichip) {
        if (debug)
            cout << "  CHIP " << ichip << endl;
        unsigned best_range  = best_range_for_chip[ichip];
        unsigned best_target = best_target_for_chip[ichip];

        if (debug)
            cout << "    optimum range = " <<  best_range
            << ", target #" << best_target
            << " = " << target[best_target]
            << ", ntrim=" << trimArray[best_range][ichip][best_target_for_chip[ichip]]->channelTrim.n() << endl;

        result->chipTrim[ichip] =
            trimArray [best_range] [ichip] [best_target];
        result->chipTrimData[ichip]=
            dataArray[best_range] [ichip];

        const Stats<double> offsets = dataArray[best_range][ichip]->getOffsets();
        const Stats<double> steps   = dataArray[best_range][ichip]->getSteps();

        double stepmin=0, stepmax=16;
        switch (all_ranges[best_range].range) {
        case 0 :
            stepmin = 1.5 ;
            stepmax = 5.0 ;
            break;
        case 1 :
            stepmin = 5.0 ;
            stepmax = 8.5 ;
            break;
        case 2 :
            stepmin = 8.5 ;
            stepmax = 12.0;
            break;
        case 3 :
            stepmin = 12.0;
            stepmax = 15.5;
            break;
        }
        // Chip Defects:
        if (steps.mean() < stepmin || steps.mean() > stepmax ) {
            result->getDefects().addDefect(Defect(StandardDefects::TR_RANGE, ModuleElement::Chip(ichip)));
        }
        // Channel Defects:
        for (unsigned iChanInChip=0; iChanInChip<nChannelChip ; ++iChanInChip) {
            // Step Defects:
            if (steps.getAt(iChanInChip).valid &&
                    ( fabs( steps.getAt(iChanInChip).value - steps.mean() )
                      > StandardDefects::TR_STEP.getParameter() * sqrt(steps.var() ) ) ) {
                result->getDefects().addDefect(Defect(StandardDefects::TR_STEP,
                                               ModuleElement::Channel(ichip*nChannelChip+iChanInChip)));
            }
            // Offset Defects:
            if (offsets.getAt(iChanInChip).valid &&
                    ( fabs( offsets.getAt(iChanInChip).value - offsets.mean() )
                      > StandardDefects::TR_OFFSET.getParameter() * sqrt(offsets.var() ) ) ) {
                result->getDefects().addDefect(Defect(StandardDefects::TR_OFFSET,
                                               ModuleElement::Channel(ichip*nChannelChip+iChanInChip)));
            }
            // Untrimmable Defects:
            if (result->chipTrim[ichip]->channelTrim.getAt(iChanInChip).valid==false) {
                result->getDefects().addDefect(Defect(StandardDefects::TR_NOTRIM, 
		                               ModuleElement::Channel(ichip*nChannelChip + iChanInChip)));
            } // if valid
        }// Channel loop
    }// chip loop

    //Check for defects - fail if more than 15 masked channels or more than 7 consecutive masked channels
    //Problem if 1 masked channel
    const DefectList::DefectCollection& defects = result->getDefects().getAllDefects();
    bool passed = true;
    unsigned int totalChannels = 0;
    for (DefectList::DefectCollection::const_iterator i=defects.begin(); i!=defects.end(); ++i) {
	if (i->getPrototype().getSeverity() >= UNUSEABLE) {
	    totalChannels += i->getModuleElement().getNChannels();
	    if (i->getModuleElement().getNChannels()>7) {
		passed = false;
		break;
	    }
	}
    }
    
    if (totalChannels > 15) passed = false;
    if (totalChannels > 0) result->setProblem(true);
    
    result->setPassed(passed);
}


/* fit the trim data with a line */
void TrimRangeAlgorithm::doTrimDataFit(TrimRangeTestResult::ChipTrimData& chipData) throw() {
    for (unsigned iChanInChip=0; iChanInChip<nChannelChip; ++iChanInChip) {
        Stat<TrimRangeTestResult::TrimData>& trimdata = chipData.channelData.modifyAt(iChanInChip);

        trimdata.valid=false;
        trimdata.value.p0 = 0;
        trimdata.value.p1 = 0;

        if (trimdata.value.graph.empty())
            continue;

        /** linear least-squares fit */
        float xbar=0, ybar=0, x2bar=0, xybar=0;
        const unsigned n=trimdata.value.graph.size();

        for (unsigned ipoint = 0; ipoint<n; ++ipoint) {
            const float& x=trimdata.value.graph[ipoint].first;
            const float& y=trimdata.value.graph[ipoint].second;
            xbar  += x;
            ybar  += y;
            x2bar += x*x;
            xybar += x*y;
        }

        const float det = n*x2bar - xbar*xbar;
        if (det>0) {
            trimdata.value.p0 = (x2bar*ybar - xbar*xybar) / det;
            trimdata.value.p1 = (n*xybar - xbar*ybar) / det;
            trimdata.valid=trimdata.value.p1>0;
        }
	
    }
}

/* interpreet the incoming data as a series of scans over different ranges */
std::vector<TrimRangeAlgorithm::TrimRange> TrimRangeAlgorithm::createRanges(const TestResult& result) throw() {
    // Check all chips in each scan have the same range.
    // if they don't, then analyse all ranges together.
    std::vector<TrimRangeAlgorithm::TrimRange> all_ranges;

    bool inputRangeVariesChipToChip=false;

    for (unsigned iscan=0; iscan<result.getNScans(); ++iscan) {
        shared_ptr<const FitScanResult> fitted = getFit(iscan);
        for (unsigned ichip=1; ichip<nChipModule; ichip++) {
            if (fitted->getConfiguration().getChipConfiguration(ichip).getTrimRange()
                    != fitted->getConfiguration().getChipConfiguration(0).getTrimRange()) {
                inputRangeVariesChipToChip=true;
            }
        }
    }

    for (unsigned iscan=0; iscan<result.getNScans(); ++iscan) {
        shared_ptr<const FitScanResult> fitted = getFit(iscan);

        // find the trim range for module 0
        int range_value=fitted->getConfiguration().getChipConfiguration(0).getTrimRange();
        // find the trim for channel 0
        int trim_value=fitted->getConfiguration().getChipConfiguration(0).getTrim(0);

        std::vector<TrimRange>::iterator it = all_ranges.begin();
        while (it!=all_ranges.end()) {
            if ( (*it).range == range_value ) {
                break;
            }
            ++it;
        }
        if (it==all_ranges.end() ) { // new trim range?
            // if there is chip to chip range variation, treat it all as one range
            if ( !inputRangeVariesChipToChip || all_ranges.empty() ) {
                all_ranges.push_back(TrimRange(range_value));
            }
            it=all_ranges.end()-1;
        }

        // store in that range the trim value and the scan identifier.
        (*it).points.push_back(TrimPoint(trim_value, iscan) );
    }
    // sort by range value.
    sort(all_ranges.begin(), all_ranges.end());
    return all_ranges;
}


shared_ptr<const TrimRangeTestResult::ChipTrim> TrimRangeAlgorithm::getChipTrim(const TrimRangeTestResult::ChipTrimData& data, const float target, const short unsigned range) throw() {

    shared_ptr<SctData::TrimRangeTestResult::ChipTrim> chipTrim (new SctData::TrimRangeTestResult::ChipTrim() );
    chipTrim->target=target;
    chipTrim->range =range;

    for (unsigned iChanInChip=0; iChanInChip<nChannelChip; ++iChanInChip) {
	const Stat<TrimRangeTestResult::TrimData>& d= data.channelData.getAt(iChanInChip);
	Stat<TrimRangeTestResult::Trim>& t= chipTrim->channelTrim.modifyAt(iChanInChip);
	
	if (!d.valid) {
	    t.valid=false;
	    continue;
	}
	
	// record the trim value.
	int i_trim=d.value.getTrim(target);
	
	// check if trimmable - if not, then do your best
	// If trim is -1 or 16, we allow it (because going to the range above wont improve matters!)
	// But we set the trim to the maximum or minimum as appropriate
	if ( i_trim>=0 && i_trim<=15) {
	    t.valid=true;
	    t.value.trim = i_trim;
	} else if (i_trim==-1) {
	    t.valid=true;
	    t.value.trim = 0;
	    if (debug) {
		cout << "Wanted trim of -1, setting to 0 and passing" << endl;
	    }
	} else if (i_trim==16) {
    	    t.valid=true;
	    t.value.trim = 15;
	    if (debug) {
		cout << "Wanted trim of 16, setting to 15 and passing" << endl;
	    }
	} else {
            t.valid=false;
	    if (debug && target>85. && target<95) {
		cout << "Couldn't trim channel: " << iChanInChip << " i_trim: " << i_trim << " for target: " << target << endl;
		cout << "Vt50 min: " << d.value.getVthr(0) << "Vt50 max: " << d.value.getVthr(15) << endl;
	    }
            if (i_trim<0)
                t.value.trim = 0;
            if (i_trim>15)
                t.value.trim = 15;
        }
	
	//Don't use i_trim - we want the actual threshold that will be set in case i_trim<0 or >15
        t.value.vthr = d.value.getVthr(t.value.trim);
    }
    return chipTrim;
}


} // end of namespace SctAnalysis
