#include "TimeWalkAlgorithm.h"
#include "AnalysisAlgorithmMap.h"
#include "SctData/TimeWalkTestResult.h"
#include "SctData/StandardDefects.h"
#include "SctData/FitObject.h"
#include "SctData/FitScanResult.h"
#include <cfloat>
// gcc296 needs a declaration of fabs
#include <cmath>

using namespace std;
using namespace boost;
using namespace SctData;
using namespace Sct;

namespace SctAnalysis {
    
    bool TimeWalkAlgorithm::inMap = AnalysisAlgorithmMap::instance().setAlgorithm("TimeWalkTest", auto_ptr<AnalysisAlgorithm>(new TimeWalkAlgorithm()));
    
    shared_ptr<AnalysisAlgorithm> TimeWalkAlgorithm::clone(const TestData& testData, const string& moduleName) const throw() {
	return shared_ptr<AnalysisAlgorithm>(new TimeWalkAlgorithm(testData, moduleName, *this));
    }

    void TimeWalkAlgorithm::loadData() {
	loadAllFits();
    }

    
    bool TimeWalkAlgorithm::canAnalyze() const{
	return hasAllFits();
    }
    
    shared_ptr<TestResult> TimeWalkAlgorithm::createTestResult() const {
	return shared_ptr<TimeWalkTestResult> (new TimeWalkTestResult());
    }
    
    void TimeWalkAlgorithm::analyze() {
	bool debug=false;

	shared_ptr<TimeWalkTestResult> result =  dynamic_pointer_cast<TimeWalkTestResult> ( getTestResult() );
	result->setScanVariable(getFit(0)->getHeader().getVariable());	

	float min_time[nChipModule];
	float max_time[nChipModule];
	for (unsigned ichip=0; ichip<nChipModule; ++ichip){
	    min_time[ichip]=FLT_MAX;  // large positive number
	    max_time[ichip]=-FLT_MAX; // large negative number
	}

	for (unsigned index=0; index<result->getNScans() ; ++index){
	    shared_ptr<const FitScanResult> fit=getFit(index);

	    if (!fit.get()) {
		ostringstream os;
		os << "TimeWalkAlgorithm: unable to get scan at index " << index <<ends;
		throw IllegalStateError(os.str(), __FILE__, __LINE__);
	    }
	    
	    if (debug) cout << "TimeWalkAlgorithm, scan #" << index << endl;

	    for (unsigned ichip=0; ichip<nChipModule; ++ichip){
		TimeWalkTestResult::ChipTWResult& cr = result->getChipResult(ichip);
		FitObject& fobj =fit->getChipFit(ichip);
		
		// on last point calculate calibration factor.
		if (debug) cout << index << "," 
				<< result->getNScans()-1 << endl;
		if (index==result->getNScans()-1){
		    cr.calibration = // mid-rise - mid-fall
			fabs(fobj.getParameter(1) - fobj.getParameter(3));
		    if (debug) {
			cout << "Calibration for chip "
			     << ichip << " = " << cr.calibration<< endl;
		    }
		    
		}
		if (fobj.getParameter(3) < min_time[ichip]){
		    min_time[ichip]=fobj.getParameter(3);
		}
		if (fobj.getParameter(3) > max_time[ichip]){
		    max_time[ichip]=fobj.getParameter(3);
		}
	    }
	}
	
	bool pass = true;

	for (unsigned ichip=0; ichip<nChipModule; ++ichip){
	    TimeWalkTestResult::ChipTWResult& cr = result->getChipResult(ichip);
	    cr.timewalk=max_time[ichip]-min_time[ichip];
	    if (debug) {
		cout << "Time-walk for chip " << ichip
		     << " = " <<  cr.timewalk << endl;
	    }
	    if (cr.timewalk<StandardDefects::TW_LO.getParameter()){
		result->getDefects().addDefect(Defect(StandardDefects::TW_LO, ModuleElement::Chip(ichip)));
		pass = false;
	    }
	    if (cr.timewalk>StandardDefects::TW_HI.getParameter()){
		result->getDefects().addDefect(Defect(StandardDefects::TW_HI, ModuleElement::Chip(ichip)));
		pass = false;
	    }
	}

	result->setPassed(pass);
	if (debug) cout << "TimeWalkAlgorithm done" << endl;
    }
} // end of namespace SctAnalysis
