#ifndef ANALYSISWORKERGROUP_H
#define ANALYSISWORKERGROUP_H
#include "SctService/WorkerGroup.h"
#include <string>
#include <list>
#include <iostream>

namespace Sct {
class IOName;
}
class TestData;

using namespace SctService;
using std::string;
using std::list;
using boost::shared_ptr;

namespace SctAnalysis{

    class AnalysisAlgorithm;
    

    /**
       Control structure for a group of AnalysisWorker thread objects.
       Does the book-keeping of where to funnel incoming ScanResult objects.
     */
    class AnalysisWorkerGroup : public WorkerGroup< shared_ptr<Sct::IOName> >{
    public:
	class TestAlgs;	
	
	/**
	   Add a new TestData to the list of tests
	   @param testdata the test to add.
	   @note one must lock the internals whilst doing the adding
	*/
	void addTest(shared_ptr<const TestData> testdata);
	/**
	   Remove a TestData object, and all older TestData's.
	   @param testdata the TestData to remove.
	 */
	void removeTestsUpTo(shared_ptr<const TestData> testdata);

	/**
	   Look for a TestAlgs which matches a given run number and scan number
	   @param runno the run number
	   @param scanno the scan number
	   @return TestAlgs for this test if these exist.
	   @note one must lock the internals whilst doing the finding.
	*/
	shared_ptr<TestAlgs> findTest(const unsigned long runno, const unsigned long scanno) const throw();
	/**
	   Look for a TestAlgs which matches a given run number and scan number
	   @param testdata the TestData we want to find.
	   @return TestAlgs for this test if these exist.
	   @note one must lock the internals whilst doing the finding.
	*/
	shared_ptr<TestAlgs> findTest(const TestData& testdata)const throw() ;

	/**
	   Prints the internal contents of its known tests to an output stream, and returns it.
	 */
	std::ostream& printStatus(std::ostream& os) const throw();
	
	/**
	   Call this to free all memory, delete references to Tests in progress,
	   and clear the queue.
	   @note Don't do this while analysis is in progress!
	*/
	void purge() throw();
	
	/**
	   @todo put description of work() here.
	*/
	virtual void work(shared_ptr<Sct::IOName> name) throw();
	
/*--------internal class-----------------------------------------------------------*/

	/** keeps a test and its results together. Associates these with a lockable mutex.*/
	class TestAlgs {
	public:
	    /** constructor takes a Test object. Also works out what sort of test it is and  */
	    TestAlgs(shared_ptr<const TestData> testdata) : m_testdata(testdata) {} 
	    /** Get the TestData object */
	    const TestData& getTest() const {return *m_testdata;}
	    /** Get the AnalysisAlgorithm for a particular module name.
		@note makes a new Algorithm of the appropriate type if one is not found.
		@return the appropriate AnalysisAlgorithm,
		@note one must lock the internals while searching.
	    */ 
	    shared_ptr<AnalysisAlgorithm> findAlgorithm(const string& modulename) ;
	    /**
	       Add a test result for a new module name.
	       @param modulename the module name for which to add.
	       @return a shared pointer to the created AnalysisAlgorithm for this module.
	       @note one must lock the internals whilst doing the adding.
	     */
	    shared_ptr<AnalysisAlgorithm> addAlgorithm(const string& modulename);
	    /**
	       Removes an algorithm from the map.
	       @param alg A shared pointer to the algorithm to remove.
	    */
	    void removeAlgorithm(shared_ptr<AnalysisAlgorithm> alg);
	    /**
	       Destructor gets the lock before destroying the internals.
	     */
	    ~TestAlgs();
	    /**
	       Print the contents of this testalgs
	    */
	    std::ostream& printStatus(std::ostream& os) const throw();
	private:
	    shared_ptr<const TestData> m_testdata;               ///< the TestData itself.
	    list< shared_ptr<AnalysisAlgorithm> > m_algorithms;  ///< the AnalysisAlgorithms for each of the modules in the test.
	    mutable boost::recursive_mutex m_access;                 ///< lock results
	};
	
    private:    
	list<shared_ptr<TestAlgs> > m_tests; ///< List of the tests, and associated analysis algorithms.
	mutable boost::recursive_mutex m_tests_access; ///< access to the list of algorithms
    };
    

    
}// end of namespace SctAnalysis

#endif // #define ANALYSISWORKERGROUP_H
