#ifndef ANALYSISSERVICE_H
#define ANALYSISSERVICE_H
#include <is/isinfo.h>
#include <ipc/object.h>
#include <ipc/server.h>

#include <string>
#include <boost/utility.hpp>

#include "SctFitter/FitStrategy.h"
#include "AnalysisWorkerGroup.h"
#include "AnalysisServiceI.hh"


namespace SctAnalysis {

class AnalysisService : public IPCObject, public virtual AnalysisServiceI_T_AnalysisServiceInterface, boost::noncopyable {
public:
    virtual ~AnalysisService() {}
    /** RUNS the IPC server and IS callback. Does not return. */
    void run();
    /** what to do when a new TestData is found, or removed */
    static void testDataCallback(ISCallbackInfo * isc);
    /** what to do when a new FitScanResult or RawScanResult is found */
    static void scanResultCallback(ISCallbackInfo * isc);
    /** set the FitStrategy by name */
    void setFitStrategy(const string& name) throw(LogicError);
    /** return a FitStrategy which tells how to do the fitting. */ 
    SctFitter::FitStrategy& getFitStrategy() const throw(LogicError);
    /** singleton access method */
    static AnalysisService& instance();
    
    /**
      Initializes the static instance with a FitStrategy.  The default
      FitStrategy is RootFitStrategy.  Will throw ConfigurationException if something goes wrong.
      */
    static AnalysisService& initialize(const string& fitStrategyName = "RootFitStrategy");
    
    /**
       Calls AnalysisWorkerGroup::purge() method to free stuck memory.
       @note Don't even think of doing this while analysis is in progress!
    */
    virtual void purge(AnalysisServiceIStatus* status) throw();
    /**  
	 Analyse all data relevant to a test.
	 @param testname the TestData object IO name
    */
    virtual void analyze(AnalysisServiceIStatus* status, char* testname) throw();
    /**  
	 Analyse all data relevant to a test for a particular module.
	 @param testname the TestData object IO name
	 @param modulename the name of the module to be analyzed
    */
    virtual void analyzeModule(AnalysisServiceIStatus* status, char* testname, 
				    char* modulename) throw();
    /**
       Output the status of the AnalysisService over IPC.
    */
    virtual char* status(AnalysisServiceIStatus* status) throw();
    /** ipc busy method */
    virtual ilu_ShortInteger busy (AnalysisServiceIStatus *_status);
    /** ipc queueLength method */
    virtual ilu_ShortInteger queueLength (AnalysisServiceIStatus *_status);

    static IPCServer& getServer() throw();
    
protected:
    /** AnalysisWorkerGroup does the internal book-keeping, and calls the appropriate AnalysisAlgorithm */
    AnalysisWorkerGroup* workergroup;
private:
    boost::recursive_mutex m_status_access ; ///< protects status from different threads.
    SctFitter::FitStrategy* fitStrategy; ///< cache telling how to do fitting.
    std::auto_ptr<ISInfoReceiver> infoReceiver;     ///<IS InfoReceiver for IS events
    static AnalysisService* service;
    AnalysisService(const string& fitStrategyName ); ///< singleton.  Will throw ConfigurationException if something goes wrong.
};
}

#endif //#ifndef ANALYSISSERVICE_H
