#ifndef ANALYSISALGORITHM_H
#define ANALYSISALGORITHM_H

#include <string>
#include <vector>
#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>
#include "Sct/IoExceptions.h"
#include "Sct/LogicErrors.h"

namespace SctData {
class TestResult;
class RawScanResult;
class FitScanResult;
}
namespace Sct {
class IOName;
}

class TestData;

using std::string;
using std::vector;
using boost::shared_ptr;

namespace SctAnalysis {
  
/**
  An abstract base class for all the analysis algorithms.
  @author Matthew Palmer
  @date 11 June 2003  
  */
class AnalysisAlgorithm {
public:
    /**
       Destructor.
      */
    virtual ~AnalysisAlgorithm() throw();
    /**
       get a pointer to the cached TestData;
    */
    const TestData& getTestData() const throw() {return m_test;}

    /**
      Get the name of the module which this algorithm is working on.
    */
    const std::string getModuleName() const throw() {return m_modulename;}

    /**
       Must be overridden by sub-classes. Returns a shared pointer to a concrete AnalysisAlgorithm
       of the correct type. Primarily used by the map to get an AnalysisAlgorithm of the correct type.
    */
    virtual boost::shared_ptr<AnalysisAlgorithm> clone(const TestData& testData, const string& moduleName) const throw() =0;
    /**
       Add a named FitScanResult
       @param name the IS name of the FitScanResult.
       @throw LogicError if name does not belong in this algorithm
    */
    void addFitScanResult(shared_ptr<Sct::IOName> name);
    /**
       Add a named RawScanResult
       @param name the IS name of the RawScanResult.
       @note nothrow
    */
    void addRawScanResult(shared_ptr<Sct::IOName> name);
    /**
       Used by other methods to ensure that a TestResult has been made.
       Call createTestResult if it doesn't then initializes it.
       @throws nothing
    */
    void checkTestResultExists();
    /**
       Method indicates when a particular analysis is finished.
    */
    bool isDone() const throw() {return m_done;}
    
    /**
       Publishes the TestResult, and sets `done'.
       @throws IOException if there is some IO problem
    */
    void finish();
    
    /**
       Access to the TestResult if required.
     */
    boost::shared_ptr<SctData::TestResult> getTestResult() const{return m_result;}
    
    /**
      Access the mutex
      */
    boost::recursive_mutex& getMutex() const {return m_access;}
    
    /**
      Should check to see if the analysis can be performed.  Probably wants to call hasAllRaws() or hasAllFits().
      @throws no exceotions.
      */
    virtual bool canAnalyze() const = 0;
    
    /**
      Does the analysis.  Should only be called after canAnalyze() returns true.
      Should not be called is isDone() returns true.
      It should also do any final initialization of the TestResult which requires data to be present
      @throw any Exception
      */
    virtual void analyze() = 0;
    
    /**
      Called to indicate the algorithm should try and load as much data as it can - 
      perhaps by calling loadAllFits or loadAllRaws().  This is guarenteed to be called
      shortly before analyze(). 
      */
    virtual void loadData() = 0;
    
    ///@{ @name Status and timing methods 
    ///@note These methods are external rather than internal so the calling method can include its
    ///overheads etc

    /**
       Adds a pass for this test.
     */
    void addFailure();
    /**
       Adds a failure for this test.
    */
    void addPass();
    /**
      Adds some time spent doing I/O.  The number of IO operations is assumed to be
      the same as the number of analyses
      @note Nothrow
      */
    void addIOTime(double ioTime);
    
    /**
      Adds to the analysis time, and increments the number of analyses successfully done
      @note Nothrow
      */
    void addAnalysisTime(double time);
    
    /**
      Adds to the overhead time.  It is assumed there is 1 overhead time per Fit or Raw ScanResult
      @note Nothrow
      */
    void addOverheadTime(double time);
    
    /**
      Returns a string giving the status of this algorithm including some info how long things are taking
      */
    string getStatus() const;
    ///@}
    
protected:
    /**
       Default constructor, to be used in creation of map.
     */
    AnalysisAlgorithm(const TestData& testData, const string& moduleName, const AnalysisAlgorithm& prototype) throw();
    
    /**
      Called if the TestResult needs to be created.
      @throws nothing.
      */
    virtual shared_ptr<SctData::TestResult> createTestResult() const = 0;
    /**
       Loads all the fits it can.
       @throw Sct::IoException if there is an IO problem
     */
    void loadAllFits();
    /**
       Loads all the raws it can.
       @throw Sct::IoException if there is an IO problem
     */
    void loadAllRaws();
    
    /**
      Returns true if all the RawScanResult objects are available in IS.
      @note nothrow
      */
    bool hasAllRaws() const;
    /**
      Returns true if all the FitScanResult objects are available in IS.
      @throws nothing
      */
    bool hasAllFits() const;
    
    /**
      Returns a shared pointer to the RawScanResult at index.  Guarenteed not to run a null pointer
      @throws InvalidStateError if the raw is not available
      */
    shared_ptr<SctData::RawScanResult> getRaw(unsigned int index) const;
    
    /**
      Returns a shared pointer to the RawScanResult at index.  Guarenteed not to run a null pointer
      @throws InvalidStateError if the raw is not available
      */
    shared_ptr<SctData::FitScanResult> getFit(unsigned int index) const;

    
    /**
      Merges all the defects from the FitScanResults into the TestResult.  Utility method for algorithms.
      Ignores FitScanResults that don't exist.
      Does nothing if the TestResult doesn't exist.
      */
    void mergeDefects();
    
    AnalysisAlgorithm() ;

private:
    /**
      Useful function that initializes TestResult from Test and module name.
      Sets up stuff that can be done here.  Anything else must be done by the algorithm in analyze.
      @note TestResult must have been created first!
    */
    void initializeTestResult();
    
    class ScanData {
    public:
	ScanData();
	bool rawAvailable;
	bool fitAvailable;
	string rawName;
	string fitName;
        shared_ptr<SctData::RawScanResult> raw;
        shared_ptr<SctData::FitScanResult> fit;
    };

    const string m_modulename;              ///< name of the module being worked on.
    shared_ptr<SctData::TestResult> m_result;  ///< pointer to the result. @note it is up to the Concrete class to make one!
    const TestData& m_test;      ///< pointer to the testdata.
    vector<ScanData> scanData;          ///a vector of ScanData objects
    bool m_done;                      ///< true if finish() has been called.
    mutable boost::recursive_mutex m_access;      ///< access to the algorithm can be locked for thread-safeness.
    
    //Timing stuff
    mutable boost::recursive_mutex m_status_access;      ///< access to the status information
    const AnalysisAlgorithm& prototype;		    ///< A reference to the prototype
    mutable double ioTime;				    ///< The total time spent doing I/O
    mutable double analysisTime;			    ///< The total time spent doing analysis
    mutable double overheadTime;			    ///< The time spent just getting stuff, checking it etc
    mutable unsigned int nCalls;			    ///< The number of times this alg has been accessed
    mutable unsigned int nAnalyses;			    ///< The number of analyses performed
    mutable unsigned int nPassed;                            ///< The number of tests which passed
    mutable unsigned int nFailed;                           ///< The number of analyses which failed
};

inline void AnalysisAlgorithm::addPass(){
  boost::recursive_mutex::scoped_lock lock(m_status_access);
  ++prototype.nPassed;
}

inline void AnalysisAlgorithm::addFailure(){
  boost::recursive_mutex::scoped_lock lock(m_status_access);
    ++prototype.nFailed;
}

inline void AnalysisAlgorithm::addIOTime(double ioTime) {
  boost::recursive_mutex::scoped_lock lock(m_status_access);
    prototype.ioTime += ioTime;
}
    
inline void AnalysisAlgorithm::addAnalysisTime(double time) {
  boost::recursive_mutex::scoped_lock lock(m_status_access);
    prototype.analysisTime += time;
    ++prototype.nAnalyses;
}
    
inline void AnalysisAlgorithm::addOverheadTime(double time) {
  boost::recursive_mutex::scoped_lock lock(m_status_access);
    prototype.overheadTime += time;
    ++prototype.nCalls;
}

}

#endif //#ifndef ANALYSISALGORITHM_H
