
// sketch for ATLAS Upgrade Interlock II

// Bart Hommels
// University of Cambridge

// 16/7/2012
// V 0.1: basic hybrid NTC readout
// v 0.3: hybrid, user ntc readout, set menu complete.
// v 0.4: 31/08/2012 reduced memory footprint
// v 0.5: 01/09/2012 various improvements
// v 0.6: 04/09 major memory footprint reduction using F() for LCD character strings
// w/ full sketch, freeMemory() returns 1428 B
// v 0.7: 11/09 added menu watchdog timer, sensirion SHT71 on I2C functionality: readout, interlock, configuration
// freememory returns 957B

//+-------------------------------------------------------+
// Libraries
#include <stdio.h>
#include "Wire.h"// for driving SPI bus and/or I2C/onewire
#include "LiquidCrystal.h"// for LCD on SPI
#include <EEPROM.h>// for using EEPROM
#include "EEPROMAnything.h" // for writing multiple-byte objects to EEPROM
#include <NTCreadlean.h>
#include "SHTxx.h"

using namespace myardu;

//----------------------------------------------------------------
// Memory Usage Monitoring +-------------------------------------+
extern unsigned int __heap_start;
extern void *__brkval;
/*
 * The free list structure as maintained by the 
 * avr-libc memory allocation routines.
 */
struct __freelist {
  size_t sz;
  struct __freelist *nx;
};
/* The head of the free list structure */
extern struct __freelist *__flp;
//---------------------------------------------------------------+

// +-------------------------------------------------------+
// Definitions
const char fwVersion[5] = "0.71";
const char fwDate[11] = "11/09/2012";
// +-------------------------------------------------------+
// pin assignments
const byte muxNTC = A0;
const byte userNTC09pin = A1;
const byte userNTC08pin = A2;
const byte menuPot = A3;
const byte i2cData = A4;
const byte i2cClk = A5;

const byte d0_output = 0;
const byte d1_output = 1;
const byte muxCtrl2 = 2;
const byte menuButton = 3;
const byte lcdLatch = 4;
const byte lcdClk = 5;
const byte lcdData = 6;
const byte duoLed = 7;
const byte outputEnable = 8;
const byte selectRseries = 9;
const byte muxCtrl0 = 10;
const byte muxCtrl1 = 11;
const byte enablePbLED = 12;
const byte pbEnable = 13;

//+-------------------------------------------------------+
// variables

// High, low temperature limits for hybrid, user NTCs, user RH/T sensor.
// values here are extremely conservative fallbacks. actual settings should be collected from EEPROM
float hybridTHi = 30.0;
float hybridTLo = 10.0;
float userNTCTHi = 30.0;
float userNTCTLo = 10.0;
float userShtTHi = 30.0;
float userShtTLo = 10.0;
float userShtRHHi = 60.0;
float userShtRHLo = 30.0;

byte hybridNTCRnom = 10;// hybrid NTC resistance at nominal T, in kOhm. Should be 1 or 10 (default)
const byte TrefHybrid = 25;
const unsigned int BntcHybrid = 3435;

// user NTC defaults (use GL23 for the moment: R(20 deg C) = 2kOhm, B=312)
byte userNTCRnom = 2;// user NTC resistance at nominal T in kOhm: [1,2,5,10,12,15,20]
const byte userNTCRseries = 10;// kOhm. fixed, but adjustable, in hardware
byte userNTCTnom = 20;// should be 20 or 25
unsigned int BUserNTC = 3125;// adjustable 3000-4000 K in 5 K steps

// EEPROM addresses for non volatile parameters
const unsigned int addrHybridTLo = 0;// float: 4B
const unsigned int addrHybridTHi = 4;// float: 4B
const unsigned int addrHybridEnableMask = 8;// byte
const unsigned int addrHybridNTCRnom = 9;// byte

const unsigned int addrUserNTCTLo = 20;// float: 4B
const unsigned int addrUserNTCTHi = 24;// float: 4B
const unsigned int addrUserNTCEnableMask = 28;// byte
const unsigned int addrUserNTCRnom = 29;// byte
const unsigned int addrUserNTCTnom = 30;// byte
const unsigned int addrUserNTCBntc = 31;// int: 2B

const unsigned int addrShtTLo = 50;// float: 4B
const unsigned int addrShtTHi = 54;// float: 4B
const unsigned int addrShtRHLo = 58;// float: 4B
const unsigned int addrShtRHHi = 62;// float: 4B
const unsigned int addrShtEnable = 66;// float: 4B

const byte nSamples = 2;
// enable masks for sensors
byte nHybridNTCs = 1;
byte nUserNTCs = 0;
byte shtEnable = 0;

char dispTemp[5];
// menu pot limits and values
const int potClickLo = 470;
const int potClickHi = 560;

const char degC[] = { 
  (char)223, 'C'};
const char kOhm[] = { 
  'k', (char)244 };

// readout interval for SHT71. 
//Device should not be active more than 10% to limit self-heating to 0.1degC
// if Fclk = 80 kHz, readout sequence takes 302ms.
const unsigned int userShtInterval = 3500UL;//3.5s
// pushbutton flash interval
const unsigned int flashInterval = 500UL;// 0.5s
// output over serial interval
const unsigned int serialInterval = 10000UL;// 10s
// menu timeout
const unsigned int maxMenuInterval = 60000UL;// 1 minute

// menu state machine variables
byte buttonRead = 0;
byte prevButtonRead = 0;
byte menuState = 0;
byte nextMenuState = 0;
//byte prevMenuState = 0;
byte menuChoice = 0;
byte menuBranch = 0;
byte eepromWritten = 0;

int setTemp = 0;
float mapTemp = 0.0;
byte setEnable = 0;

byte enableLedState = LOW;
byte outputState = LOW;
byte userEnable = LOW;
// timing control of refresh rates
unsigned long serialTimerLast, flashTimerLast, userShtTime, menuWatchDog;


//+-------------------------------------------------------+
// instantiations

// SPI LCD interface through 74LS595 shift register, see:
// http://www.ladyada.net/products/i2cspilcdbackpack/index.html
// DATA (p14 on 74LS595), CLK (p11 on 74LS595), LATCH (p12 on 74LS595)
LiquidCrystal lcd(6,5,4);

// NTC serving 8x MUXed input
NTCtemp hybridNTC;
byte hybridTnotOK = 0x00;
// NTC serving user08, user09
NTCtemp userNTC08;
NTCtemp userNTC09;
byte userTnotOK = 0x00;
// Sensirion SHT71 on I2C port
SHTxx shtRHT(i2cData, i2cClk);
byte userShtNotOK = 0x00;

// temperature variables
float hybridTemp[8];// keeping this fixed length is lots faster
float userTemp08 = 20.0;
float userTemp09 = 20.0;
float shtTemp = 20.0;
float shtRHum = 50.0;

//+-------------------------------------------------------+
void setup(){
  Serial.begin(9600);
  // set mode for all remaining pins
  pinMode(muxCtrl0, OUTPUT);
  pinMode(muxCtrl1, OUTPUT);
  pinMode(muxCtrl2, OUTPUT);
  pinMode(menuButton, INPUT);
  pinMode(duoLed, OUTPUT);
  pinMode(outputEnable, OUTPUT);
  pinMode(selectRseries, OUTPUT);
  pinMode(d0_output, OUTPUT);
  pinMode(d1_output, OUTPUT);
  pinMode(enablePbLED, OUTPUT);
  pinMode(pbEnable, INPUT);

  // write default outputs to pins
  digitalWrite(muxCtrl0, LOW);
  digitalWrite(muxCtrl1, LOW);
  digitalWrite(muxCtrl2, LOW);

  digitalWrite(selectRseries, HIGH);
  // use external analog reference
  analogReference(EXTERNAL);

  // set variables
  menuState = 0;
  nextMenuState = 0;
  menuBranch = 0;
  eepromWritten = 0;

  enableLedState = LOW;
  outputState = LOW;

  serialTimerLast = millis();
  flashTimerLast = millis();
  userShtTime = millis();
  menuWatchDog = millis();

  // retrieve variables from EEPROM
  // Hybrid NTCs
  EEPROM_readAnything(addrHybridTLo, hybridTLo);
  EEPROM_readAnything(addrHybridTHi, hybridTHi);
  EEPROM_readAnything(addrHybridEnableMask, nHybridNTCs);
  EEPROM_readAnything(addrHybridNTCRnom, hybridNTCRnom);
  // User NTCs
  EEPROM_readAnything(addrUserNTCTLo, userNTCTLo);
  EEPROM_readAnything(addrUserNTCTHi, userNTCTHi);
  EEPROM_readAnything(addrUserNTCEnableMask, nUserNTCs);
  EEPROM_readAnything(addrUserNTCRnom, userNTCRnom);
  EEPROM_readAnything(addrUserNTCTnom, userNTCTnom);
  EEPROM_readAnything(addrUserNTCBntc, BUserNTC);
  // SHT 71 RH/T
  EEPROM_readAnything(addrShtTLo, userShtTLo);
  EEPROM_readAnything(addrShtTHi, userShtTHi);
  EEPROM_readAnything(addrShtRHLo, userShtRHLo);
  EEPROM_readAnything(addrShtRHHi, userShtRHHi);
  EEPROM_readAnything(addrShtEnable, shtEnable);

  // initialise NTC instances
  hybridNTC.setNTCup(muxNTC, nSamples, hybridNTCRnom);
  hybridNTC.setNTCParameters(hybridNTCRnom, TrefHybrid, BntcHybrid);

  userNTC08.setNTCup(userNTC08pin, nSamples, userNTCRseries);
  userNTC08.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);

  userNTC09.setNTCup(userNTC09pin, nSamples, userNTCRseries);
  userNTC09.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);

  // setup lcd + print welcome message
  lcd.begin(40,2);
  lcd.setCursor(10,0);
  lcd.print(F("Temperature Interlock"));// 21 chars 
  lcd.setBacklight(HIGH);
  lcd.setCursor(6,1);
  lcd.print(F("firmware: "));// 10 chars
  lcd.setCursor(16,1);
  lcd.print(fwVersion);
  lcd.setCursor(20,1);
  lcd.print(' ');
  lcd.setCursor(21,1);
  lcd.print(fwDate);
  delay(2000);
  lcd.clear();
}


void loop(){
  // readout loop for hybrid NTCs
  // set series R according to hybrid type. 
  // For 1 kOhm, drive selectRseries line low during mux'ed measurement only to avoid excessive self-heating when idle
  if(hybridNTCRnom == 1) digitalWrite(selectRseries, LOW);

  for(byte count=0; count < 8; count++){
    if(count < nHybridNTCs){
      digitalWrite(muxCtrl0, (bitRead(count,0) ) );
      digitalWrite(muxCtrl1, (bitRead(count,1) ) );
      digitalWrite(muxCtrl2, (bitRead(count,2) ) );

      // read out temperature
      hybridTemp[count] = hybridNTC.getDegC();

      if(hybridTemp[count] > hybridTHi || hybridTemp[count] < hybridTLo){ 
        bitWrite(hybridTnotOK, count, HIGH);
      }
      else{
        bitWrite(hybridTnotOK, count, LOW);
      }
    }
    else{
      bitWrite(hybridTnotOK, count, LOW);
    }
  }
  digitalWrite(selectRseries, HIGH);

  // read out user NTCs if enabled
  // check temperature readings against limits and set bits in userTnotOK accordingly
  if(nUserNTCs > 0){
    userTemp08 = userNTC08.getDegC();
    if(userTemp08 > userNTCTHi || userTemp08 < userNTCTLo) bitWrite(userTnotOK, 0, HIGH);
    else bitWrite(userTnotOK, 0, LOW);

    if(nUserNTCs > 1){ 
      userTemp09 = userNTC09.getDegC();
      if(userTemp09 > userNTCTHi || userTemp09 < userNTCTLo) bitWrite(userTnotOK, 1, HIGH);
      else bitWrite(userTnotOK, 1, LOW);

    }
  }
  else userTnotOK = 0x00;


  // read out SHT71 on I2C port if enabled. 
  //Keep track of timing interval
  if(shtEnable){// cheap byte comparison first
    if( (millis() - userShtTime) > userShtInterval ){
      shtTemp = shtRHT.getTemp();
      shtRHum = shtRHT.getHum();
      if(shtTemp > userShtTHi || shtTemp < userShtTLo) userShtNotOK = 1;
      else if(shtRHum > userShtRHHi || shtRHum < userShtRHLo) userShtNotOK = 1;
      else userShtNotOK = 0;
      userShtTime = millis();
    }
  }
  else userShtNotOK = 0;

  // read enable pushbutton pin, see if Temp. is OK and set outputState accordingly
  userEnable = digitalRead(pbEnable);

  // check if temperatures are OK, and if there is at least one sensor enabled
  if(hybridTnotOK == 0 && userTnotOK == 0 && userShtNotOK == 0 && ((nHybridNTCs+nUserNTCs+shtEnable) > 0) ){
    digitalWrite(duoLed, HIGH);// T indicator LEDs go green
    if(userEnable == HIGH){// Temperature OK, enable PB pushed
      outputState = HIGH;
    }  
    if(outputState == HIGH) enableLedState = HIGH;
    else if(millis()- flashTimerLast > flashInterval){
      flashTimerLast = millis();
      if(enableLedState == LOW) enableLedState = HIGH;
      else enableLedState = LOW;
    }
  }
  else{// temperature not good, or no sensors enabled
    digitalWrite(duoLed, LOW);
    outputState = LOW;
    enableLedState = LOW;
  }
  digitalWrite(outputEnable, outputState);
  digitalWrite(enablePbLED, enableLedState);
  // 

  // output temperature measurement results on serial every 10s
  if(Serial.available()){
    if(millis()-serialTimerLast > serialInterval){
      serialTimerLast = millis();

      Serial.print("freeMemory():");
      Serial.println( freeMemory() );

      for(byte count = 0; count < 8; count++){
        if(count < nHybridNTCs){ 
          Serial.print(hybridTemp[count],1  );
          Serial.print(',');
        }
        else{
          Serial.print("-99.9,");
        }
      }
      switch(nUserNTCs){
      case 0:
        Serial.print("-99.9,-99.9");
        break;
      case 1:
        Serial.print(userTemp08,1);
        Serial.print(",-99.9");
      case 2:
        Serial.print(userTemp08,1);
        Serial.print(',');
        Serial.print(userTemp09,1);
        break;
      }
      if(shtEnable > 0){
        if(nUserNTCs > 0) Serial.print(',');
        Serial.print(shtTemp,1);
        Serial.print(',');
        Serial.print(shtRHum,1);
      }
      else Serial.print("-99.9,99.9");
      Serial.println();
    }
  }


  // do menu stuff
  int menuPotVal = analogRead(menuPot);// for use in menu
  // read menu button, debounce, increase menustate if necessary
  int buttonRead = digitalRead(menuButton);

  // check button is pressed, advance to next state
  if(buttonRead != prevButtonRead && buttonRead == HIGH){// leading edge
    delay(50);//require button to be pressed for a bit of time & debounce as well
    buttonRead = digitalRead(menuButton);
    if( buttonRead == HIGH ){
      menuState = nextMenuState;
      menuWatchDog = millis();
    }
  }
  prevButtonRead = buttonRead;

  // watchdog timeout: check interval between menu button push actions
  if(menuState){// skip watchdog if not in menu (cheap byte check)
    if(millis()-menuWatchDog > maxMenuInterval){// see if timeout occurred (expensive unsigned long operations)
      menuState = 0;
      nextMenuState = 0;
    }
  }

  // +=+=+=+=+=+=+=+=+ MENU STATE MACHINE +=+=+=+=+=+=+=+=+
  // menu state rewrites entire first line. Avoid clearing LCD as it causes flicker
  lcd.home();
  // print Temperature values to lcd on bottom line
  // default is to print hybrid temperatures. If no hybrid NTCs enabled: print others
  if(nHybridNTCs){
    for(byte count = 0; count < 8; count++){
      lcd.setCursor((count*5),1);
      if(count < nHybridNTCs){
        dtostrf(hybridTemp[count], 5, 1, dispTemp);
        lcd.print(dispTemp);
      }
      else lcd.print(F("     "));
    }
  }
  else{// print others
    lcd.setCursor(0,1);
    if(nUserNTCs > 0){
      lcd.print("U8:");// 3 chars
      lcd.setCursor(3,1);
      dtostrf(userTemp08, 5,1, dispTemp);
      lcd.print(dispTemp);// 5 chars
      lcd.setCursor(8,1);
      lcd.print(degC);// 2 chars
      lcd.setCursor(10,1);
      if(nUserNTCs > 1){
        lcd.print(" U9:");// 4 chars
        lcd.setCursor(14,1);
        dtostrf(userTemp09, 5,1, dispTemp);
        lcd.print(dispTemp);// 5 chars
        lcd.setCursor(19,1);
        lcd.print(degC);// 2 chars
      }
      else lcd.print(F("          "));// 11 chars
    }
    else lcd.print(F("                     "));// 21 chars
    lcd.setCursor(21,1);
    if(shtEnable){
      lcd.print(F(" RH/T:"));// 6 chars
      lcd.setCursor(27,1);
      dtostrf(shtTemp, 5,1, dispTemp);
      lcd.print(dispTemp);// 5 chars
      lcd.setCursor(32,1);
      lcd.print(degC);// 2 chars
      lcd.setCursor(34,1);
      lcd.print(", ");
      lcd.setCursor(36,1);
      dtostrf(shtRHum, 2,0, dispTemp);
      lcd.print(dispTemp);// 2 chars
      lcd.setCursor(38,1);
      lcd.print(" %");
    }
    else lcd.print(F("                   "));// 19 chars
  }


  switch(menuState){
  case 0:
    if(menuPotVal < 100){
      lcd.print(F("T, RH limits SHT71:  "));// 21 chars
      lcd.setCursor(21,0);
      dtostrf(userShtTLo, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(24,0);
      lcd.print(",");
      lcd.setCursor(25,0);
      dtostrf(userShtTHi, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(28,0);
      lcd.print(degC);

      lcd.setCursor(30,0);
      lcd.print(" | ");
      lcd.setCursor(33,0);
      dtostrf(userShtRHLo, 2,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(35,0);
      lcd.print(",");
      lcd.setCursor(36,0);
      dtostrf(userShtRHHi, 2,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(38,0);
      lcd.print(" %");
    }
    else if(menuPotVal > 100 && menuPotVal < potClickLo){ 
      lcd.print(F("T limits H0-H7: "));// 16 chars
      lcd.setCursor(16,0);
      dtostrf(hybridTLo, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(19,0);
      dtostrf(hybridTHi, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(22,0);
      lcd.print(degC);
      lcd.setCursor(24,0);
      lcd.print(F(" U8,U9: "));// 8 chars
      lcd.setCursor(32,0);
      dtostrf(userNTCTLo, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(35,0);
      dtostrf(userNTCTHi, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(38,0);
      lcd.print(degC);
    }
    else if(menuPotVal > potClickHi && menuPotVal < 900){
      //
      if(nUserNTCs == 0 && shtEnable == 0){
        lcd.print(F("User NTC, I2C RH/T disabled             "));
      }
      else{
        if(nUserNTCs > 0){
          lcd.print(F("T U8:"));// 5 chars
          lcd.setCursor(5,0);
          dtostrf(userTemp08, 5,1, dispTemp);
          lcd.print(dispTemp);
          lcd.setCursor(10,0);
          if(nUserNTCs == 1) lcd.print(F("           "));// 11 chars
        }
        if(nUserNTCs > 1){
          lcd.print(F(" T U9:"));// 6 chars
          lcd.setCursor(16,0);
          dtostrf(userTemp09, 5,1, dispTemp);
          lcd.print(dispTemp);
        }
        lcd.setCursor(21,0);
        if(shtEnable > 0){
          lcd.print(F(" T, RH:"));//7 chars
          lcd.setCursor(28,0);
          dtostrf(shtTemp, 5,1, dispTemp);
          lcd.print(dispTemp);
          lcd.setCursor(33,0);
          lcd.print(degC);
          lcd.setCursor(35,0);
          lcd.print(' ');
          lcd.setCursor(36,0);
          dtostrf(shtRHum, 2,0, dispTemp);
          lcd.print(dispTemp);
          lcd.setCursor(38,0);
          lcd.print(" %");
        }
        else lcd.print(F("                   "));// 19 chars
      }
    }
    else if(menuPotVal > 900){
      lcd.print(F("U8,U9 Rnom,T(Rnom), B: "));//23 chars
      lcd.setCursor(23,0);
      if(userNTCRnom < 10){
        lcd.print(' ');
        lcd.setCursor(24,0);
      }
      lcd.print(userNTCRnom, DEC);
      lcd.setCursor(25,0);
      lcd.print(kOhm);// 2 chars
      lcd.setCursor(27,0);
      lcd.print(", ");
      lcd.setCursor(29,0);
      lcd.print(userNTCTnom, DEC);
      lcd.setCursor(31,0);
      lcd.print(degC);
      lcd.setCursor(33,0);
      lcd.print(", ");//2
      lcd.setCursor(35,0);
      lcd.print(BUserNTC);
      lcd.setCursor(39,0);
      lcd.print('K');
    }
    else{
      if(nHybridNTCs) lcd.print(F("|H0  |H1  |H2  |H3  |H4  |H5  |H6  |H7  "));
      else lcd.print(F("Hybrid NTCs disabled                    "));
    }  

    nextMenuState = 1;
    break;
    //    case :
    //    lcd.print("                                        ");
    //    if(menuPotVal < potClickLo){
    //    }
    //    else if(menuPotVal > potClickHi){
    //    }
    //    else{
    //    }
    //    break;
  case 1:
    if(menuPotVal <= 255){
      lcd.print(F("HYBRID_NTC user_ntc  user_rh/t     exit  "));
      menuBranch = 1;
      nextMenuState = 21;
    }
    else if(menuPotVal > 255 && menuPotVal < 512){
      lcd.print(F("hybrid_ntc USER_NTC  user_rh/t     exit  "));
      menuBranch = 2;
      nextMenuState = 25;
    }
    else if(menuPotVal > 512 && menuPotVal < 900){
      lcd.print(F("hybrid_ntc user_ntc  USER_RH/T     exit  "));
      menuBranch = 3;
      nextMenuState = 28;
    }
    else{
      lcd.print(F("hybrid_ntc user_ntc  user_rh/t     EXIT  "));
      menuBranch = 0;
      nextMenuState = 0;
    }
    break;

  case 21:
    lcd.print(F("hybrid NTC :"));//12 chars
    lcd.setCursor(12,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("T LIMITS  N enable  set Type"));//28 chars
      nextMenuState = 31;//31
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("t limits  N enable  SET TYPE"));
      nextMenuState = 32;//32
    }
    else{
      lcd.print(F("t limits  N ENABLE  set Type"));
      nextMenuState = 33;//33
    }
    break;

  case 25:
    lcd.print(F("user NTCs:"));//10 chars
    lcd.setCursor(10,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("T LIMITS  N enable  set values"));//30 chars
      nextMenuState = 31;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("t limits  N enable  SET VALUES"));
      nextMenuState = 36;
    }
    else{
      lcd.print(F("t limits  N ENABLE  set values"));
      nextMenuState = 37;
    }
    break;

  case 28:
    eepromWritten = 0;
    lcd.print(F("SHT71 RH/T: "));//12 chars
    lcd.setCursor(12,0);
    if(menuPotVal < potClickLo){
      if(shtEnable){
        lcd.print(F("DISABLE"));// 7 chars
        menuChoice = 1;
      }
      else{
        lcd.print(F("ENABLE "));
        menuChoice = 2;
      }
      lcd.setCursor(19,0);
      lcd.print(F("  t limits  rh limits"));// 21 chars
      nextMenuState = 38;
    }
    else{  
      if(shtEnable) lcd.print(F("disable"));
      else lcd.print(F("enable "));
      lcd.setCursor(19,0);

      if(menuPotVal > potClickHi){
        lcd.print(F("  t limits  RH LIMITS"));
        nextMenuState = 39;
      }
      else{
        lcd.print(F("  T LIMITS  rh limits"));
        nextMenuState = 31;
      }
      menuChoice = 0;
    }
    break;

  case 31:
    switch(menuBranch){
    case 1: 
      lcd.print(F("H0-H7 NTC T limits: "));//20 chars
      break;
    case 2: 
      lcd.print(F("user NTCs T limits: "));
      break;
    case 3: 
      lcd.print(F("SHT71 RH/T T limits:"));
      break;
    case 0: 
      lcd.print(F("+-+-+   ERROR  +-+-+")); 
      nextMenuState = 0;
      break;
    }
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("LOW     exit    high"));// 20 chars
      nextMenuState = 41;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("low     exit    HIGH"));
      nextMenuState = 41;
      menuChoice = 2;
    }
    else{
      lcd.print(F("low     EXIT    high"));
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 32:
    eepromWritten = 0;
    lcd.print(F("Hybrid NTC R(25"));//15 chars
    lcd.setCursor(15,0);
    lcd.print(degC);
    lcd.setCursor(17,0);
    lcd.print(F("): "));// 4 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print(F(" 10KOHM  exit  1kohm"));
      menuChoice = 1;
      nextMenuState = 42;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F(" 10kohm  exit  1KOHM"));
      menuChoice = 2;
      nextMenuState = 42;
    }
    else{
      lcd.print(F(" 10kohm  EXIT  1kohm"));
      menuChoice = 0;
      nextMenuState = 0;
    }
    break;

  case 33:
    lcd.print(F("hybrid NTC enable (0..8, exit): "));// 32 chars
    setEnable = map(menuPotVal, 0, 1023, 0, 10);

    lcd.setCursor(32,0);
    if(setEnable >= 9){
      lcd.print(F("    EXIT"));
      nextMenuState = 0;
    }
    else{
      for(byte count = 0; count < 8; count++){
        lcd.setCursor((32+count),0);
        if(count < setEnable) lcd.print('O');
        else lcd.print('X');
      }
      nextMenuState = 43;
    }
    break;

    //  case 35:
    //    lcd.print(F("user NTC T limits:  "));//20 chars
    //    lcd.setCursor(20,0);
    //    if(menuPotVal < potClickLo){
    //      lcd.print(F("LOW     exit    high"));// 20 chars
    //      nextMenuState = 45;
    //      menuChoice = 1;
    //    }
    //    else if(menuPotVal > potClickHi){
    //      lcd.print(F("low     exit    HIGH"));
    //      nextMenuState = 45;
    //      menuChoice = 2;
    //    }
    //    else{
    //      lcd.print(F("low     EXIT    high"));
    //      nextMenuState = 0;
    //      menuChoice = 0;
    //    }
    //    break;

  case 36:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set nominal NTC R    "));// 36 chars
    setTemp = map(menuPotVal, 0, 1024, 0, 7);
    switch(setTemp){
    case 0:
      userNTCRnom = 1;
      break;
    case 1:
      userNTCRnom = 2;
      break;
    case 3:
      userNTCRnom = 3;
      break;
    case 4:
      userNTCRnom = 5;
      break;
    case 5:
      userNTCRnom = 10;
      break;
    case 6:
      userNTCRnom = 20;
      break;
    }
    lcd.setCursor(36,0);
    if(userNTCRnom < 10){
      lcd.print(' ');
      lcd.setCursor(37,0);
    }
    lcd.print(userNTCRnom, DEC);
    lcd.setCursor(38,0);
    lcd.print(kOhm);
    nextMenuState = 46;
    break;


  case 37:
    eepromWritten = 0;
    lcd.print(F("user NTC enable (0..2, exit): "));// 30 chars
    lcd.setCursor(30,0);
    if(menuPotVal < 255){
      lcd.print(F("        XX"));
      setEnable = 0;
      nextMenuState = 47;
    }
    else if(menuPotVal > 254 && menuPotVal < potClickHi){
      lcd.print(F("        OX"));
      setEnable = 1;
      nextMenuState = 47;
    }
    else if(menuPotVal > potClickHi && menuPotVal < 800){
      lcd.print(F("        OO"));
      setEnable = 2;
      nextMenuState = 47;
    }
    else{
      lcd.print(F("      EXIT"));
      nextMenuState = 0;
    }
    break;

  case 38:
    if(menuChoice == 2){// previous disabled state should become enabled
      lcd.print(F("SHT71 RH/T sensor is now ENABLED        "));
      shtEnable = 1;
    }
    else{
      lcd.print(F("SHT71 RH/T sensor is now DISABLED       "));
      shtEnable = 0;
    }
    if(eepromWritten == 0){
      EEPROM_writeAnything(addrShtEnable, shtEnable);
      eepromWritten = 1;
    }
    nextMenuState = 1;
    break;

  case 39:
    lcd.print(F("SHT71 RH/T RH limits: "));// 22 chars
    lcd.setCursor(22,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("LOW    exit   high"));// 18 chars
      nextMenuState = 49;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("low    exit   HIGH"));
      nextMenuState = 49;
      menuChoice = 2;
    }
    else{
      lcd.print(F("low    EXIT   high"));
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 41:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set Temperature:  "));//35 chars
    setTemp = map(menuPotVal, 0, 1023, -25, 65);
    lcd.setCursor(35,0);
    dtostrf(setTemp, 3, 0, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(38,0);
    lcd.print(degC);
    eepromWritten = 0;
    nextMenuState = 51;
    break;

  case 42:
    lcd.print(F("Hybrid NTC R(25"));//15 chars 
    lcd.setCursor(15,0);
    lcd.print(degC);
    lcd.setCursor(17,0);
    lcd.print(F("):                "));//17 chars, reaching up to 35
    lcd.setCursor(35,0);
    if(menuChoice == 1){
      hybridNTCRnom = 10;
      lcd.print(F("10 "));
    }
    else if(menuChoice == 2){
      lcd.print(F(" 1 "));
      hybridNTCRnom = 1;
    }
    lcd.setCursor(38,0);
    lcd.print(kOhm);

    if(eepromWritten == 0){
      EEPROM_writeAnything(addrHybridNTCRnom, hybridNTCRnom);
      eepromWritten == 1;
    }
    // re-initialise NTC instances
    hybridNTC.setNTCup(muxNTC, nSamples, hybridNTCRnom);
    hybridNTC.setNTCParameters(hybridNTCRnom, TrefHybrid, BntcHybrid);
    nextMenuState = 32;
    break;

  case 43:
    nHybridNTCs = setEnable;
    if(eepromWritten == 0){
      EEPROM_writeAnything(addrHybridEnableMask, setEnable);
      eepromWritten == 1;
    }
    lcd.print(F("hybrid NTC enable mask set to:  "));// 32 chars
    for(byte count = 0; count < 8; count++){
      lcd.setCursor((32+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 33;
    break;

  case 45: 
    eepromWritten = 0;
    lcd.print(F("Dial & push to set Temperature:    "));//35 chars
    setTemp = map(menuPotVal, 0, 1024, -25.0, 75.0);
    lcd.setCursor(35,0);
    dtostrf(setTemp, 3, 0, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(38,0);
    lcd.print(degC);
    nextMenuState = 55;
    break;

  case 46:
    lcd.print(F("Nominal user NTC R set to: "));//27 chars 
    lcd.setCursor(27,0);
    if(userNTCRnom < 10){
      lcd.print(' ');
      lcd.setCursor(28,0);
    }
    lcd.print(userNTCRnom, DEC);
    lcd.setCursor(29,0);
    lcd.print(kOhm);
    lcd.setCursor(31,0);
    lcd.print(F("         "));
    if(eepromWritten == 0){
      EEPROM_writeAnything(addrUserNTCRnom, userNTCRnom);
      eepromWritten == 1;
    }
    nextMenuState = 56;
    break;

  case 47:
    nUserNTCs = setEnable;
    if(eepromWritten == 0){
      EEPROM_writeAnything(addrUserNTCEnableMask, setEnable);
      eepromWritten == 1;
    }
    lcd.print(F("User NTC enable mask set to:          "));// 38 chars
    for(byte count = 0; count < 2; count++){
      lcd.setCursor((38+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 37;
    break;

  case 49:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set Rel. Humidity:   "));//36 chars
    setTemp = map(menuPotVal, 0, 1023, 5, 95);
    lcd.setCursor(36,0);
    dtostrf(setTemp, 2, 0, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(38,0);
    lcd.print(" %");
    nextMenuState = 59;
    break;

  case 51:
    float compTemp;
    unsigned int addrTemp;
    if(menuChoice == 1){// low T limit
      switch(menuBranch){
      case 1:
        compTemp = hybridTHi;
        addrTemp = addrHybridTLo;
        break;
      case 2:
        compTemp = userNTCTHi;
        addrTemp = addrUserNTCTLo;
        break;
      case 3:
        compTemp = userShtTHi;
        addrTemp = addrShtTLo;
        break;
      }

      if(float(setTemp) > compTemp){
        lcd.print(F("Temperature not set: Low > High limit   "));
      }
      else{
        //hybridTLo = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrTemp, float(setTemp) );
          eepromWritten == 1;
        }
        lcd.print(F("Low Temperature limit set to:      "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else if(menuChoice == 2){
      switch(menuBranch){
      case 1:
        compTemp = hybridTLo;
        addrTemp = addrHybridTHi;
        break;
      case 2:
        compTemp = userNTCTLo;
        addrTemp = addrUserNTCTHi;
        break;
      case 3:
        compTemp = userShtTLo;
        addrTemp = addrShtTHi;
        break;
      }
      if(float(setTemp) < compTemp){
        lcd.print(F("Temperature not set: High < Low limit   "));
      }
      else{
        hybridTHi = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrTemp, float(setTemp));
          eepromWritten == 1;
        }
        lcd.print(F("High Temperature limit set to:     "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else{
      lcd.print(F("Error: temperature limit not set        "));
    }
    if(eepromWritten == 0){
      switch(menuBranch){
      case 1:
        EEPROM_readAnything(addrHybridTLo, hybridTLo);
        EEPROM_readAnything(addrHybridTHi, hybridTHi);
        break;
      case 2:
        EEPROM_readAnything(addrUserNTCTLo, userNTCTLo);
        EEPROM_readAnything(addrUserNTCTHi, userNTCTHi);
        break;
      case 3:
        EEPROM_readAnything(addrShtTLo, userShtTLo);
        EEPROM_readAnything(addrShtTHi, userShtTHi);
        break;
      }
      eepromWritten == 1;
    }
    nextMenuState = 31;
    break;

  case 55:
    if(menuChoice == 1){// low T limit
      if(float(setTemp) > userNTCTHi){
        lcd.print(F("Temperature not set: Low > High limit   "));
      }
      else{
        userNTCTLo = (float)setTemp;
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrUserNTCTLo, userNTCTLo);
          eepromWritten == 1;
        }
        lcd.print(F("Low Temperature limit set to:      "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else if(menuChoice == 2){
      if(float(setTemp) < userNTCTLo){
        lcd.print(F("Temperature not set: High < Low limit   "));
      }
      else{
        userNTCTHi = (float)setTemp;
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrUserNTCTHi, userNTCTHi);
          eepromWritten == 1;
        }
        lcd.print(F("High Temperature limit set to:     "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else{
      lcd.print(F("Error: temperature limit not set        "));
    }
    nextMenuState = 35;
    break;

  case 56:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set NTC T(Rnom):     "));// 36 chars
    lcd.setCursor(36,0);
    if(menuPotVal < potClickHi){
      lcd.print(F("20"));
      userNTCTnom = 20;
    }
    else{
      lcd.print(F("25"));// deg C");
      userNTCTnom = 25;
    }
    lcd.setCursor(38,0);
    lcd.print(degC);
    nextMenuState = 66;
    break;

  case 59:
    if(menuChoice == 1){// low RH limit
      if(float(setTemp) > userShtRHHi){
        lcd.print(F("Rel. Humidity not set: Low > High limit "));
      }
      else{
        userShtRHLo = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrShtRHLo, userShtRHLo);
          eepromWritten == 1;
        }
        lcd.print(F("Low Relative Humidity limit set to: "));// 36 chars
      }
    }
    else if(menuChoice == 2){// high RH limit
      if(float(setTemp) < userShtRHLo){
        lcd.print(F("Rel. Humidity not set: High < Low limit "));
      }
      else{
        userShtRHHi = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrShtRHHi, userShtRHHi);
          eepromWritten == 1;
        }
        lcd.print(F("High Relative Humidity limit set to:"));// 36 chars
      }
    }
    lcd.setCursor(36,0);
    lcd.print(setTemp);
    lcd.setCursor(38,0);
    lcd.print(" %");
    nextMenuState = 39;
    break;


  case 66:
    lcd.print(F("NTC Temperature (Rnominal) set to:  "));//36 chars
    lcd.setCursor(36,0);
    lcd.print(userNTCTnom);
    lcd.setCursor(38,0);
    lcd.print(degC);
    if(eepromWritten == 0){
      EEPROM_writeAnything(addrUserNTCTnom, userNTCTnom);
      eepromWritten == 1;
    }
    nextMenuState = 76;
    break;

  case 76:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set NTC B value:    "));//34 chars
    //BUserNTC = map(menuPotVal, 0, 1024, 3100, 3500);
    BUserNTC = 5*(map(menuPotVal, 0, 1024, 600, 800));// BUserNTC: 3000-4000 steps of 5
    lcd.setCursor(34,0);
    lcd.print(BUserNTC); 
    lcd.setCursor(38,0);
    lcd.print(" K");  
    nextMenuState = 86;
    break;

  case 86:
    lcd.print(F("NTC 08, 09 B value set to:        "));// 34 chars
    lcd.setCursor(34,0);
    lcd.print(BUserNTC);
    lcd.setCursor(38,0);
    lcd.print(" K");
    if(eepromWritten == 0){
      EEPROM_writeAnything(addrUserNTCBntc, BUserNTC);
      userNTC08.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);
      userNTC09.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);
      eepromWritten == 1;
    }
    nextMenuState = 1;
  }

}// loop()

// FUNCTIONS

// clear lcd line
void lcdClearLine(int lineNum){
  if(lineNum == 0 || lineNum == 1){
    lcd.setCursor(0,lineNum);
    lcd.print(F("                                        "));
  }
  return;
}


/* Calculates the size of the free list */
int freeListSize() {
  struct __freelist* current;
  int total = 0;

  for (current = __flp; current; current = current->nx) {
    total += 2; /* Add two bytes for the memory block's header  */
    total += (int) current->sz;
  }

  return total;
}

int freeMemory() {
  int free_memory;

  if ((int)__brkval == 0) {
    free_memory = ((int)&free_memory) - ((int)&__heap_start);
  } 
  else {
    free_memory = ((int)&free_memory) - ((int)__brkval);
    free_memory += freeListSize();
  }
  return free_memory;
}

















































































