// sketch for ATLAS Upgrade Interlock II

// Bart Hommels
// University of Cambridge

// 16/7/2012
// V 0.1

//+-------------------------------------------------------+
// Libraries
//#include <stdlib.h>
#include "Wire.h"// for driving SPI bus and/or I2C/onewire
#include "LiquidCrystal.h"// for LCD on SPI
#include <EEPROM.h>// for using EEPROM
#include "EEPROMAnything.h" // for writing multiple-byte objects to EEPROM
#include <NTCreadout.h>

// +-------------------------------------------------------+
// Definitions
#define DEBUG_TO_SERIAL 0 // debug information to serial port

// +-------------------------------------------------------+
// pin assignments
const int muxNTC = A0;
const int userNTC09 = A1;
const int userNTC08 = A2;
const int menuPot = A3;

const int d0_output = 0;
const int d1_output = 1;
const int muxCtrl0 = 10;//0
const int muxCtrl1 = 11;//1
const int muxCtrl2 = 2;
const int menuButton = 3;
const int lcdLatch = 4;
const int lcdClk = 5;
const int lcdData = 6;
const int duoLed = 7;

const int outputEnable = 8;
const int selectRseries = 9;
//const int d0_output = 10;
//const int d1_output = 11;
const int singleLed = 12;
const int pbEnable = 13;

//+-------------------------------------------------------+
// constants
const float absTzero = -273.15;// 0 Kelvin in deg C

// 10kOhm hybrid NTCs
const float RseriesHybrid = 10000.0;
const float RrefHybrid = 10000.0;
const float TrefHybrid = 25.0-absTzero;
const float BntcHybrid = 3435.0;

// GL23
//const float RseriesNTC09 = 10000.0;
//const float RrefNTC09 = 2000.0;
//const float TrefNTC09 = 20.0-absTzero;
//const float BntcNTC09 = 3125.0;

//+-------------------------------------------------------+
// variables
float hybridTHi = 30.0;// deg C. Extremely conservative fallback, should be collected from EEPROM
float hybridTLo = 10.0;//deg C. Extremely conservative fallback, should be collected from EEPROM

unsigned int addrHybridTLo = 0;
unsigned int addrHybridTHi = 4;
unsigned int addrHybridEnableMask = 5;

int nSamples = 2;
byte nHybrids = 1;

char dispTemp[5];
// menu pot limits and values
const int potClickLo = 470;
const int potClickHi = 560;

int lastMenuPotVal = 0;
//int saveMenuPotVal = 0;

// menu state machine variables
byte buttonRead = 0;
byte prevButtonRead = 0;
byte menuState = 0;
byte nextMenuState = 0;
byte prevMenuState = 0;
byte menuChoice = 0;

float setTemp = 0.0;
byte setEnable = 0;

//+-------------------------------------------------------+
// instantiations

// SPI LCD interface through 74LS595 shift register, see:
// http://www.ladyada.net/products/i2cspilcdbackpack/index.html
// DATA (p14 on 74LS595), CLK (p11 on 74LS595), LATCH (p12 on 74LS595)
LiquidCrystal lcd(6,5,4);

// NTC serving MUXed input, user08, user09
NTCtemp hybridNTC;
byte hybridTnotOK = 0x00;
//NTCtemp NTC08;`
//NTCtemp NTC09;

//+-------------------------------------------------------+
void setup(){
#if DEBUG_TO_SERIAL// initialise serial
  Serial.begin(9600);
#endif //DEBUG_TO_SERIAL

  // set mode for all remaining pins
  pinMode(muxCtrl0, OUTPUT);
  pinMode(muxCtrl1, OUTPUT);
  pinMode(muxCtrl2, OUTPUT);
  pinMode(menuButton, INPUT);
  pinMode(duoLed, OUTPUT);
  pinMode(outputEnable, OUTPUT);
  pinMode(selectRseries, OUTPUT);
  pinMode(d0_output, OUTPUT);
  pinMode(d1_output, OUTPUT);
  pinMode(singleLed, OUTPUT);
    pinMode(pbEnable, INPUT);

  // write default outputs to pins
  digitalWrite(muxCtrl0, LOW);
  digitalWrite(muxCtrl1, LOW);
  digitalWrite(muxCtrl2, LOW);

digitalWrite(selectRseries, HIGH);
  // use external analog reference
  //analogReference(EXTERNAL);

  // set variables
  menuState = 0;
  nextMenuState = 0;

  // retrieve variables from EEPROM
  EEPROM_readAnything(addrHybridTLo, hybridTLo);
  EEPROM_readAnything(addrHybridTHi, hybridTHi);
  EEPROM_readAnything(addrHybridEnableMask, nHybrids);

  // initialise NTC instances
  hybridNTC.setNTCup(muxNTC, nSamples, RseriesHybrid);
  hybridNTC.setNTCParameters(RrefHybrid, TrefHybrid, BntcHybrid);

  //  NTC09.setNTC(userNTC09, nSamples, RseriesNTC09);
  //  NTC.setNTCParameters(RrefNTC09, TrefNTC09, BntcNTC09);

  // setup lcd + print welcome message
  lcd.begin(40,2);
  lcd.setCursor(10,0);
  lcd.print("Temperature Interlock");// 21 chars 
  lcd.setBacklight(HIGH);
  lcd.setCursor(6,1);
  lcd.print("firmware v0.1 dd 20/07/2012");// 27 chars
  delay(2000);
  lcd.clear();
}


void loop(){
  float hybridTemp[8];// keeping this fixed length is lots faster
  byte count = 0;

  //  unsigned long timerlast = millis();

  for(count=0; count < 8; count++){
    if(count < nHybrids){
      digitalWrite(muxCtrl0, (bitRead(count,0) ) );
      digitalWrite(muxCtrl1, (bitRead(count,1) ) );
      digitalWrite(muxCtrl2, (bitRead(count,2) ) );
      hybridTemp[count] = hybridNTC.getDegC();

      if(hybridTemp[count] > hybridTHi || hybridTemp[count] < hybridTLo){ 
        bitWrite(hybridTnotOK, count, HIGH);
      }
      else{
        bitWrite(hybridTnotOK, count, LOW);
      }
    }
    else{
      bitWrite(hybridTnotOK, count, LOW);
    }

  }
#if DEBUG_TO_SERIAL
  Serial.print("Hybrid T OK bitmap: ");
  Serial.println(hybridTnotOK, BIN);
#endif // DEBUG TO SERIAL

  //  unsigned long t_diff = millis()-timerlast;

  if(hybridTnotOK == 0){
    digitalWrite(duoLed, HIGH);
  }
  else{
    digitalWrite(duoLed, LOW);
  }


  // print timing information
  //lcd.setCursor(38,0);
  //lcd.print(t_diff);

  // do menu stuff
  int menuPotVal = analogRead(menuPot);// for use in menu
  // read menu button, debounce, increase menustate if necessary
  int buttonRead = digitalRead(menuButton);

  // check button is pressed, advance to next state
  if(buttonRead != prevButtonRead && buttonRead == HIGH){// leading edge
    delay(50);//require button to be pressed 50ms (= debounce as well)
    if( buttonRead == HIGH ){
      //saveMenuPotVal = menuPotVal;//??
      prevMenuState = menuState; 
      menuState = nextMenuState;

#if DEBUG_TO_SERIAL
      Serial.print("Menu Choice: ");
      Serial.println(menuChoice, DEC);
      Serial.print("Menu State: ");
      Serial.println(menuState, DEC);
      Serial.print("Next Menu State: ");
      Serial.println(nextMenuState, DEC);
      Serial.println("+---+");
#endif // DEBUG TO SERIAL

    }
  }
  prevButtonRead = buttonRead;

  // +=+=+=+=+=+=+=+=+ MENU STATE MACHINE +=+=+=+=+=+=+=+=+
  // menu state must rewrite entire first line. Avoid clearing LCD as it causes flicker
  //lcdClearLine(0);
  lcd.home();
  switch(menuState){
  case 0:
    if(menuPotVal < potClickLo){ 
      lcd.print("H0-H7 High, Low Temp. Limits:");// 29 chars
      lcd.setCursor(29,0);
      dtostrf(hybridTLo, 5,1, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(34,0);
      lcd.print(" ");
      lcd.setCursor(35,0);
      dtostrf(hybridTHi, 5,1, dispTemp);
      lcd.print(dispTemp);

    }
    else if(menuPotVal > potClickHi){
      lcd.print("User NTC, 1Wire RH/T disabled           ");
    }
    else{
      lcd.print("|H0  |H1  |H2  |H3  |H4  |H5  |H6  |H7 |");
    }  
    // print Temperature values to lcd on bottom line
    for(count = 0; count < 8; count++){
      lcd.setCursor((count*5),1);
      if(count < nHybrids){
        dtostrf(hybridTemp[count], 5, 1, dispTemp);
        lcd.print(dispTemp);
      }
      else lcd.print("     ");
    }
    nextMenuState = 1;
    break;
    //    case :
    //    lcd.print("                                        ");
    //    if(menuPotVal < potClickLo){
    //    }
    //    else if(menuPotVal > potClickHi){
    //    }
    //    else{
    //    }
    //    break;
  case 1:
    if(menuPotVal < 255){
      lcd.print("HYBRID_NTC user_ntc  user_rh/t     exit  ");
      nextMenuState = 21;
    }
    else if(menuPotVal > 255 || menuPotVal < 512){
      lcd.print("hybrid_ntc USER_NTC  user_rh/t     exit  ");
      nextMenuState = 0;
    }
    else if(menuPotVal > 512 || menuPotVal < 768){
      lcd.print("hybrid_ntc user_ntc  USER_RH/T     exit  ");
      nextMenuState = 0;
    }
    else{
      lcd.print("hybrid_ntc user_ntc  user_rh/t     EXIT  ");
      nextMenuState = 0;
    }
    break;
  case 21:
    lcd.print("hybrid NTC :");//12 chars
    lcd.setCursor(12,0);
    if(menuPotVal < potClickLo){
      lcd.print("T LIMITS  N enable  set Type");//28 chars
      nextMenuState = 31;//31
    }
    else if(menuPotVal > potClickHi){
      lcd.print("t limits  N enable  SET TYPE");
      nextMenuState = 32;//32
    }
    else{
      lcd.print("t limits  N ENABLE  set Type");
      nextMenuState = 33;//33
    }
    break;
  case 31:
    lcd.print("hybrid NTC T limits:");//20 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print("LOW     exit    high");// 20 chars
      nextMenuState = 41;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print("low     exit    HIGH");
      nextMenuState = 41;
      menuChoice = 2;
    }
    else{
      lcd.print("low     EXIT    high");
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;
  case 33:
    lcd.print("hybrid NTC enable (0..8, exit): ");// 32 chars
    setEnable = map(menuPotVal, 0, 1024, 0, 10);
#if DEBUG_TO_SERIAL
    Serial.print("Set Enable: ");
    Serial.println(setEnable, DEC);
#endif // DEBUG TO SERIAL

    lcd.setCursor(32,0);
    if(setEnable >= 9){
      lcd.print("    EXIT");
      nextMenuState = 0;
    }
    else{
      for(count = 0; count < 8; count++){
        lcd.setCursor((32+count),0);
        if(count < setEnable) lcd.print('0');
        else lcd.print('X');
      }
      nextMenuState = 43;
    }
    break;
  case 41:
    lcd.print("Push to set Temperature: ");//27 chars
    setTemp = map(menuPotVal, 0, 1024, -25.0, 65.0);

    lcd.setCursor(27,0);
    dtostrf(setTemp, 5,1, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(32,0);
    lcd.print("        ");
    nextMenuState = 51;
    break;
  case 43:
    nHybrids = setEnable;
    EEPROM_writeAnything(addrHybridEnableMask, setEnable);
    lcd.print("hybrid NTC enable mask set to:  ");// 32 chars
    for(count = 0; count < 8; count++){
      lcd.setCursor((32+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 33;
    break;
  case 51:
    if(menuChoice == 1){// low T limit
      hybridTLo = setTemp;
      EEPROM_writeAnything(addrHybridTLo, setTemp);
      lcd.print("Low Temperature limit set to:      ");
      lcd.setCursor(35,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else if(menuChoice == 2){
      hybridTHi = setTemp;
      EEPROM_writeAnything(addrHybridTHi, setTemp);
      lcd.print("High Temperature limit set to:     ");
      lcd.setCursor(35,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else{
      lcd.print("Error: temperature limit not set        ");
    }
    nextMenuState = 31;
    break;
  }


}

// FUNCTIONS

// clear lcd line
void lcdClearLine(int lineNum){
  if(lineNum == 0 || lineNum == 1){
    lcd.setCursor(0,lineNum);
    lcd.print("                                        ");
  }
  return;
}



































