
// sketch for ATLAS Upgrade Interlock II

// Bart Hommels
// University of Cambridge

// 16/7/2012
// V 0.1: basic hybrid NTC readout
// v 0.3: hybrid, user ntc readout, set menu complete.
// v 0.4: 31/08/2012 reduced memory footprint
// v 0.5: 01/09/2012 various improvements
// v 0.6: 04/09 major memory footprint reduction using F() for LCD character strings
// w/ full sketch, freeMemory() returns 1428 B
// v 0.7: 11/09 added menu watchdog timer, sensirion SHT71 on I2C functionality: readout, interlock, configuration
// freememory returns 957B
// v 0.8: menu branching added to reduce code footprint. Cleanup & variable name rationalisation. 

//+-------------------------------------------------------+
// Libraries
#include <stdio.h>
#include "Wire.h"// for driving SPI bus and/or I2C/onewire
#include "LiquidCrystal.h"// for LCD on SPI
#include <EEPROM.h>// for using EEPROM
#include "EEPROMAnything.h" // for writing multiple-byte objects to EEPROM
#include <NTCreadlean.h>
#include "SHTxx.h"

using namespace myardu;

//----------------------------------------------------------------
// Memory Usage Monitoring +-------------------------------------+
extern unsigned int __heap_start;
extern void *__brkval;
/*
   * The free list structure as maintained by the 
 * avr-libc memory allocation routines.
 */
struct __freelist {
  size_t sz;
  struct __freelist *nx;
};
/* The head of the free list structure */
extern struct __freelist *__flp;
//---------------------------------------------------------------+

// +-------------------------------------------------------+
// Definitions7
const char fwVersion[5] = "0.81";
const char fwDate[11] = "17/09/2012";
// +-------------------------------------------------------+
// pin assignments
const byte hNTCMuxPin = A0;
const byte uNTC09Pin = A1;
const byte uNTC08Pin = A2;
const byte menuPot = A3;
const byte i2cData = A4;
const byte i2cClk = A5;

const byte d0_output = 0;
const byte d1_output = 1;
const byte muxCtrl2 = 2;
const byte menuButton = 3;
const byte lcdLatch = 4;
const byte lcdClk = 5;
const byte lcdData = 6;
const byte duoLed = 7;
const byte outputEnable = 8;
const byte selectRseries = 9;
const byte muxCtrl0 = 10;
const byte muxCtrl1 = 11;
const byte enablePbLED = 12;
const byte pbEnable = 13;

//+-------------------------------------------------------+
// variables

// High, low temperature limits for hybrid, user NTCs, user RH/T sensor.
// values here are extremely conservative fallbacks. actual settings should be collected from EEPROM
float h_NTC_T_Hi = 30.0;
float h_NTC_T_Lo = 10.0;
float u_NTC_T_Hi = 30.0;
float u_NTC_T_Lo = 10.0;
float SHT_T_Hi = 30.0;
float SHT_T_Lo = 10.0;
float SHT_RH_Hi = 60.0;
float SHT_RH_Lo = 30.0;

byte h_NTC_Rnom = 10;// hybrid NTC resistance at nominal T, in kOhm. Should be 1 or 10 (default)
byte h_NTC_Tnom = 25;
unsigned int h_NTC_B = 3435;

// user NTC defaults (use GL23 for the moment: R(20 deg C) = 2kOhm, B=312)
byte u_NTC_Rnom = 2;// user NTC resistance at nominal T in kOhm: [1,2,5,10,12,15,20]
const byte u_NTC_Rseries = 10;// kOhm. fixed, but adjustable, in hardware
byte u_NTC_Tnom = 20;// should be 20 or 25
unsigned int u_NTC_B = 3125;// adjustable 3000-4000 K in 5 K steps

// EEPROM addresses for non volatile parameters
const unsigned int addr_h_NTC_T_Lo = 0;// float: 4B
const unsigned int addr_h_NTC_T_Hi = 4;// float: 4B
const unsigned int addr_h_NTC_N_enable = 8;// byte
const unsigned int addr_h_NTC_Rnom = 9;// byte
const unsigned int addr_h_NTC_Tnom = 10;// byte
const unsigned int addr_h_NTC_B = 11;// u_int: 2B

const unsigned int addr_u_NTC_T_Lo = 20;// float: 4B
const unsigned int addr_u_NTC_T_Hi = 24;// float: 4B
const unsigned int addr_u_NTC_N_enable = 28;// byte
const unsigned int addr_u_NTC_Rnom = 29;// byte
const unsigned int addr_u_NTC_Tnom = 30;// byte
const unsigned int addr_u_NTC_B = 31;// u_int: 2B

const unsigned int addr_Sht_T_Lo = 50;// float: 4B
const unsigned int addr_Sht_T_Hi = 54;// float: 4B
const unsigned int addr_Sht_RH_Lo = 58;// float: 4B
const unsigned int addr_Sht_RH_Hi = 62;// float: 4B
const unsigned int addr_Sht_Enable = 66;// float: 4B

const byte nSamples = 2;
// enable masks for sensors
byte h_NTC_N_enable = 1;
byte u_NTC_N_enable = 0;
byte sht_enable = 0;

char dispTemp[5];
// menu pot limits and values
const int menuPotLeftmost = 70;
const int potClickLo = 470;
const int potClickHi = 560;
const int menuPotRightmost = 950; 

const char degC[] = { 
  (char)223, 'C'};
const char kOhm[] = { 
  'k', (char)244 };

// readout interval for SHT71. 
//Device should not be active more than 10% to limit self-heating to 0.1degC
// if Fclk = 80 kHz, readout sequence takes 302ms.
const unsigned int userShtInterval = 3500UL;//3.5s
// pushbutton flash interval
const unsigned int flashInterval = 500UL;// 0.5s
// output over serial interval
const unsigned int serialInterval = 10000UL;// 10s
// menu timeout
const unsigned int maxMenuInterval = 120000UL;// 2 minutes

// menu state machine variables
byte menuButtonRead = 0;
byte prevMenuButtonRead = 0;
byte menuState = 0;
byte nextMenuState = 0;
byte menuChoice = 0;
byte menuBranch = 0;
byte eepromWritten = 0;
int prevMenuPotVal = 0;
byte refreshMenuButton = 0;
byte refreshMenuPot = 0;

int setTemp = 0;
float mapTemp = 0.0;
byte setEnable = 0;

byte enableLedState = LOW;
byte outputState = LOW;
byte userEnable = LOW;
// timing control of refresh rates
unsigned long serialTimerLast, flashTimerLast, userShtTime, menuWatchDog;


//+-------------------------------------------------------+
// instantiations

// SPI LCD interface through 74LS595 shift register, see:
// http://www.ladyada.net/products/i2cspilcdbackpack/index.html
// DATA (p14 on 74LS595), CLK (p11 on 74LS595), LATCH (p12 on 74LS595)
LiquidCrystal lcd(6,5,4);

// NTC serving 8x MUXed input
NTCtemp h_NTC_Mux;
byte h_T_notOK = 0x00;
// NTC serving user08, user09
NTCtemp u_NTC_8;
NTCtemp u_NTC_9;
byte u_T_notOK = 0x00;
// Sensirion SHT71 on I2C port
SHTxx shtRHT(i2cData, i2cClk);
byte sht_RHT_notOK = 0x00;

// temperature variables
float h_NTC_T[8];// keeping this fixed length is lots faster
float u_NTC_8_T = 20.0;
float u_NTC_9_T = 20.0;
float sht_T = 20.0;
float sht_RH = 50.0;

//+-------------------------------------------------------+
void setup(){
  Serial.begin(9600);
  // set mode for all remaining pins
  pinMode(muxCtrl0, OUTPUT);
  pinMode(muxCtrl1, OUTPUT);
  pinMode(muxCtrl2, OUTPUT);
  pinMode(menuButton, INPUT);
  pinMode(duoLed, OUTPUT);
  pinMode(outputEnable, OUTPUT);
  pinMode(selectRseries, OUTPUT);
  pinMode(d0_output, OUTPUT);
  pinMode(d1_output, OUTPUT);
  pinMode(enablePbLED, OUTPUT);
  pinMode(pbEnable, INPUT);

  // write default outputs to pins
  digitalWrite(muxCtrl0, LOW);
  digitalWrite(muxCtrl1, LOW);
  digitalWrite(muxCtrl2, LOW);

  digitalWrite(selectRseries, HIGH);
  // use external analog reference
  analogReference(EXTERNAL);

  // set variables
  menuState = 0;
  nextMenuState = 0;
  menuBranch = 0;
  eepromWritten = 0;

  enableLedState = LOW;
  outputState = LOW;

  serialTimerLast = millis();
  flashTimerLast = millis();
  userShtTime = millis();
  menuWatchDog = millis();

  // retrieve variables from EEPROM
  // Hybrid NTCs
  EEPROM_readAnything(addr_h_NTC_T_Lo, h_NTC_T_Lo);
  EEPROM_readAnything(addr_h_NTC_T_Hi, h_NTC_T_Hi);
  EEPROM_readAnything(addr_h_NTC_N_enable, h_NTC_N_enable);
  EEPROM_readAnything(addr_h_NTC_Rnom, h_NTC_Rnom);
  EEPROM_readAnything(addr_h_NTC_Tnom, h_NTC_Tnom);
  EEPROM_readAnything(addr_h_NTC_B, h_NTC_B);
  // User NTCs
  EEPROM_readAnything(addr_u_NTC_T_Lo, u_NTC_T_Lo);
  EEPROM_readAnything(addr_u_NTC_T_Hi, u_NTC_T_Hi);
  EEPROM_readAnything(addr_u_NTC_N_enable, u_NTC_N_enable);
  EEPROM_readAnything(addr_u_NTC_Rnom, u_NTC_Rnom);
  EEPROM_readAnything(addr_u_NTC_Tnom, u_NTC_Tnom);
  EEPROM_readAnything(addr_u_NTC_B, u_NTC_B);
  // SHT 71 RH/T
  EEPROM_readAnything(addr_Sht_T_Lo, SHT_T_Lo);
  EEPROM_readAnything(addr_Sht_T_Hi, SHT_T_Hi);
  EEPROM_readAnything(addr_Sht_RH_Lo, SHT_RH_Lo);
  EEPROM_readAnything(addr_Sht_RH_Hi, SHT_RH_Hi);
  EEPROM_readAnything(addr_Sht_Enable, sht_enable);

  // initialise NTC instances
  h_NTC_Mux.setNTCup(hNTCMuxPin, nSamples, h_NTC_Rnom);
  h_NTC_Mux.setNTCParameters(h_NTC_Rnom, h_NTC_Tnom, h_NTC_B);

  u_NTC_8.setNTCup(uNTC08Pin, nSamples, u_NTC_Rseries);
  u_NTC_8.setNTCParameters(u_NTC_Rnom, u_NTC_Tnom, u_NTC_B);

  u_NTC_9.setNTCup(uNTC09Pin, nSamples, u_NTC_Rseries);
  u_NTC_9.setNTCParameters(u_NTC_Rnom, u_NTC_Tnom, u_NTC_B);

  // setup lcd + print welcome message
  lcd.begin(40,2);

  lcd.setCursor(10,0); 
  lcd.setBacklight(HIGH);
  lcd.print(F("Temperature Interlock"));// 21 chars 
  lcd.setCursor(6,1);
  lcd.print(F("firmware: "));// 10 chars
  lcd.setCursor(16,1);
  lcd.print(fwVersion);
  lcd.setCursor(20,1);
  lcd.print(' ');
  lcd.setCursor(21,1);
  lcd.print(fwDate);
  delay(500);
  lcd.setBacklight(LOW);
  delay(1500);
  lcd.setBacklight(HIGH);
  lcd.clear();
}


void loop(){
  // readout loop for hybrid NTCs
  // set series R according to hybrid type. 
  // For 1 kOhm, drive selectRseries line low during mux'ed measurement only to avoid excessive self-heating when idle
  if(h_NTC_Rnom == 1) digitalWrite(selectRseries, LOW);

  for(byte count=0; count < 8; count++){
    if(count < h_NTC_N_enable){
      digitalWrite(muxCtrl0, (bitRead(count,0) ) );
      digitalWrite(muxCtrl1, (bitRead(count,1) ) );
      digitalWrite(muxCtrl2, (bitRead(count,2) ) );

      // read out temperature
      h_NTC_T[count] = h_NTC_Mux.getDegC();

      if(h_NTC_T[count] > h_NTC_T_Hi || h_NTC_T[count] < h_NTC_T_Lo){ 
        bitWrite(h_T_notOK, count, HIGH);
      }
      else{
        bitWrite(h_T_notOK, count, LOW);
      }
    }
    else{
      bitWrite(h_T_notOK, count, LOW);
    }
  }
  digitalWrite(selectRseries, HIGH);

  // read out user NTCs if enabled
  // check temperature readings against limits and set bits in u_T_notOK accordingly
  if(u_NTC_N_enable > 0){
    u_NTC_8_T = u_NTC_8.getDegC();
    if(u_NTC_8_T > u_NTC_T_Hi || u_NTC_8_T < u_NTC_T_Lo) bitWrite(u_T_notOK, 0, HIGH);
    else bitWrite(u_T_notOK, 0, LOW);

    if(u_NTC_N_enable > 1){ 
      u_NTC_9_T = u_NTC_9.getDegC();
      if(u_NTC_9_T > u_NTC_T_Hi || u_NTC_9_T < u_NTC_T_Lo) bitWrite(u_T_notOK, 1, HIGH);
      else bitWrite(u_T_notOK, 1, LOW);

    }
  }
  else u_T_notOK = 0x00;


  // read out SHT71 on I2C port if enabled. 
  //Keep track of timing interval
  if(sht_enable){// cheap byte comparison first
    if( (millis() - userShtTime) > userShtInterval ){
      sht_T = shtRHT.getTemp();
      sht_RH = shtRHT.getHum();
      if(sht_T > SHT_T_Hi || sht_T < SHT_T_Lo) sht_RHT_notOK = 1;
      else if(sht_RH > SHT_RH_Hi || sht_RH < SHT_RH_Lo) sht_RHT_notOK = 1;
      else sht_RHT_notOK = 0;
      userShtTime = millis();
    }
  }
  else sht_RHT_notOK = 0;

  // read enable pushbutton pin, see if Temp. is OK and set outputState accordingly
  userEnable = digitalRead(pbEnable);

  // check if temperatures are OK, and if there is at least one sensor enabled
  if(h_T_notOK == 0 && u_T_notOK == 0 && sht_RHT_notOK == 0 && ((h_NTC_N_enable+u_NTC_N_enable+sht_enable) > 0) ){
    digitalWrite(duoLed, HIGH);// T indicator LEDs go green
    if(userEnable == HIGH){// Temperature OK, enable PB pushed
      outputState = HIGH;
    }  
    if(outputState == HIGH) enableLedState = HIGH;
    else if(millis()- flashTimerLast > flashInterval){
      flashTimerLast = millis();
      if(enableLedState == LOW) enableLedState = HIGH;
      else enableLedState = LOW;
    }
  }
  else{// temperature not good, or no sensors enabled
    digitalWrite(duoLed, LOW);
    outputState = LOW;
    enableLedState = LOW;
  }
  digitalWrite(outputEnable, outputState);
  digitalWrite(enablePbLED, enableLedState);
  // 

  // output temperature measurement results on serial every 10s
  if(Serial.available()){
    if(millis()-serialTimerLast > serialInterval){
      serialTimerLast = millis();

      Serial.print("freeMemory():");
      Serial.println( freeMemory() );

      for(byte count = 0; count < 8; count++){
        if(count < h_NTC_N_enable){   
          Serial.print(h_NTC_T[count],1  );
          Serial.print(',');
        }
        else{
          Serial.print("-99.9,");
        }
      }
      switch(u_NTC_N_enable){
      case 0:
        Serial.print("-99.9,-99.9,");
        break;
      case 1:
        Serial.print(u_NTC_8_T,1);
        Serial.print("-99.9,");
      case 2:
        Serial.print(u_NTC_8_T,1);
        Serial.print(',');
        Serial.print(u_NTC_9_T,1);
        Serial.print(',');
        break;
      }
      if(sht_enable > 0){
        Serial.print(sht_T,1);
        Serial.print(',');
        Serial.print(sht_RH,1);
        Serial.print(',');
      }
      else Serial.print("-99.9,99.9,");
      if(outputState) Serial.println("ENABLED");
      else Serial.println("DISABLED");
    }
  }


  // do menu stuff
  int menuPotVal = analogRead(menuPot);// for use in menu
  if(menuPotVal != prevMenuPotVal){// check if pot has changed since last pass
    prevMenuPotVal = menuPotVal;
    refreshMenuPot = 1;
  }
  // read menu button, debounce, increase menustate if necessary
  menuButtonRead = digitalRead(menuButton);


  // check button is pressed, advance to next state
  if(menuButtonRead != prevMenuButtonRead && menuButtonRead == HIGH){// leading edge
    delay(50);//require button to be pressed for a bit of time & debounce as well
    menuButtonRead = digitalRead(menuButton);
    if( menuButtonRead == HIGH ){
      menuState = nextMenuState;
      menuWatchDog = millis();
      refreshMenuButton = 1;
    }
  }
  prevMenuButtonRead = menuButtonRead;

  // watchdog timeout: check interval between menu button push actions
  if(menuState){// skip watchdog if not in menu (cheap byte check)
    if(millis()-menuWatchDog > maxMenuInterval){// see if timeout occurred (expensive unsigned long operations)
      menuState = 0;
      nextMenuState = 0;
    }
  }

  // +=+=+=+=+=+=+=+=+ MENU STATE MACHINE +=+=+=+=+=+=+=+=+
  // menu state rewrites entire first line. Avoid clearing LCD as it causes flicker
  lcd.home();




  // print Temperature values to lcd on bottom line
  // default is to print hybrid temperatures. If no hybrid NTCs enabled: print others
  if(h_NTC_N_enable){
    for(byte count = 0; count < 8; count++){
      lcd.setCursor((count*5),1);
      if(count < h_NTC_N_enable){
        dtostrf(h_NTC_T[count], 5, 1, dispTemp);
        lcd.print(dispTemp);
      }
      else lcd.print(F("     "));
    }
  }

  else{// print others
    lcd.setCursor(0,1);
    if(u_NTC_N_enable > 0){
      lcd.print("U8:");// 3 chars
      lcd.setCursor(3,1);
      dtostrf(u_NTC_8_T, 5,1, dispTemp);
      lcd.print(dispTemp);// 5 chars
      lcd.setCursor(8,1);
      lcd.print(degC);// 2 chars
      lcd.setCursor(10,1);
      if(u_NTC_N_enable > 1){
        lcd.print(" U9:");// 4 chars
        lcd.setCursor(14,1);
        dtostrf(u_NTC_9_T, 5,1, dispTemp);
        lcd.print(dispTemp);// 5 chars
        lcd.setCursor(19,1);
        lcd.print(degC);// 2 chars
      }
      else lcd.print(F("          "));// 11 chars
    }
    else lcd.print(F("                     "));// 21 chars
    lcd.setCursor(21,1);
    if(sht_enable){
      lcd.print(F(" RH/T:"));// 6 chars
      lcd.setCursor(27,1);
      dtostrf(sht_T, 5,1, dispTemp);
      lcd.print(dispTemp);// 5 chars
      lcd.setCursor(32,1);
      lcd.print(degC);// 2 chars
      lcd.setCursor(34,1);
      lcd.print(", ");
      lcd.setCursor(36,1);
      dtostrf(sht_RH, 2,0, dispTemp);
      lcd.print(dispTemp);// 2 chars
      lcd.setCursor(38,1);
      lcd.print(" %");
    }
    else lcd.print(F("                   "));// 19 chars
  }




  switch(menuState){
  case 0:
    if(menuPotVal < menuPotLeftmost){
      lcd.print(F("H0-7 Rnom,T(Rnom), B:  "));//23 chars
      lcd.setCursor(23,0);
      if(h_NTC_Rnom < 10){
        lcd.print(' ');
        lcd.setCursor(24,0);
      }
      lcd.print(h_NTC_Rnom, DEC);
      lcd.setCursor(25,0);
      lcd.print(kOhm);// 2 chars
      lcd.setCursor(27,0);
      lcd.print(", ");
      lcd.setCursor(29,0);
      lcd.print(h_NTC_Tnom, DEC);
      lcd.setCursor(31,0);
      lcd.print(degC);
      lcd.setCursor(33,0);
      lcd.print(", ");//2
      lcd.setCursor(35,0);
      lcd.print(h_NTC_B);
      lcd.setCursor(39,0);
      lcd.print('K');


      //      lcd.print(F("T, RH limits SHT71:  "));// 21 chars
      //      lcd.setCursor(21,0);
      //      dtostrf(SHT_T_Lo, 3,0, dispTemp);
      //      lcd.print(dispTemp);
      //      lcd.setCursor(24,0);
      //      lcd.print(",");
      //      lcd.setCursor(25,0);
      //      dtostrf(SHT_T_Hi, 3,0, dispTemp);
      //      lcd.print(dispTemp);
      //      lcd.setCursor(28,0);
      //      lcd.print(degC);
      //
      //      lcd.setCursor(30,0);
      //      lcd.print(" | ");
      //      lcd.setCursor(33,0);
      //      dtostrf(SHT_RH_Lo, 2,0, dispTemp);
      //      lcd.print(dispTemp);
      //      lcd.setCursor(35,0);
      //      lcd.print(",");
      //      lcd.setCursor(36,0);
      //      dtostrf(SHT_RH_Hi, 2,0, dispTemp);
      //      lcd.print(dispTemp);
      //      lcd.setCursor(38,0);
      //      lcd.print(" %");
    }
    else if(menuPotVal > menuPotLeftmost && menuPotVal < potClickLo){ 
      lcd.print(F("T limits H0-H7:"));// 15 chars
      lcd.setCursor(15,0);
      dtostrf(h_NTC_T_Lo, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(18,0);
      dtostrf(h_NTC_T_Hi, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(21,0);
      lcd.print(degC);
      lcd.setCursor(23,0);
      lcd.print(F("  U8,U9: "));// 9 chars
      lcd.setCursor(32,0);
      dtostrf(u_NTC_T_Lo, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(35,0);
      dtostrf(u_NTC_T_Hi, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(38,0);
      lcd.print(degC);
    }
    else if(menuPotVal > potClickHi && menuPotVal < (menuPotRightmost-150)){

      if(u_NTC_N_enable == 0){
        lcd.print(F("User NTC disabled    "));// 21 chars
      }
      else{
        lcd.setCursor(0,0);
        if(u_NTC_N_enable > 0){
          lcd.print("U8:");// 3 chars
          lcd.setCursor(3,0);
          dtostrf(u_NTC_8_T, 5,1, dispTemp);
          lcd.print(dispTemp);// 5 chars
          lcd.setCursor(8,0);
          lcd.print(degC);// 2 chars
          lcd.setCursor(10,0);
          if(u_NTC_N_enable > 1){
            lcd.print(" U9:");// 4 chars
            lcd.setCursor(14,0);
            dtostrf(u_NTC_9_T, 5,1, dispTemp);
            lcd.print(dispTemp);// 5 chars
            lcd.setCursor(19,0);
            lcd.print(degC);// 2 chars
          }
          else lcd.print(F("          "));// 11 chars
        }
        else lcd.print(F("                     "));// 21 chars
        lcd.setCursor(21,0);
        if(sht_enable){
          lcd.print(F(" RH/T:"));// 6 chars
          lcd.setCursor(27,0);
          dtostrf(sht_T, 5,1, dispTemp);
          lcd.print(dispTemp);// 5 chars
          lcd.setCursor(32,0);
          lcd.print(degC);// 2 chars
          lcd.setCursor(34,0);
          lcd.print(", ");
          lcd.setCursor(36,0);
          dtostrf(sht_RH, 2,0, dispTemp);
          lcd.print(dispTemp);// 2 chars
          lcd.setCursor(38,0);
          lcd.print(" %");
        }
        else lcd.print(F("                   "));// 19 chars
      }
    }
    else if(menuPotVal > (menuPotRightmost-150) && menuPotVal < (menuPotRightmost+30) ){


      lcd.print(F("T, RH limits SHT71:  "));// 21 chars
      lcd.setCursor(21,0);
      dtostrf(SHT_T_Lo, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(24,0);
      lcd.print(",");
      lcd.setCursor(25,0);
      dtostrf(SHT_T_Hi, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(28,0);
      lcd.print(degC);

      lcd.setCursor(30,0);
      lcd.print(" | ");
      lcd.setCursor(33,0);
      dtostrf(SHT_RH_Lo, 2,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(35,0);
      lcd.print(",");
      lcd.setCursor(36,0);
      dtostrf(SHT_RH_Hi, 2,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(38,0);
      lcd.print(" %");
    }
    else if(menuPotVal > menuPotRightmost+30){
      lcd.print(F("U8,U9 Rnom,T(Rnom), B: "));//23 chars
      lcd.setCursor(23,0);
      if(u_NTC_Rnom < 10){
        lcd.print(' ');
        lcd.setCursor(24,0);
      }
      lcd.print(u_NTC_Rnom, DEC);
      lcd.setCursor(25,0);
      lcd.print(kOhm);// 2 chars
      lcd.setCursor(27,0);
      lcd.print(", ");
      lcd.setCursor(29,0);
      lcd.print(u_NTC_Tnom, DEC);
      lcd.setCursor(31,0);
      lcd.print(degC);
      lcd.setCursor(33,0);
      lcd.print(", ");//2
      lcd.setCursor(35,0);
      lcd.print(u_NTC_B);
      lcd.setCursor(39,0);
      lcd.print('K');
    }


    else{
      if(h_NTC_N_enable) lcd.print(F("|H0  |H1  |H2  |H3  |H4  |H5  |H6  |H7  "));
      else lcd.print(F("H0-H7 NTCs disabled                     "));
    }  
    nextMenuState = 1;
    break;
    //    case :
    //    lcd.print("                                        ");
    //    if(menuPotVal < potClickLo){
    //    }
    //    else if(menuPotVal > potClickHi){
    //    }
    //    else{
    //    }
    //    break;
  case 1:
    if(menuPotVal <= menuPotLeftmost){
      //lcd.print(F("                                        "));
      lcd.print(F("H0-7 NTC   u8-9_ntc   sht71 rh/t    exit"));
      menuBranch = 1;
      nextMenuState = 21;
    }
    else if(menuPotVal > menuPotLeftmost && menuPotVal < 512){
      lcd.print(F("h0-7 ntc   U8-9 NTC   sht71 rh/t    exit"));
      menuBranch = 2;
      nextMenuState = 21;//used to be 25;
    }
    else if(menuPotVal > 512 && menuPotVal < menuPotRightmost){
      lcd.print(F("h0-7 ntc   u8-9 ntc   SHT71 RH/T    exit"));
      menuBranch = 3;
      nextMenuState = 28;
    }
    else{
      lcd.print(F("h0-7 ntc   u8-9 ntc   sht71 rh/t    EXIT"));
      menuBranch = 0;
      nextMenuState = 0;
    }
    break;

  case 21:
    if(menuBranch == 1) lcd.print(F("H0-7 NTCs:"));//10 chars
    else if(menuBranch == 2) lcd.print(F("U8-9 NTCs:"));//10 chars
    lcd.setCursor(10,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("T LIMITS  N enable  set values"));//30 chars
      nextMenuState = 31;//31
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("t limits  N enable  SET VALUES"));
      nextMenuState = 32;
    }
    else{
      lcd.print(F("t limits  N ENABLE  set values"));
      if(menuBranch == 1) nextMenuState = 33;
      else if(menuBranch == 2) nextMenuState = 37;
    }
    break;

  case 28:
    eepromWritten = 0;
    lcd.print(F("SHT71 RH/T: "));//12 chars
    lcd.setCursor(12,0);
    if(menuPotVal < potClickLo){
      if(sht_enable){
        lcd.print(F("DISABLE"));// 7 chars
        menuChoice = 1;
      }
      else{
        lcd.print(F("ENABLE "));
        menuChoice = 2;
      }
      lcd.setCursor(19,0);
      lcd.print(F("  t limits  rh limits"));// 21 chars
      nextMenuState = 38;
    }
    else{  
      if(sht_enable) lcd.print(F("disable"));
      else lcd.print(F("enable "));
      lcd.setCursor(19,0);

      if(menuPotVal > potClickHi){
        lcd.print(F("  t limits  RH LIMITS"));
        nextMenuState = 39;
      }
      else{
        lcd.print(F("  T LIMITS  rh limits"));
        nextMenuState = 31;
      }
      menuChoice = 0;
    }
    break;

  case 31:
    switch(menuBranch){
    case 1: 
      lcd.print(F("H0-H7 NTC T limits: "));//20 chars
      break;
    case 2: 
      lcd.print(F("user NTCs T limits: "));
      break;
    case 3: 
      lcd.print(F("SHT71 RH/T T limits:"));
      break;
    case 0: 
      lcd.print(F("+-+-+   ERROR  +-+-+")); 
      nextMenuState = 0;
      break;
    }
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("LOW     exit    high"));// 20 chars
      nextMenuState = 41;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("low     exit    HIGH"));
      nextMenuState = 41;
      menuChoice = 2;
    }
    else{
      lcd.print(F("low     EXIT    high"));
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 32:
    eepromWritten = 0;
    if(menuBranch == 1) lcd.print(F("H0-7"));// 4 chars
    if(menuBranch == 2) lcd.print(F("U8-9"));
    lcd.setCursor(4,0);
    lcd.print(F(": dial& push to set nominal R   "));// 32 chars
    setTemp = map(menuPotVal, 0, 1024, 0, 5);
    if(menuBranch == 2){
      switch(setTemp){
      case 0:
        menuChoice = 1;
        break;
      case 1:
        menuChoice = 2;
        break;
      case 2:
        menuChoice = 3;
        break;
      case 3:
        menuChoice = 5;
        break;
      case 4:
        menuChoice = 10;
        break;
      case 5:
        menuChoice = 20;
        break;
      }
      u_NTC_Rnom = menuChoice;
    }
    else{
      if(setTemp < 3) menuChoice = 1;
      else menuChoice = 10;
      h_NTC_Rnom = menuChoice;
    }
    lcd.setCursor(36,0);
    if(menuChoice < 10){
      lcd.print(' ');
      lcd.setCursor(37,0);
    }
    lcd.print(menuChoice, DEC);
    lcd.setCursor(38,0);
    lcd.print(kOhm);
    menuChoice = 0;
    nextMenuState = 42;
    break;

  case 33:
    lcd.print(F("hybrid NTC enable (0..8, exit): "));// 32 chars
    setEnable = map(menuPotVal, 0, 1023, 0, 10);

    lcd.setCursor(32,0);
    if(setEnable >= 9){
      lcd.print(F("    EXIT"));
      nextMenuState = 0;
    }
    else{
      for(byte count = 0; count < 8; count++){
        lcd.setCursor((32+count),0);
        if(count < setEnable) lcd.print('O');
        else lcd.print('X');
      }
      nextMenuState = 43;
    }
    break;

  case 37:
    eepromWritten = 0;
    lcd.print(F("user NTC enable (0..2, exit): "));// 30 chars
    lcd.setCursor(30,0);
    if(menuPotVal < menuPotLeftmost){
      lcd.print(F("        XX"));
      setEnable = 0;
      nextMenuState = 47;
    }
    else if(menuPotVal > menuPotLeftmost && menuPotVal < potClickHi){
      lcd.print(F("        OX"));
      setEnable = 1;
      nextMenuState = 47;
    }
    else if(menuPotVal > potClickHi && menuPotVal < menuPotRightmost){
      lcd.print(F("        OO"));
      setEnable = 2;
      nextMenuState = 47;
    }
    else{
      lcd.print(F("      EXIT"));
      nextMenuState = 0;
    }
    break;

  case 38:
    if(menuChoice == 2){// previous disabled state should become enabled
      lcd.print(F("SHT71 RH/T sensor is now ENABLED        "));
      sht_enable = 1;
    }
    else{
      lcd.print(F("SHT71 RH/T sensor is now DISABLED       "));
      sht_enable = 0;
    }
    if(eepromWritten == 0){
      EEPROM_writeAnything(addr_Sht_Enable, sht_enable);
      eepromWritten = 1;
    }
    nextMenuState = 1;
    break;

  case 39:
    lcd.print(F("SHT71 RH/T RH limits: "));// 22 chars
    lcd.setCursor(22,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("LOW    exit   high"));// 18 chars
      nextMenuState = 49;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("low    exit   HIGH"));
      nextMenuState = 49;
      menuChoice = 2;
    }
    else{
      lcd.print(F("low    EXIT   high"));
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 41:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set Temperature:  "));//35 chars
    setTemp = map(menuPotVal, 0, 1023, -25, 65);
    lcd.setCursor(35,0);
    dtostrf(setTemp, 3, 0, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(38,0);
    lcd.print(degC);
    eepromWritten = 0;
    nextMenuState = 51;
    break;

  case 42:
    if(menuBranch == 1){
      if(eepromWritten == 0){
        EEPROM_writeAnything(addr_h_NTC_Rnom, h_NTC_Rnom);
        eepromWritten = 1;
      }
      lcd.print(F("H0-7"));// 4 chars
    }
    else if(menuBranch == 2){
      if(eepromWritten == 0){
        EEPROM_writeAnything(addr_u_NTC_Rnom, u_NTC_Rnom);
        eepromWritten = 1;
      }
      lcd.print(F("U8-9"));
    }
    lcd.setCursor(4,0);
    lcd.print(F(" NTC nominal R set to:         "));//31 chars 
    lcd.setCursor(35,0);
    lcd.print(setTemp); 
    lcd.setCursor(38,0);
    lcd.print(kOhm);
    nextMenuState = 52;
    break;

  case 43:
    h_NTC_N_enable = setEnable;
    if(eepromWritten == 0){
      EEPROM_writeAnything(addr_h_NTC_N_enable, setEnable);
      eepromWritten = 1;
    }
    lcd.print(F("hybrid NTC enable mask set to:  "));// 32 chars
    for(byte count = 0; count < 8; count++){
      lcd.setCursor((32+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 33;
    break;

  case 47:
    u_NTC_N_enable = setEnable;
    if(eepromWritten == 0){
      EEPROM_writeAnything(addr_u_NTC_N_enable, setEnable);
      eepromWritten = 1;
    }
    lcd.print(F("User NTC enable mask set to:          "));// 38 chars
    for(byte count = 0; count < 2; count++){
      lcd.setCursor((38+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 37;
    break;

  case 49:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set Rel. Humidity:   "));//36 chars
    setTemp = map(menuPotVal, 0, 1023, 5, 95);
    lcd.setCursor(36,0);
    dtostrf(setTemp, 2, 0, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(38,0);
    lcd.print(" %");
    nextMenuState = 59;
    break;

  case 51:
    float compTemp;
    unsigned int addrTemp;
    if(menuChoice == 1){// low T limit
      switch(menuBranch){
      case 1:
        compTemp = h_NTC_T_Hi;
        addrTemp = addr_h_NTC_T_Lo;
        break;
      case 2:
        compTemp = u_NTC_T_Hi;
        addrTemp = addr_u_NTC_T_Lo;
        break;
      case 3:
        compTemp = SHT_T_Hi;
        addrTemp = addr_Sht_T_Lo;
        break;
      }

      if(float(setTemp) > compTemp){
        lcd.print(F("Temperature not set: Low > High limit   "));
      }
      else{
        //h_NTC_T_Lo = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrTemp, float(setTemp) );
          eepromWritten = 1;
        }
        lcd.print(F("Low Temperature limit set to:      "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else if(menuChoice == 2){
      switch(menuBranch){
      case 1:
        compTemp = h_NTC_T_Lo;
        addrTemp = addr_h_NTC_T_Hi;
        break;
      case 2:
        compTemp = u_NTC_T_Lo;
        addrTemp = addr_u_NTC_T_Hi;
        break;
      case 3:
        compTemp = SHT_T_Lo;
        addrTemp = addr_Sht_T_Hi;
        break;
      }
      if(float(setTemp) < compTemp){
        lcd.print(F("Temperature not set: High < Low limit   "));
      }
      else{
        h_NTC_T_Hi = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addrTemp, float(setTemp));
          eepromWritten = 1;
        }
        lcd.print(F("High Temperature limit set to:     "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else{
      lcd.print(F("Error: temperature limit not set        "));
    }
    if(eepromWritten == 0){
      switch(menuBranch){
      case 1:
        EEPROM_readAnything(addr_h_NTC_T_Lo, h_NTC_T_Lo);
        EEPROM_readAnything(addr_h_NTC_T_Hi, h_NTC_T_Hi);
        break;
      case 2:
        EEPROM_readAnything(addr_u_NTC_T_Lo, u_NTC_T_Lo);
        EEPROM_readAnything(addr_u_NTC_T_Hi, u_NTC_T_Hi);
        break;
      case 3:
        EEPROM_readAnything(addr_Sht_T_Lo, SHT_T_Lo);
        EEPROM_readAnything(addr_Sht_T_Hi, SHT_T_Hi);
        break;
      }
      eepromWritten = 1;
    }
    nextMenuState = 31;
    break;

  case 52:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set NTC T(Rnom):     "));// 36 chars
    lcd.setCursor(36,0);
    if(menuPotVal < potClickHi){
      lcd.print(F("20"));
      menuChoice = 20;
    }
    else{
      lcd.print(F("25"));// deg C");
      menuChoice = 25;
    }
    lcd.setCursor(38,0);
    lcd.print(degC);
    nextMenuState = 62;
    break;

  case 59:
    if(menuChoice == 1){// low RH limit
      if(float(setTemp) > SHT_RH_Hi){
        lcd.print(F("Rel. Humidity not set: Low > High limit "));
      }
      else{
        SHT_RH_Lo = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addr_Sht_RH_Lo, SHT_RH_Lo);
          eepromWritten = 1;
        }
        lcd.print(F("Low Relative Humidity limit set to: "));// 36 chars
      }
    }
    else if(menuChoice == 2){// high RH limit
      if(float(setTemp) < SHT_RH_Lo){
        lcd.print(F("Rel. Humidity not set: High < Low limit "));
      }
      else{
        SHT_RH_Hi = float(setTemp);
        if(eepromWritten == 0){
          EEPROM_writeAnything(addr_Sht_RH_Hi, SHT_RH_Hi);
          eepromWritten = 1;
        }
        lcd.print(F("High Relative Humidity limit set to:"));// 36 chars
      }
    }
    lcd.setCursor(36,0);
    lcd.print(setTemp);
    lcd.setCursor(38,0);
    lcd.print(" %");
    nextMenuState = 39;
    break;


  case 62:
    lcd.print(F("NTC Temperature (Rnominal) set to:  "));//36 chars
    lcd.setCursor(36,0);
    lcd.print(menuChoice, DEC);
    lcd.setCursor(38,0);
    lcd.print(degC);
    if(eepromWritten == 0){
      if(menuBranch == 1){
        EEPROM_writeAnything(addr_h_NTC_Tnom, menuChoice);
        h_NTC_Tnom = menuChoice;
      }
      else if(menuBranch == 2){
        EEPROM_writeAnything(addr_u_NTC_Tnom, menuChoice);
        u_NTC_Tnom = menuChoice;
      }
      eepromWritten = 1;
    }
    nextMenuState = 72;
    break;

  case 72:
    eepromWritten = 0;
    lcd.print(F("Dial & push to set NTC B value:    "));//34 chars
    setTemp = 5*(map(menuPotVal, 0, 1024, 600, 800));// u_NTC_B: 3000-4000 steps of 5
    lcd.setCursor(34,0);
    lcd.print(setTemp); 
    lcd.setCursor(38,0);
    lcd.print(" K");
    if(menuBranch == 1) h_NTC_B = setTemp;
    else if (menuBranch == 2) u_NTC_B = setTemp;
    nextMenuState = 82;
    break;

  case 82:
    if(menuBranch == 1){
      if(eepromWritten == 0){
        EEPROM_writeAnything(addr_h_NTC_B, h_NTC_B);


        EEPROM_readAnything(addr_h_NTC_Rnom, h_NTC_Rnom);
        EEPROM_readAnything(addr_h_NTC_Tnom, h_NTC_Tnom);
        EEPROM_readAnything(addr_h_NTC_B, h_NTC_B);

        Serial.print("H0-7 Rnom, Tnom, B set to: ");
        Serial.print(h_NTC_Rnom);
        Serial.print(", ");
        Serial.print(h_NTC_Tnom);
        Serial.print(", ");
        Serial.println(h_NTC_B);

        h_NTC_Mux.setNTCup(hNTCMuxPin, nSamples, h_NTC_Rnom);
        h_NTC_Mux.setNTCParameters(h_NTC_Rnom, h_NTC_Tnom, h_NTC_B);
        eepromWritten = 1;
      }
      lcd.print(F("H0-7"));

    }
    else if(menuBranch == 2){
      if(eepromWritten == 0){
        EEPROM_writeAnything(addr_u_NTC_B, u_NTC_B);

        EEPROM_readAnything(addr_u_NTC_Rnom, u_NTC_Rnom);
        EEPROM_readAnything(addr_u_NTC_Tnom, u_NTC_Tnom);
        EEPROM_readAnything(addr_u_NTC_B, u_NTC_B);

        Serial.print("U8-9 Rnom, Tnom, B set to: ");
        Serial.print(u_NTC_Rnom);
        Serial.print(", ");
        Serial.print(u_NTC_Tnom);
        Serial.print(", ");
        Serial.println(u_NTC_B);

        u_NTC_8.setNTCParameters(u_NTC_Rnom, u_NTC_Tnom, u_NTC_B);
        u_NTC_9.setNTCParameters(u_NTC_Rnom, u_NTC_Tnom, u_NTC_B);
        eepromWritten = 1;
      }
      lcd.print(F("U8-9"));
    }
    lcd.setCursor(4,0);
    lcd.print(F(" NTC B value set to:          "));// 30 chars
    lcd.setCursor(34,0);
    if(menuBranch == 1) lcd.print(h_NTC_B);
    if(menuBranch == 2) lcd.print(u_NTC_B);
    lcd.setCursor(38,0);
    lcd.print(" K");
    nextMenuState = 1;
    break;
  }

}// loop()

// FUNCTIONS

// clear lcd line
void lcdClearLine(int lineNum){
  if(lineNum == 0 || lineNum == 1){
    lcd.setCursor(0,lineNum);
    lcd.print(F("                                        "));
  }
  return;
}


/* Calculates the size of the free list */
int freeListSize() {
  struct __freelist* current;
  int total = 0;

  for (current = __flp; current; current = current->nx) {
    total += 2; /* Add two bytes for the memory block's header  */
    total += (int) current->sz;
  }

  return total;
}

int freeMemory() {
  int free_memory;

  if ((int)__brkval == 0) {
    free_memory = ((int)&free_memory) - ((int)&__heap_start);
  } 
  else {
    free_memory = ((int)&free_memory) - ((int)__brkval);
    free_memory += freeListSize();
  }
  return free_memory;
}












