// sketch for ATLAS Upgrade Interlock II

// Bart Hommels
// University of Cambridge

// 16/7/2012
// V 0.1

//+-------------------------------------------------------+
// Libraries
//#include <stdio.h>
#include "Wire.h"// for driving SPI bus and/or I2C/onewire
#include "LiquidCrystal.h"// for LCD on SPI
#include <EEPROM.h>// for using EEPROM
#include "EEPROMAnything.h" // for writing multiple-byte objects to EEPROM
#include <NTCreadout.h>

// +-------------------------------------------------------+
// Definitions

// +-------------------------------------------------------+
// pin assignments
const int muxNTC = A0;
const int userNTC09pin = A1;
const int userNTC08pin = A2;
const int menuPot = A3;

const int d0_output = 0;
const int d1_output = 1;
const int muxCtrl0 = 10;//0
const int muxCtrl1 = 11;//1
const int muxCtrl2 = 2;
const int menuButton = 3;
const int lcdLatch = 4;
const int lcdClk = 5;
const int lcdData = 6;
const int duoLed = 7;

const int outputEnable = 8;
const int selectRseries = 9;
//const int d0_output = 10;
//const int d1_output = 11;
const int enablePbLED = 12;
const int pbEnable = 13;

//+-------------------------------------------------------+
// constants
const float absTzero = -273.15;// 0 Kelvin in deg C

// GL23
//const float RseriesNTC09 = 10000.0;
//const float RrefNTC09 = 2000.0;
//const float TrefNTC09 = 20.0-absTzero;
//const float BntcNTC09 = 3125.0;

//+-------------------------------------------------------+
// variables

// High, low temperature limits for hybrid, user NTCs.
// values here are extremely conservative fallbacks. actual settings should be collected from EEPROM
float hybridTHi = 30.0;
float hybridTLo = 10.0;
float userNTCTHi = 30.0;
float userNTCTLo = 10.0;

byte hybridNTCRnom = 10;// hybrid NTC resistance at nominal T, in kOhm. Should be 1 or 10 (default)
byte userNTCRnom = 2;// user NTC resistance at nominal T in kOhm: [1,2,5,10,12,15,20]
byte userNTCTnom = 20;// user NTC nominal T in deg C. Should be 20 or 25.

// 10kOhm hybrid NTCs as default
float RseriesHybrid = 10000.0;
float RrefHybrid = 10000.0;
const float TrefHybrid = 25.0-absTzero;
const float BntcHybrid = 3435.0;

// user NTC defaults (use GL23 for the moment)
const float RseriesUserNTC = 10000.0;
float RrefUserNTC = 2000.0;
float TrefUserNTC = 20-absTzero;
float BUserNTC = 3125.0;

// EEPROM addresses for non volatile parameters
unsigned int addrHybridTLo = 0;
unsigned int addrHybridTHi = 4;
unsigned int addrHybridEnableMask = 8;
unsigned int addrHybridNTCRnom = 9;

unsigned int addrUserNTCTLo = 20;
unsigned int addrUserNTCTHi = 24;
unsigned int addrUserNTCEnableMask = 28;
unsigned int addrUserNTCRnom = 29;
unsigned int addrUserNTCTnom = 30;
unsigned int addrUserNTCBntc = 31;

int nSamples = 2;
byte nHybridNTCs = 1;
byte nUserNTCs = 0;

char dispTemp[5];
// menu pot limits and values
const int potClickLo = 470;
const int potClickHi = 560;

int lastMenuPotVal = 0;
//int saveMenuPotVal = 0;

// menu state machine variables
byte buttonRead = 0;
byte prevButtonRead = 0;
byte menuState = 0;
byte nextMenuState = 0;
byte prevMenuState = 0;
byte menuChoice = 0;

float setTemp = 0.0;
int tempInt = 0;
byte setEnable = 0;

byte enableLedState = LOW;
byte outputState = LOW;
byte userEnable = LOW;
// timing control of refresh rates
unsigned long timerLast;

//+-------------------------------------------------------+
// instantiations

// SPI LCD interface through 74LS595 shift register, see:
// http://www.ladyada.net/products/i2cspilcdbackpack/index.html
// DATA (p14 on 74LS595), CLK (p11 on 74LS595), LATCH (p12 on 74LS595)
LiquidCrystal lcd(6,5,4);


// NTC serving MUXed input, user08, user09
NTCtemp hybridNTC;
byte hybridTnotOK = 0x00;

NTCtemp userNTC08;
NTCtemp userNTC09;
byte userTnotOK = 0x00;

//+-------------------------------------------------------+
void setup(){
  // set mode for all remaining pins
  pinMode(muxCtrl0, OUTPUT);
  pinMode(muxCtrl1, OUTPUT);
  pinMode(muxCtrl2, OUTPUT);
  pinMode(menuButton, INPUT);
  pinMode(duoLed, OUTPUT);
  pinMode(outputEnable, OUTPUT);
  pinMode(selectRseries, OUTPUT);
  pinMode(d0_output, OUTPUT);
  pinMode(d1_output, OUTPUT);
  pinMode(enablePbLED, OUTPUT);
  pinMode(pbEnable, INPUT);

  // write default outputs to pins
  digitalWrite(muxCtrl0, LOW);
  digitalWrite(muxCtrl1, LOW);
  digitalWrite(muxCtrl2, LOW);

  digitalWrite(selectRseries, HIGH);
  // use external analog reference
  analogReference(EXTERNAL);

  // set variables
  menuState = 0;
  nextMenuState = 0;

  enableLedState = LOW;
  outputState = LOW;

  // retrieve variables from EEPROM
  EEPROM_readAnything(addrHybridTLo, hybridTLo);
  EEPROM_readAnything(addrHybridTHi, hybridTHi);
  EEPROM_readAnything(addrHybridEnableMask, nHybridNTCs);
  EEPROM_readAnything(addrHybridNTCRnom, hybridNTCRnom);

  EEPROM_readAnything(addrUserNTCTLo, userNTCTLo);
  EEPROM_readAnything(addrUserNTCTHi, userNTCTHi);
  EEPROM_readAnything(addrUserNTCEnableMask, nUserNTCs);
  EEPROM_readAnything(addrUserNTCRnom, userNTCRnom);
  EEPROM_readAnything(addrUserNTCTnom, userNTCTnom);
  EEPROM_readAnything(addrUserNTCBntc, BUserNTC);

  RrefHybrid = 1000.0 * float(hybridNTCRnom);// either 1 or 10 kOhm.
  RseriesHybrid = RrefHybrid;

  // initialise NTC instances
  hybridNTC.setNTCup(muxNTC, nSamples, RseriesHybrid);
  hybridNTC.setNTCParameters(RrefHybrid, TrefHybrid, BntcHybrid);

  RrefUserNTC = 1000.0 * float(userNTCRnom);
  TrefUserNTC = float(userNTCTnom)-absTzero;

  userNTC08.setNTCup(userNTC08pin, nSamples, RseriesUserNTC);
  userNTC08.setNTCParameters(RrefUserNTC, TrefUserNTC, BUserNTC);

  userNTC09.setNTCup(userNTC09pin, nSamples, RseriesUserNTC);
  userNTC09.setNTCParameters(RrefUserNTC, TrefUserNTC, BUserNTC);

  // setup lcd + print welcome message
  lcd.begin(40,2);
  lcd.setCursor(10,0);
  lcd.print("Temperature Interlock");// 21 chars 
  lcd.setBacklight(HIGH);
  lcd.setCursor(6,1);
  lcd.print("firmware v0.2 dd 29/08/2012");// 27 chars
  delay(2000);
  lcd.clear();
}


void loop(){
  float hybridTemp[8];// keeping this fixed length is lots faster
  float userTemp08 = 20.0;
  float userTemp09 = 20.0;

  byte count = 0;
  //  unsigned long timerlast = millis();


  // readout loop for hybrid NTCs
  // set series R according to hybrid type. 
  // For 1 kOhm, drive selectRseries line low during mux'ed measurement to avoid dissipation when idle
  if(hybridNTCRnom == 1) digitalWrite(selectRseries, LOW);

  for(count=0; count < 8; count++){
    if(count < nHybridNTCs){
      digitalWrite(muxCtrl0, (bitRead(count,0) ) );
      digitalWrite(muxCtrl1, (bitRead(count,1) ) );
      digitalWrite(muxCtrl2, (bitRead(count,2) ) );

      // read out temperature
      hybridTemp[count] = hybridNTC.getDegC();

      if(hybridTemp[count] > hybridTHi || hybridTemp[count] < hybridTLo){ 
        bitWrite(hybridTnotOK, count, HIGH);
      }
      else{
        bitWrite(hybridTnotOK, count, LOW);
      }
    }
    else{
      bitWrite(hybridTnotOK, count, LOW);
    }
  }
  digitalWrite(selectRseries, HIGH);

  // read out user NTCs if enabled
  // check temperature readings against limits and set bits in userTnotOK accordingly
  if(nUserNTCs > 0){
    userTemp08 = userNTC08.getDegC();
    if(userTemp08 > userNTCTHi || userTemp08 < userNTCTLo) bitWrite(userTnotOK, 0, HIGH);
    else bitWrite(userTnotOK, 0, LOW);

    if(nUserNTCs > 1){ 
      userTemp09 = userNTC09.getDegC();
      if(userTemp09 > userNTCTHi || userTemp09 < userNTCTLo) bitWrite(userTnotOK, 1, HIGH);
      else bitWrite(userTnotOK, 1, LOW);

    }
  }
  else userTnotOK = 0x00;

  //  unsigned long t_diff = millis()-timerlast;

  // read enable pushbutton pin, see if Temp. is OK and set outputState accordingly
  userEnable = digitalRead(pbEnable);

  if(hybridTnotOK == 0 && nHybridNTCs > 0 && userTnotOK == 0){// enabled NTCs Temperatures are all OK
    digitalWrite(duoLed, HIGH);// T indicator LEDs go green
    if(userEnable == HIGH){// Temperature OK, enable PB pushed
      outputState = HIGH;
    }
  }
  else{
    digitalWrite(duoLed, LOW);
    outputState = LOW;
  }
  digitalWrite(outputEnable, outputState);

  // set enable PB LED state:
  // solid on when T OK and SSR enabled, blink when T OK and SSR disabled. Off when T not OK.
  if(millis()-timerLast > 500UL){
    timerLast = millis();

    if(hybridTnotOK == 0 && nHybridNTCs > 0 && userTnotOK == 0){
      if(outputState == HIGH) enableLedState = HIGH;
      else{
        if(enableLedState == LOW) enableLedState = HIGH;
        else enableLedState = LOW;
      }
    }     
    else enableLedState = LOW;
  }
  digitalWrite(enablePbLED, enableLedState);

  // print timing information
  //lcd.setCursor(38,0);
  //lcd.print(t_diff);


  // do menu stuff
  int menuPotVal = analogRead(menuPot);// for use in menu
  // read menu button, debounce, increase menustate if necessary
  int buttonRead = digitalRead(menuButton);

  // check button is pressed, advance to next state
  if(buttonRead != prevButtonRead && buttonRead == HIGH){// leading edge
    delay(50);//require button to be pressed 50ms (= debounce as well)
    if( buttonRead == HIGH ){
      //saveMenuPotVal = menuPotVal;//??
      prevMenuState = menuState; 
      menuState = nextMenuState;

    }
  }
  prevButtonRead = buttonRead;

  // +=+=+=+=+=+=+=+=+ MENU STATE MACHINE +=+=+=+=+=+=+=+=+
  // menu state must rewrite entire first line. Avoid clearing LCD as it causes flicker
  //lcdClearLine(0);
  lcd.home();
  switch(menuState){
  case 0:
    if(menuPotVal < potClickLo){ 
      lcd.print("Tlim. H0-H7:");// 12 chars max
      lcd.setCursor(12,0);
      dtostrf(hybridTLo, 5,1, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(17,0);
      dtostrf(hybridTHi, 5,1, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(22,0);
      lcd.print(" U8,U9: ");// 8 chars
      dtostrf(userNTCTLo, 5,1, dispTemp);
      lcd.setCursor(30,0);
      lcd.print(dispTemp);
      dtostrf(userNTCTHi, 5,1, dispTemp);
      lcd.setCursor(35,0);
      lcd.print(dispTemp);
    }
    else if(menuPotVal > potClickHi && menuPotVal < 900){
      //
      if(nUserNTCs == 0) lcd.print("User NTC, I2C RH/T disabled             ");
      else{
        if(nUserNTCs > 0){
          lcd.print("T U8:");// 5 chars
          lcd.setCursor(5,0);
          dtostrf(userTemp08, 5,1, dispTemp);
          lcd.print(dispTemp);
          lcd.setCursor(10,0);
          if(nUserNTCs == 1) lcd.print("          ");// 10 chars
        }
        if(nUserNTCs > 1){
          lcd.print(" T U9:");// 6 chars
          lcd.setCursor(16,0);
          dtostrf(userTemp09, 5,1, dispTemp);
          lcd.print(dispTemp);
        }
        lcd.setCursor(21,0);
        lcd.print("                   ");// 19 chars
      }


    }
    else if(menuPotVal > 900){
      lcd.print("U8,U9 R,T(nom), B:");//18 chars
      lcd.setCursor(18,0);
      if(userNTCRnom < 10){
        lcd.print(' ');
        lcd.setCursor(19,0);
      }
      lcd.print(userNTCRnom, DEC);
      lcd.setCursor(20,0);
      lcd.print("kOhm,");// 6 chars
      lcd.print(25,0);
      lcd.print(userNTCTnom, DEC);
      lcd.setCursor(29,0);
      lcd.print((char)223);
      lcd.setCursor(30,0);
      lcd.print("C    ");//5
      lcd.setCursor(35,0);
      dtostrf(BUserNTC, 4,0, dispTemp);
      lcd.print(BUserNTC);
      lcd.setCursor(39,0);
      lcd.print('K');
    }
    else{
      lcd.print("|H0  |H1  |H2  |H3  |H4  |H5  |H6  |H7  ");
    }  
    // print Temperature values to lcd on bottom line
    for(count = 0; count < 8; count++){
      lcd.setCursor((count*5),1);
      if(count < nHybridNTCs){
        dtostrf(hybridTemp[count], 5, 1, dispTemp);
        lcd.print(dispTemp);
      }
      else lcd.print("     ");
    }
    nextMenuState = 1;
    break;
    //    case :
    //    lcd.print("                                        ");
    //    if(menuPotVal < potClickLo){
    //    }
    //    else if(menuPotVal > potClickHi){
    //    }
    //    else{
    //    }
    //    break;
  case 1:
    if(menuPotVal <= 255){
      lcd.print("HYBRID_NTC user_ntc  user_rh/t     exit  ");
      nextMenuState = 21;
    }
    else if(menuPotVal > 255 && menuPotVal < 512){
      lcd.print("hybrid_ntc USER_NTC  user_rh/t     exit  ");
      nextMenuState = 25;
    }
    else if(menuPotVal > 512 && menuPotVal < 768){
      lcd.print("hybrid_ntc user_ntc  USER_RH/T     exit  ");
      nextMenuState = 0;
    }
    else{
      lcd.print("hybrid_ntc user_ntc  user_rh/t     EXIT  ");
      nextMenuState = 0;
    }
    break;

  case 21:
    lcd.print("hybrid NTC :");//12 chars
    lcd.setCursor(12,0);
    if(menuPotVal < potClickLo){
      lcd.print("T LIMITS  N enable  set Type");//28 chars
      nextMenuState = 31;//31
    }
    else if(menuPotVal > potClickHi){
      lcd.print("t limits  N enable  SET TYPE");
      nextMenuState = 32;//32
    }
    else{
      lcd.print("t limits  N ENABLE  set Type");
      nextMenuState = 33;//33
    }
    break;

  case 25:
    lcd.print("user NTCs:");//10 chars
    lcd.setCursor(10,0);
    if(menuPotVal < potClickLo){
      lcd.print("T LIMITS  N enable  set values");//30 chars
      nextMenuState = 35;
    }
    else if(menuPotVal > potClickHi){
      lcd.print("t limits  N enable  SET VALUES");
      nextMenuState = 36;
    }
    else{
      lcd.print("t limits  N ENABLE  set values");
      nextMenuState = 37;
    }
    break;

  case 31:
    lcd.print("hybrid NTC T limits:");//20 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print("LOW     exit    high");// 20 chars
      nextMenuState = 41;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print("low     exit    HIGH");
      nextMenuState = 41;
      menuChoice = 2;
    }
    else{
      lcd.print("low     EXIT    high");
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 32:
    //lcd.print("Hybrid NTC R @ 25degC: ");//23 chars
    lcd.print("Hybrid NTC R(25");//15 chars
    lcd.setCursor(15,0);
    lcd.print((char)223);
    lcd.setCursor(16,0);
    lcd.print("C): ");// 4 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print(" 10KOHM  exit  1kohm");
      menuChoice = 1;
      nextMenuState = 42;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(" 10kohm  exit  1KOHM");
      menuChoice = 2;
      nextMenuState = 42;
    }
    else{
      lcd.print(" 10kohm  EXIT  1kohm");
      menuChoice = 0;
      nextMenuState = 0;
    }
    break;

  case 33:
    lcd.print("hybrid NTC enable (0..8, exit): ");// 32 chars
    setEnable = map(menuPotVal, 0, 1024, 0, 10);

    lcd.setCursor(32,0);
    if(setEnable >= 9){
      lcd.print("    EXIT");
      nextMenuState = 0;
    }
    else{
      for(count = 0; count < 8; count++){
        lcd.setCursor((32+count),0);
        if(count < setEnable) lcd.print('O');
        else lcd.print('X');
      }
      nextMenuState = 43;
    }
    break;

  case 35:
    lcd.print("user NTC T limits:  ");//20 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print("LOW     exit    high");// 20 chars
      nextMenuState = 45;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print("low     exit    HIGH");
      nextMenuState = 45;
      menuChoice = 2;
    }
    else{
      lcd.print("low     EXIT    high");
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 36:
    lcd.print("Dial & Push to set nominal R (kOhm):  ");// 38 chars
    tempInt = map(menuPotVal, 0, 1024, 0, 7);
    lcd.print(tempInt);
    switch(tempInt){
    case 0:
      userNTCRnom = 1;
      break;
    case 1:
      userNTCRnom = 2;
      break;
    case 3:
      userNTCRnom = 3;
      break;
    case 4:
      userNTCRnom = 5;
      break;
    case 5:
      userNTCRnom = 10;
      break;
    case 6:
      userNTCRnom = 20;
      break;
    }
    lcd.setCursor(38,0);
    if(tempInt < 5){
      lcd.print(' ');
      lcd.setCursor(39,0);
    }
    lcd.print(userNTCRnom, DEC);

    nextMenuState = 46;
    break;


  case 37:
    lcd.print("user NTC enable (0..2, exit): ");// 30 chars
    lcd.setCursor(30,0);
    if(menuPotVal < 255){
      lcd.print("        XX");
      setEnable = 0;
      nextMenuState = 47;
    }
    else if(menuPotVal > 254 && menuPotVal < potClickHi){
      lcd.print("        OX");
      setEnable = 1;
      nextMenuState = 47;
    }
    else if(menuPotVal > potClickHi && menuPotVal < 800){
      lcd.print("        OO");
      setEnable = 2;
      nextMenuState = 47;
    }
    else{
      lcd.print("      EXIT");
      nextMenuState = 0;
    }
    break;

  case 41:
    lcd.print("Dial & Push to set Temperature:    ");//35 chars
    setTemp = map(menuPotVal, 0, 1024, -25.0, 65.0);
    lcd.setCursor(35,0);
    dtostrf(setTemp, 5,1, dispTemp);
    lcd.print(dispTemp);
    nextMenuState = 51;
    break;

  case 42:
    lcd.print("Hybrid NTC R(25 at 25 deg C set to: ");//23 chars
    lcd.setCursor(32,0);
    if(menuChoice == 1){
      hybridNTCRnom = 10;
      lcd.print("10 kOhm");
    }
    else if(menuChoice == 2){
      lcd.print(" 1 kOhm");
      hybridNTCRnom = 1;
    }

    EEPROM_writeAnything(addrHybridNTCRnom, hybridNTCRnom);
    RrefHybrid = 1000.0 * float(hybridNTCRnom);
    RseriesHybrid = RrefHybrid;

    // initialise NTC instances
    hybridNTC.setNTCup(muxNTC, nSamples, RseriesHybrid);
    hybridNTC.setNTCParameters(RrefHybrid, TrefHybrid, BntcHybrid);
    nextMenuState = 32;
    break;

  case 43:
    nHybridNTCs = setEnable;
    EEPROM_writeAnything(addrHybridEnableMask, setEnable);
    lcd.print("hybrid NTC enable mask set to:  ");// 32 chars
    for(count = 0; count < 8; count++){
      lcd.setCursor((32+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 33;
    break;

  case 45:
    lcd.print("Dial & Push to set Temperature:    ");//35 chars
    setTemp = map(menuPotVal, 0, 1024, -25.0, 75.0);
    lcd.setCursor(35,0);
    dtostrf(setTemp, 5,1, dispTemp);
    lcd.print(dispTemp);
    nextMenuState = 55;
    break;

  case 46:
    lcd.print("Nominal user NTC R set to: ");//27 chars 
    lcd.setCursor(27,0);
    lcd.print(userNTCRnom, DEC);
    lcd.setCursor(29,0);
    lcd.print(" kOhm     ");
    EEPROM_writeAnything(addrUserNTCRnom, userNTCRnom);
    RrefUserNTC = 1000.0 * float(userNTCRnom);
    nextMenuState = 56;
    break;

  case 47:
    nUserNTCs = setEnable;
    EEPROM_writeAnything(addrUserNTCEnableMask, setEnable);
    lcd.print("User NTC enable mask set to:          ");// 38 chars
    for(count = 0; count < 2; count++){
      lcd.setCursor((38+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 37;
    break;

  case 51:
    if(menuChoice == 1){// low T limit
      hybridTLo = setTemp;
      EEPROM_writeAnything(addrHybridTLo, setTemp);
      lcd.print("Low Temperature limit set to:      ");
      lcd.setCursor(35,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else if(menuChoice == 2){
      hybridTHi = setTemp;
      EEPROM_writeAnything(addrHybridTHi, setTemp);
      lcd.print("High Temperature limit set to:     ");
      lcd.setCursor(35,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else{
      lcd.print("Error: temperature limit not set        ");
    }
    nextMenuState = 31;
    break;

  case 55:
    if(menuChoice == 1){// low T limit
      userNTCTLo = float(setTemp);
      EEPROM_writeAnything(addrUserNTCTLo, userNTCTLo);
      lcd.print("Low Temperature limit set to:      ");
      lcd.setCursor(35,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else if(menuChoice == 2){
      userNTCTHi = float(setTemp);
      EEPROM_writeAnything(addrUserNTCTHi, userNTCTHi);
      lcd.print("High Temperature limit set to:     ");
      lcd.setCursor(35,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else{
      lcd.print("Error: temperature limit not set        ");
    }
    nextMenuState = 35;
    break;

  case 56:
    lcd.print("Dial & Push to set NTC T(Rnom): ");// 32 chars
    lcd.setCursor(32,0);
    if(menuPotVal < potClickHi){
      lcd.print("20 deg C");
      userNTCTnom = 20;
    }
    else{
      lcd.print("25 deg C");
      userNTCTnom = 25;
    }
    EEPROM_writeAnything(addrUserNTCTnom, userNTCTnom);
    nextMenuState = 66;
    break;

  case 66:
    lcd.print("Dial & Push to set NTC B value:     ");//36 chars
    tempInt = int(map(menuPotVal, 0, 1024, 3100, 3500));
    lcd.setCursor(36,0);
    lcd.print(tempInt);
    BUserNTC = float(tempInt);
    TrefUserNTC = float(userNTCTnom)-absTzero;
    EEPROM_writeAnything(addrUserNTCBntc, BUserNTC);
    userNTC08.setNTCParameters(RrefUserNTC, TrefUserNTC, BUserNTC);
    userNTC09.setNTCParameters(RrefUserNTC, TrefUserNTC, BUserNTC);
    nextMenuState = 1;
    break;

  }

}

// FUNCTIONS

// clear lcd line
void lcdClearLine(int lineNum){
  if(lineNum == 0 || lineNum == 1){
    lcd.setCursor(0,lineNum);
    lcd.print("                                        ");
  }
  return;
}
















































