
// sketch for ATLAS Upgrade Interlock II

// Bart Hommels
// University of Cambridge

// 16/7/2012
// V 0.1: basic hybrid NTC readout
// v 0.3: hybrid, user ntc readout, set menu complete.
// v 0.4: 31/08/2012 reduced memory footprint
// v 0.5: 01/09/2012 various improvements
// v 0.6: 04/09 major memory footprint reduction using F() for LCD character strings
// w/ full sketch, freeMemory() returns 1428 B
// v 0.7: 06/09 minor improvements



//+-------------------------------------------------------+
// Libraries
#include <stdio.h>
#include "Wire.h"// for driving SPI bus and/or I2C/onewire
#include "LiquidCrystal.h"// for LCD on SPI
#include <EEPROM.h>// for using EEPROM
#include "EEPROMAnything.h" // for writing multiple-byte objects to EEPROM
#include <NTCreadlean.h>
//----------------------------------------------------------------
// Memory Usage Monitoring +-------------------------------------+
extern unsigned int __heap_start;
extern void *__brkval;
/*
 * The free list structure as maintained by the 
 * avr-libc memory allocation routines.
 */
struct __freelist {
  size_t sz;
  struct __freelist *nx;
};
/* The head of the free list structure */
extern struct __freelist *__flp;
//---------------------------------------------------------------+

// +-------------------------------------------------------+
// Definitions
const char fwVersion[4] = "0.6";
const char fwDate[11] = "10/09/2012";
// +-------------------------------------------------------+
// pin assignments
const byte muxNTC = A0;
const byte userNTC09pin = A1;
const byte userNTC08pin = A2;
const byte menuPot = A3;

const byte d0_output = 0;
const byte d1_output = 1;
const byte muxCtrl0 = 10;//0
const byte muxCtrl1 = 11;//1
const byte muxCtrl2 = 2;
const byte menuButton = 3;
const byte lcdLatch = 4;
const byte lcdClk = 5;
const byte lcdData = 6;
const byte duoLed = 7;

const byte outputEnable = 8;
const byte selectRseries = 9;
//const byte d0_output = 10;
//const byte d1_output = 11;
const byte enablePbLED = 12;
const byte pbEnable = 13;

//+-------------------------------------------------------+
// variables

// High, low temperature limits for hybrid, user NTCs.
// values here are extremely conservative fallbacks. actual settings should be collected from EEPROM
float hybridTHi = 30.0;// keep to floats not to slow down actual interlock code
float hybridTLo = 10.0;
float userNTCTHi = 30.0;
float userNTCTLo = 10.0;

byte hybridNTCRnom = 10;// hybrid NTC resistance at nominal T, in kOhm. Should be 1 or 10 (default)
const byte TrefHybrid = 25;
const int BntcHybrid = 3435;

// user NTC defaults (use GL23 for the moment: R(20 deg C) = 2kOhm, B=312)
byte userNTCRnom = 2;// user NTC resistance at nominal T in kOhm: [1,2,5,10,12,15,20]
const byte userNTCRseries = 10;// kOhm
byte userNTCTnom = 20;
int BUserNTC = 3125;

// EEPROM addresses for non volatile parameters
const unsigned int addrHybridTLo = 0;
const unsigned int addrHybridTHi = 4;
const unsigned int addrHybridEnableMask = 8;
const unsigned int addrHybridNTCRnom = 9;

const unsigned int addrUserNTCTLo = 20;
const unsigned int addrUserNTCTHi = 24;
const unsigned int addrUserNTCEnableMask = 28;
const unsigned int addrUserNTCRnom = 29;
const unsigned int addrUserNTCTnom = 30;
const unsigned int addrUserNTCBntc = 31;

const byte nSamples = 2;
byte nHybridNTCs = 1;
byte nUserNTCs = 2;

char dispTemp[5];
// menu pot limits and values
const int potClickLo = 470;
const int potClickHi = 560;

const char degC[] = { 
  (char)223, 'C'};
const char kOhm[] = { 
  'k', (char)244 };

int lastMenuPotVal = 0;

// menu state machine variables
byte buttonRead = 0;
byte prevButtonRead = 0;
byte menuState = 0;
byte nextMenuState = 0;
byte prevMenuState = 0;
byte menuChoice = 0;

int setTemp = 0;
float mapTemp = 0.0;
byte setEnable = 0;

byte enableLedState = LOW;
byte outputState = LOW;
byte userEnable = LOW;
// timing control of refresh rates
unsigned long serialTimerLast, flashTimerLast;

//+-------------------------------------------------------+
// instantiations

// SPI LCD interface through 74LS595 shift register, see:
// http://www.ladyada.net/products/i2cspilcdbackpack/index.html
// DATA (p14 on 74LS595), CLK (p11 on 74LS595), LATCH (p12 on 74LS595)
LiquidCrystal lcd(6,5,4);

// NTC serving MUXed input, user08, user09
NTCtemp hybridNTC;
byte hybridTnotOK = 0x00;

NTCtemp userNTC08;
NTCtemp userNTC09;
byte userTnotOK = 0x00;
//+-------------------------------------------------------+
void setup(){
  Serial.begin(9600);
  // set mode for all remaining pins
  pinMode(muxCtrl0, OUTPUT);
  pinMode(muxCtrl1, OUTPUT);
  pinMode(muxCtrl2, OUTPUT);
  pinMode(menuButton, INPUT);
  pinMode(duoLed, OUTPUT);
  pinMode(outputEnable, OUTPUT);
  pinMode(selectRseries, OUTPUT);
  pinMode(d0_output, OUTPUT);
  pinMode(d1_output, OUTPUT);
  pinMode(enablePbLED, OUTPUT);
  pinMode(pbEnable, INPUT);

  // write default outputs to pins
  digitalWrite(muxCtrl0, LOW);
  digitalWrite(muxCtrl1, LOW);
  digitalWrite(muxCtrl2, LOW);

  digitalWrite(selectRseries, HIGH);
  // use external analog reference
  analogReference(EXTERNAL);

  // set variables
  menuState = 0;
  nextMenuState = 0;

  enableLedState = LOW;
  outputState = LOW;

  serialTimerLast = millis();
  flashTimerLast = serialTimerLast;

  // retrieve variables from EEPROM
  EEPROM_readAnything(addrHybridTLo, hybridTLo);
  EEPROM_readAnything(addrHybridTHi, hybridTHi);
  EEPROM_readAnything(addrHybridEnableMask, nHybridNTCs);
  EEPROM_readAnything(addrHybridNTCRnom, hybridNTCRnom);

  EEPROM_readAnything(addrUserNTCTLo, userNTCTLo);
  EEPROM_readAnything(addrUserNTCTHi, userNTCTHi);
  EEPROM_readAnything(addrUserNTCEnableMask, nUserNTCs);
  EEPROM_readAnything(addrUserNTCRnom, userNTCRnom);
  EEPROM_readAnything(addrUserNTCTnom, userNTCTnom);
  EEPROM_readAnything(addrUserNTCBntc, BUserNTC);

  // initialise NTC instances
  hybridNTC.setNTCup(muxNTC, nSamples, hybridNTCRnom);
  hybridNTC.setNTCParameters(hybridNTCRnom, TrefHybrid, BntcHybrid);

  userNTC08.setNTCup(userNTC08pin, nSamples, userNTCRseries);
  userNTC08.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);

  userNTC09.setNTCup(userNTC09pin, nSamples, userNTCRseries);
  userNTC09.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);

  // setup lcd + print welcome message
  lcd.begin(40,2);
  lcd.setCursor(10,0);
  lcd.print(F("Temperature Interlock"));// 21 chars 
  lcd.setBacklight(HIGH);
  lcd.setCursor(6,1);
  lcd.print(F("firmware: "));// 10 chars
  lcd.setCursor(16,1);
  lcd.print(fwVersion);
  lcd.setCursor(19,1);
  lcd.print(' ');
  lcd.setCursor(20,1);
  lcd.print(fwDate);
  delay(2000);
  lcd.clear();
}


void loop(){
  float hybridTemp[8];// keeping this fixed length is lots faster
  float userTemp08 = 20.0;
  float userTemp09 = 20.0;

  //byte count = 0;
  // readout loop for hybrid NTCs
  // set series R according to hybrid type. 
  // For 1 kOhm, drive selectRseries line low during mux'ed measurement to avoid dissipation when idle
  if(hybridNTCRnom == 1) digitalWrite(selectRseries, LOW);

  for(byte count=0; count < 8; count++){
    if(count < nHybridNTCs){
      digitalWrite(muxCtrl0, (bitRead(count,0) ) );
      digitalWrite(muxCtrl1, (bitRead(count,1) ) );
      digitalWrite(muxCtrl2, (bitRead(count,2) ) );

      // read out temperature
      hybridTemp[count] = hybridNTC.getDegC();

      if(hybridTemp[count] > hybridTHi || hybridTemp[count] < hybridTLo){ 
        bitWrite(hybridTnotOK, count, HIGH);
      }
      else{
        bitWrite(hybridTnotOK, count, LOW);
      }
    }
    else{
      bitWrite(hybridTnotOK, count, LOW);
    }
  }
  digitalWrite(selectRseries, HIGH);

  // read out user NTCs if enabled
  // check temperature readings against limits and set bits in userTnotOK accordingly
  if(nUserNTCs > 0){
    userTemp08 = userNTC08.getDegC();
    if(userTemp08 > userNTCTHi || userTemp08 < userNTCTLo) bitWrite(userTnotOK, 0, HIGH);
    else bitWrite(userTnotOK, 0, LOW);

    if(nUserNTCs > 1){ 
      userTemp09 = userNTC09.getDegC();
      if(userTemp09 > userNTCTHi || userTemp09 < userNTCTLo) bitWrite(userTnotOK, 1, HIGH);
      else bitWrite(userTnotOK, 1, LOW);

    }
  }
  else userTnotOK = 0x00;

  // read enable pushbutton pin, see if Temp. is OK and set outputState accordingly
  userEnable = digitalRead(pbEnable);

  if(hybridTnotOK == 0 && nHybridNTCs > 0 && userTnotOK == 0){// enabled NTCs Temperatures are all OK
    digitalWrite(duoLed, HIGH);// T indicator LEDs go green
    if(userEnable == HIGH){// Temperature OK, enable PB pushed
      outputState = HIGH;
    }
  }
  else{
    digitalWrite(duoLed, LOW);
    outputState = LOW;
  }
  digitalWrite(outputEnable, outputState);

  // set enable PB LED state:
  // solid on when T OK and SSR enabled, blink when T OK and SSR disabled. Off when T not OK.
  if(millis()-serialTimerLast > 10000UL){
    serialTimerLast = millis();

    Serial.print("freeMemory():");
    Serial.println( freeMemory() );

    for(byte count = 0; count < 8; count++){
      if(count < nHybridNTCs){ 
        Serial.print(hybridTemp[count],1  );
        Serial.print(',');
      }
      else{
        Serial.print("0.0,");
      }
    }
    if(nUserNTCs > 0){
      Serial.print(userTemp08,1);
    }
    if(nUserNTCs > 1){
      Serial.print(',');
      Serial.print(userTemp09,1);
    }
    Serial.println();
  }

  if(millis()-flashTimerLast > 500UL){
    flashTimerLast = millis();

    if(hybridTnotOK == 0 && nHybridNTCs > 0 && userTnotOK == 0){
      if(outputState == HIGH) enableLedState = HIGH;
      else{
        if(enableLedState == LOW) enableLedState = HIGH;
        else enableLedState = LOW;
      }
    }     
    else enableLedState = LOW;
  }
  digitalWrite(enablePbLED, enableLedState);

  // do menu stuff
  int menuPotVal = analogRead(menuPot);// for use in menu
  // read menu button, debounce, increase menustate if necessary
  int buttonRead = digitalRead(menuButton);

  // check button is pressed, advance to next state
  if(buttonRead != prevButtonRead && buttonRead == HIGH){// leading edge
    delay(20);//require button to be pressed 20ms (= debounce as well)
    if( buttonRead == HIGH ){ 
      menuState = nextMenuState;
    }
  }
  prevButtonRead = buttonRead;

  // +=+=+=+=+=+=+=+=+ MENU STATE MACHINE +=+=+=+=+=+=+=+=+
  // menu state rewrites entire first line. Avoid clearing LCD as it causes flicker
  lcd.home();
  // print Temperature values to lcd on bottom line
  for(byte count = 0; count < 8; count++){
    lcd.setCursor((count*5),1);
    if(count < nHybridNTCs){
      dtostrf(hybridTemp[count], 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else lcd.print(F("     "));
  }
  switch(menuState){
  case 0:
    if(menuPotVal < potClickLo){ 
      lcd.print(F("Tlim. H0-H7:"));// 12 chars max
      lcd.setCursor(12,0);
      dtostrf(hybridTLo, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(15,0);
      dtostrf(hybridTHi, 3,0, dispTemp);
      lcd.print(dispTemp);
      lcd.setCursor(18,0);
      lcd.print(degC);
      lcd.setCursor(20,0);
      lcd.print(F(" U8,U9: "));// 8 chars
      dtostrf(userNTCTLo, 3,0, dispTemp);
      lcd.setCursor(28,0);
      lcd.print(dispTemp);
      dtostrf(userNTCTHi, 3,0, dispTemp);
      lcd.setCursor(31,0);
      lcd.print(dispTemp);
      lcd.setCursor(34,0);
      lcd.print(degC);
      lcd.setCursor(36,0);
      lcd.print(F("    "));
    }
    else if(menuPotVal > potClickHi && menuPotVal < 900){
      //
      if(nUserNTCs == 0) lcd.print(F("User NTC, I2C RH/T disabled             "));
      else{
        if(nUserNTCs > 0){
          lcd.print(F("T U8:"));// 5 chars
          lcd.setCursor(5,0);
          dtostrf(userTemp08, 5,1, dispTemp);
          lcd.print(dispTemp);
          lcd.setCursor(10,0);
          if(nUserNTCs == 1) lcd.print(F("           "));// 11 chars
        }
        if(nUserNTCs > 1){
          lcd.print(F(" T U9:"));// 6 chars
          lcd.setCursor(16,0);
          dtostrf(userTemp09, 5,1, dispTemp);
          lcd.print(dispTemp);
        }
        lcd.setCursor(21,0);
        lcd.print(F("                   "));// 19 chars
      }


    }
    else if(menuPotVal > 900){
      lcd.print(F("U8,U9 Rnom,T(Rnom), B: "));//23 chars
      lcd.setCursor(23,0);
      if(userNTCRnom < 10){
        lcd.print(' ');
        lcd.setCursor(24,0);
      }
      lcd.print(userNTCRnom, DEC);
      lcd.setCursor(25,0);
      lcd.print(kOhm);// 2 chars
      lcd.setCursor(27,0);
      lcd.print(", ");
      lcd.setCursor(29,0);
      lcd.print(userNTCTnom, DEC);
      lcd.setCursor(31,0);
      lcd.print(degC);
      lcd.setCursor(33,0);
      lcd.print(", ");//2
      lcd.setCursor(35,0);
      lcd.print(BUserNTC);
      lcd.setCursor(39,0);
      lcd.print('K');
    }
    else{
      lcd.print(F("|H0  |H1  |H2  |H3  |H4  |H5  |H6  |H7  "));
    }  

    nextMenuState = 1;
    break;
    //    case :
    //    lcd.print("                                        ");
    //    if(menuPotVal < potClickLo){
    //    }
    //    else if(menuPotVal > potClickHi){
    //    }
    //    else{
    //    }
    //    break;
  case 1:
    if(menuPotVal <= 255){
      lcd.print(F("HYBRID_NTC user_ntc  user_rh/t     exit  "));
      nextMenuState = 21;
    }
    else if(menuPotVal > 255 && menuPotVal < 512){
      lcd.print(F("hybrid_ntc USER_NTC  user_rh/t     exit  "));
      nextMenuState = 25;
    }
    else if(menuPotVal > 512 && menuPotVal < 900){
      lcd.print(F("hybrid_ntc user_ntc  USER_RH/T     exit  "));
      nextMenuState = 0;
    }
    else{
      lcd.print(F("hybrid_ntc user_ntc  user_rh/t     EXIT  "));
      nextMenuState = 0;
    }
    break;

  case 21:
    lcd.print(F("hybrid NTC :"));//12 chars
    lcd.setCursor(12,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("T LIMITS  N enable  set Type"));//28 chars
      nextMenuState = 31;//31
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("t limits  N enable  SET TYPE"));
      nextMenuState = 32;//32
    }
    else{
      lcd.print(F("t limits  N ENABLE  set Type"));
      nextMenuState = 33;//33
    }
    break;

  case 25:
    lcd.print(F("user NTCs:"));//10 chars
    lcd.setCursor(10,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("T LIMITS  N enable  set values"));//30 chars
      nextMenuState = 35;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("t limits  N enable  SET VALUES"));
      nextMenuState = 36;
    }
    else{
      lcd.print(F("t limits  N ENABLE  set values"));
      nextMenuState = 37;
    }
    break;

  case 31:
    lcd.print(F("hybrid NTC T limits:"));//20 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("LOW     exit    high"));// 20 chars
      nextMenuState = 41;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("low     exit    HIGH"));
      nextMenuState = 41;
      menuChoice = 2;
    }
    else{
      lcd.print(F("low     EXIT    high"));
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 32:
    lcd.print(F("Hybrid NTC R(25"));//15 chars
    lcd.setCursor(15,0);
    lcd.print(degC);
    lcd.setCursor(17,0);
    lcd.print(F("): "));// 4 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print(F(" 10KOHM  exit  1kohm"));
      menuChoice = 1;
      nextMenuState = 42;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F(" 10kohm  exit  1KOHM"));
      menuChoice = 2;
      nextMenuState = 42;
    }
    else{
      lcd.print(F(" 10kohm  EXIT  1kohm"));
      menuChoice = 0;
      nextMenuState = 0;
    }
    break;

  case 33:
    lcd.print(F("hybrid NTC enable (0..8, exit): "));// 32 chars
    setEnable = map(menuPotVal, 0, 1023, 0, 10);

    lcd.setCursor(32,0);
    if(setEnable >= 9){
      lcd.print(F("    EXIT"));
      nextMenuState = 0;
    }
    else{
      for(byte count = 0; count < 8; count++){
        lcd.setCursor((32+count),0);
        if(count < setEnable) lcd.print('O');
        else lcd.print('X');
      }
      nextMenuState = 43;
    }
    break;

  case 35:
    lcd.print(F("user NTC T limits:  "));//20 chars
    lcd.setCursor(20,0);
    if(menuPotVal < potClickLo){
      lcd.print(F("LOW     exit    high"));// 20 chars
      nextMenuState = 45;
      menuChoice = 1;
    }
    else if(menuPotVal > potClickHi){
      lcd.print(F("low     exit    HIGH"));
      nextMenuState = 45;
      menuChoice = 2;
    }
    else{
      lcd.print(F("low     EXIT    high"));
      nextMenuState = 0;
      menuChoice = 0;
    }
    break;

  case 36:
    lcd.print(F("Dial & push to set nominal NTC R    "));// 36 chars
    setTemp = map(menuPotVal, 0, 1024, 0, 7);
    switch(setTemp){
    case 0:
      userNTCRnom = 1;
      break;
    case 1:
      userNTCRnom = 2;
      break;
    case 3:
      userNTCRnom = 3;
      break;
    case 4:
      userNTCRnom = 5;
      break;
    case 5:
      userNTCRnom = 10;
      break;
    case 6:
      userNTCRnom = 20;
      break;
    }
    lcd.setCursor(36,0);
    if(userNTCRnom < 10){
      lcd.print(' ');
      lcd.setCursor(37,0);
    }
    lcd.print(userNTCRnom, DEC);
    lcd.setCursor(38,0);
    lcd.print(kOhm);
    nextMenuState = 46;
    break;


  case 37:
    lcd.print(F("user NTC enable (0..2, exit): "));// 30 chars
    lcd.setCursor(30,0);
    if(menuPotVal < 255){
      lcd.print(F("        XX"));
      setEnable = 0;
      nextMenuState = 47;
    }
    else if(menuPotVal > 254 && menuPotVal < potClickHi){
      lcd.print(F("        OX"));
      setEnable = 1;
      nextMenuState = 47;
    }
    else if(menuPotVal > potClickHi && menuPotVal < 800){
      lcd.print(F("        OO"));
      setEnable = 2;
      nextMenuState = 47;
    }
    else{
      lcd.print(F("      EXIT"));
      nextMenuState = 0;
    }
    break;

  case 41:
    lcd.print(F("Dial & push to set Temperature:  "));//35 chars
    setTemp = map(menuPotVal, 0, 1023, -25, 65);
    lcd.setCursor(35,0);
    dtostrf(setTemp, 3, 0, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(38,0);
    lcd.print(degC);
    nextMenuState = 51;
    break;

  case 42:
    lcd.print(F("Hybrid NTC R(25"));//15 chars 
    lcd.setCursor(15,0);
    lcd.print(degC);
    lcd.setCursor(17,0);
    lcd.print(F("):                "));//17 chars, reaching up to 35
    lcd.setCursor(35,0);
    if(menuChoice == 1){
      hybridNTCRnom = 10;
      lcd.print(F("10 "));
    }
    else if(menuChoice == 2){
      lcd.print(F(" 1 "));
      hybridNTCRnom = 1;
    }
    lcd.setCursor(38,0);
    lcd.print(kOhm);


    EEPROM_writeAnything(addrHybridNTCRnom, hybridNTCRnom);
    // re-initialise NTC instances
    hybridNTC.setNTCup(muxNTC, nSamples, hybridNTCRnom);
    hybridNTC.setNTCParameters(hybridNTCRnom, TrefHybrid, BntcHybrid);
    nextMenuState = 32;
    break;

  case 43:
    nHybridNTCs = setEnable;
    EEPROM_writeAnything(addrHybridEnableMask, setEnable);
    lcd.print(F("hybrid NTC enable mask set to:  "));// 32 chars
    for(byte count = 0; count < 8; count++){
      lcd.setCursor((32+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 33;
    break;

  case 45:
    lcd.print(F("Dial & push to set Temperature:    "));//35 chars
    setTemp = map(menuPotVal, 0, 1024, -25.0, 75.0);
    lcd.setCursor(35,0);
    dtostrf(setTemp, 3, 0, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(38,0);
    lcd.print(degC);
    nextMenuState = 55;
    break;

  case 46:
    lcd.print(F("Nominal user NTC R set to: "));//27 chars 
    lcd.setCursor(27,0);
    if(userNTCRnom < 10){
      lcd.print(' ');
      lcd.setCursor(28,0);
    }
    lcd.print(userNTCRnom, DEC);
    lcd.setCursor(29,0);
    lcd.print(kOhm);
    lcd.setCursor(31,0);
    lcd.print(F("         "));
    EEPROM_writeAnything(addrUserNTCRnom, userNTCRnom);
    nextMenuState = 56;
    break;

  case 47:
    nUserNTCs = setEnable;
    EEPROM_writeAnything(addrUserNTCEnableMask, setEnable);
    lcd.print(F("User NTC enable mask set to:          "));// 38 chars
    for(byte count = 0; count < 2; count++){
      lcd.setCursor((38+count),0);
      if(count < setEnable) lcd.print('O');
      else lcd.print('X');
    }
    nextMenuState = 37;
    break;

  case 51:
    if(menuChoice == 1){// low T limit
      if(float(setTemp) > hybridTHi){
        lcd.print(F("Temperature not set: Low > High limit   "));
      }
      else{
        hybridTLo = float(setTemp);
        EEPROM_writeAnything(addrHybridTLo, hybridTLo);
        lcd.print(F("Low Temperature limit set to:      "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else if(menuChoice == 2){
      if(float(setTemp) < hybridTLo){
        lcd.print(F("Temperature not set: High < Low limit   "));
      }
      else{
        hybridTHi = float(setTemp);
        EEPROM_writeAnything(addrHybridTHi, hybridTHi);
        lcd.print(F("High Temperature limit set to:     "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else{
      lcd.print(F("Error: temperature limit not set        "));
    }
    nextMenuState = 31;
    break;

  case 55:
    if(menuChoice == 1){// low T limit
      if(float(setTemp) > userNTCTHi){
        lcd.print(F("Temperature not set: Low > High limit   "));
      }
      else{
        userNTCTLo = (float)setTemp;
        EEPROM_writeAnything(addrUserNTCTLo, userNTCTLo);
        lcd.print(F("Low Temperature limit set to:      "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else if(menuChoice == 2){
      if(float(setTemp) < userNTCTLo){
        lcd.print(F("Temperature not set: High < Low limit   "));
      }
      else{
        userNTCTHi = (float)setTemp;
        EEPROM_writeAnything(addrUserNTCTHi, userNTCTHi);
        lcd.print(F("High Temperature limit set to:     "));
        lcd.setCursor(35,0);
        dtostrf(setTemp, 3, 0, dispTemp);
        lcd.print(dispTemp);
        lcd.setCursor(38,0);
        lcd.print(degC);
      }
    }
    else{
      lcd.print(F("Error: temperature limit not set        "));
    }
    nextMenuState = 35;
    break;

  case 56:
    lcd.print(F("Dial & push to set NTC T(Rnom):     "));// 36 chars
    lcd.setCursor(36,0);
    if(menuPotVal < potClickHi){
      lcd.print(F("20"));
      userNTCTnom = 20;
    }
    else{
      lcd.print(F("25"));// deg C");
      userNTCTnom = 25;
    }
    lcd.setCursor(38,0);
    lcd.print(degC);
    //EEPROM_writeAnything(addrUserNTCTnom, userNTCTnom);
    nextMenuState = 66;
    break;

  case 66:
    lcd.print(F("NTC Temperature (Rnominal) set to:  "));//36 chars
    lcd.setCursor(36,0);
    lcd.print(userNTCTnom);
    lcd.setCursor(38,0);
    lcd.print(degC);
    EEPROM_writeAnything(addrUserNTCTnom, userNTCTnom);
    nextMenuState = 76;
    break;

  case 76:
    lcd.print(F("Dial & push to set NTC B value:    "));//34 chars
    //BUserNTC = map(menuPotVal, 0, 1024, 3100, 3500);
    BUserNTC = 5*(map(menuPotVal, 0, 1024, 600, 800));// BUserNTC: 3000-4000 steps of 5
    lcd.setCursor(34,0);
    lcd.print(BUserNTC); 
    lcd.setCursor(38,0);
    lcd.print(" K");  
    nextMenuState = 86;
    break;

  case 86:
    lcd.print(F("NTC 08, 09 B value set to:        "));// 34 chars
    lcd.setCursor(34,0);
    lcd.print(BUserNTC);
    lcd.setCursor(38,0);
    lcd.print(" K");
    EEPROM_writeAnything(addrUserNTCBntc, BUserNTC);
    userNTC08.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);
    userNTC09.setNTCParameters(userNTCRnom, userNTCTnom, BUserNTC);
    nextMenuState = 1;
  }
  
}// loop()

// FUNCTIONS

// clear lcd line
void lcdClearLine(int lineNum){
  if(lineNum == 0 || lineNum == 1){
    lcd.setCursor(0,lineNum);
    lcd.print(F("                                        "));
  }
  return;
}


/* Calculates the size of the free list */
int freeListSize() {
  struct __freelist* current;
  int total = 0;

  for (current = __flp; current; current = current->nx) {
    total += 2; /* Add two bytes for the memory block's header  */
    total += (int) current->sz;
  }

  return total;
}

int freeMemory() {
  int free_memory;

  if ((int)__brkval == 0) {
    free_memory = ((int)&free_memory) - ((int)&__heap_start);
  } 
  else {
    free_memory = ((int)&free_memory) - ((int)__brkval);
    free_memory += freeListSize();
  }
  return free_memory;
}





























































