// Menu structure demonstrator for interlock-II

// Bart Hommels, University of Cambridge
// 11/05/2012 first version

// hardware:
// SPI LCD interface through 74LS595 shift register
// compatible with LCD serial backpack:
// http://www.ladyada.net/products/i2cspilcdbackpack/index.html
//
// Digital 4 is LATCH (p12 on 74LS595)
// Digital 5 is CLK (p11 on 74LS595)
// Digital 6 is DATA (p14 on 74LS595)
//

// include libraries
#include "Wire.h"// for serial LCD
#include "LiquidCrystal.h"// for LCD 
#include <EEPROM.h>// for EEPROM
#include "EEPROMAnything.h"

LiquidCrystal lcd(6,5,4);

// pin definitions
const int menuPotPin = A3;
const int ledPin = 7;
const int menuButton = 3;

// timer value
unsigned long timerLast = 0;

int lastMenuPotVal = 0;
int saveMenuPotVal = 0;
int savePotVal = 0;
const int potClickLo = 500;
const int potClickHi = 560;

int buttonRead = 0;
int prevButtonRead = 0;
int menuState = 0;
int prevMenuState = 0;
int menuPrint = 0;

int addrPotVal = 0;

void setup(){
  pinMode(ledPin, OUTPUT);

  // setup lcd
  lcd.begin(40,2);
  lcd.setCursor(11,0);
  lcd.setBacklight(LOW);
  lcd.print("Menu demonstrator");
  delay(500);
  lcd.setBacklight(HIGH);
}

void loop(){
  unsigned long timerCurrent = millis();

  // read menupot
  int menuPotVal = analogRead(menuPotPin);

  // read menu button, debounce, increase menustate if necessary
  int buttonRead = digitalRead(menuButton);
  if(buttonRead != prevButtonRead && buttonRead == HIGH){// leading edge
    delay(50);//debounce 50ms
    if( buttonRead == HIGH ){
      saveMenuPotVal = menuPotVal;
      menuState += 1;
      menuPrint = LOW;
    }
  }
  prevButtonRead = buttonRead;
  if(menuState == 4) menuState = 0;

  if(menuPrint == 0){
    lcd.setCursor(39,1);
    lcd.print(menuState);

    // case statement to drive one-off functions according to menu state
    switch (menuState){
    case 0:
      lcdClearLine(0);
      lcd.setCursor(11,0);
      lcd.print("Menu demonstrator");
      break;
    case 1:
      savePotVal = menuPotVal;
      lcdClearLine(0);
      lcd.setCursor(10,0);
      lcd.print("Current value:");
      lcd.setCursor(25,0);
      lcd.print(savePotVal);
      break;
    case 2:
      if(saveMenuPotVal > potClickHi){
        savePotVal = EEPROM.read(addrPotVal);
        lcdClearLine(0);
        lcd.setCursor(0,0);
        lcd.print("Read value from EEPROM: ");
        lcd.print(savePotVal);
      }
      else if(saveMenuPotVal < potClickLo) {
        EEPROM.write(addrPotVal, savePotVal);
        lcdClearLine(0);
        lcd.setCursor(0,0);
        lcd.print("Saved value to EEPROM: ");
        lcd.print(savePotVal);
      }
      break;
    case 3:
      lcdClearLine(0);
      lcd.setCursor(6,0);
      lcd.print("Press menu button to return");
      break;
    }
    menuPrint = 1;
  }
  else{
    // case statement defining repeating menuState dependent functions
    switch(menuState){
    case 1:
      if(menuPotVal < potClickLo){
        lcd.setCursor(1,0);
        lcd.print("SAVE"); 
        lcd.setCursor(35,0);
        lcd.print("read");
      }
      else if(menuPotVal > potClickHi){
        lcd.setCursor(1,0);
        lcd.print("save"); 
        lcd.setCursor(35,0);
        lcd.print("READ");
      }
      else{
        lcd.setCursor(1,0);
        lcd.print("save"); 
        lcd.setCursor(35,0); 
        lcd.print("read");
      }
      break;
      
    }
  }

  // asynchronous functions that may depend on the menu state

  if(abs(menuPotVal-lastMenuPotVal) > 2){
    lcd.setCursor(10,1);
    lcd.print("          ");
    lcd.setCursor(13,1);
    lcd.print(menuPotVal);
    lastMenuPotVal = menuPotVal;
  }

  if(menuState == 0) digitalWrite(ledPin, HIGH);
  else digitalWrite(ledPin, LOW);

}

// clear lcd line
void lcdClearLine(int lineNum){
  if(lineNum == 0 || lineNum == 1){
    lcd.setCursor(0,lineNum);
    lcd.print("                                        ");
  }
  return;
}
















