// sketch for ATLAS Upgrade Interlock II

// Bart Hommels
// University of Cambridge

// 16/7/2012
// V 0.1

//+-------------------------------------------------------+
// Libraries
//#include <stdlib.h>
#include "Wire.h"// for driving SPI bus and/or I2C/onewire
#include "LiquidCrystal.h"// for LCD on SPI
#include <EEPROM.h>// for using EEPROM
#include "EEPROMAnything.h" // for writing multiple-byte objects to EEPROM
#include <NTCreadout.h>

//+-------------------------------------------------------+
// pin assignments
const int muxNTC = A0;
const int userNTC09 = A1;
const int userNTC08 = A2;
const int menuPot = A3;

const int muxCtrl0 = 0;
const int muxCtrl1 = 1;
const int muxCtrl2 = 2;
const int menuButton = 3;
const int lcdLatch = 4;
const int lcdClk = 5;
const int lcdData = 6;
const int duoLed = 7;

const int outputEnable = 8;
const int selectRseries = 9;
const int d0_output = 10;
const int d1_output = 11;
const int singleLed = 12;
const int pbEnable = 13;

//+-------------------------------------------------------+
// constants
const float absTzero = -273.15;// 0 Kelvin in deg C

// 10kOhm hybrid NTCs
const float RseriesHybrid = 10000.0;
const float RrefHybrid = 2000.0;
const float TrefHybrid = 20.0-absTzero;
const float BntcHybrid = 3125.0;

// GL23
//const float RseriesNTC09 = 10000.0;
//const float RrefNTC09 = 2000.0;
//const float TrefNTC09 = 20.0-absTzero;
//const float BntcNTC09 = 3125.0;

//+-------------------------------------------------------+
// variables
float hybridTHi = 30.0;// deg C. Extremely conservative fallback, should be collected from EEPROM
float hybridTLo = 10.0;//deg C. Extremely conservative fallback, should be collected from EEPROM
float newTLo = -25.0;//deg C

unsigned int addrHybridTLo = 0;
unsigned int addrHybridTHi = 4;

int nSamples = 2;
byte nHybrids = 8;

char dispTemp[5];
// menu pot limits and values
const int potClickLo = 470;
const int potClickHi = 560;

int lastMenuPotVal = 0;
int saveMenuPotVal = 0;
int savePotVal = 0;

// menu state machine variables
byte buttonRead = 0;
byte prevButtonRead = 0;
byte menuState = 0;
byte nextMenuState = 0;
byte prevMenuState = 0;
byte menuRePrint = 0;
byte menuChoice = 0;
byte menuPrevChoice = 0;

int addrPotVal = 0;

float setTemp = 0.0;

//+-------------------------------------------------------+
// instantiations

// SPI LCD interface through 74LS595 shift register, see:
// http://www.ladyada.net/products/i2cspilcdbackpack/index.html
// DATA (p14 on 74LS595), CLK (p11 on 74LS595), LATCH (p12 on 74LS595)
LiquidCrystal lcd(6,5,4);

// NTC serving MUXed input, user08, user09
NTCtemp hybridNTC;
byte hybridTnotOK = 0x00;
//NTCtemp NTC08;`
//NTCtemp NTC09;

//+-------------------------------------------------------+
void setup(){
  // set mode for all remaining pins
  pinMode(muxCtrl0, OUTPUT);
  pinMode(muxCtrl1, OUTPUT);
  pinMode(muxCtrl2, OUTPUT);
  pinMode(menuButton, INPUT);
  pinMode(duoLed, OUTPUT);
  pinMode(outputEnable, OUTPUT);
  pinMode(selectRseries, OUTPUT);
  pinMode(d0_output, OUTPUT);
  pinMode(d1_output, OUTPUT);
  pinMode(singleLed, OUTPUT);
  pinMode(pbEnable, INPUT);

  // write default outputs to pins
  digitalWrite(muxCtrl0, LOW);
  digitalWrite(muxCtrl1, LOW);
  digitalWrite(muxCtrl2, LOW);

  // use external analog reference
  //analogReference(EXTERNAL);

  // set variables
  menuState = 0;
  nextMenuState = 0;

  // retrieve variables from EEPROM
  EEPROM_readAnything(addrHybridTLo, hybridTLo);
  EEPROM_readAnything(addrHybridTHi, hybridTHi);

  // initialise NTC instances
  hybridNTC.setNTCup(muxNTC, nSamples, RseriesHybrid);
  hybridNTC.setNTCParameters(RrefHybrid, TrefHybrid, BntcHybrid);

  //  NTC09.setNTC(userNTC09, nSamples, RseriesNTC09);
  //  NTC.setNTCParameters(RrefNTC09, TrefNTC09, BntcNTC09);

  // setup lcd + print welcome message
  lcd.begin(40,2);
  lcd.setCursor(11,0);
  lcd.print("Menu demonstrator v0.1  18/07/2012");
  lcd.setBacklight(HIGH);
  lcd.setCursor(0,1);
  lcd.print("TLimits Lo, Hi:    ");
  lcd.setCursor(20,1);
  lcd.print(hybridTLo);
  lcd.setCursor(30,1);
  lcd.print(hybridTHi);

  delay(500);

  lcd.clear();
  lcd.setCursor(0,0);
  lcd.print("Temperature");
}


void loop(){
  float hybridTemp[8];// keeping this fixed length is lots faster
  byte count = 0;

  //  unsigned long timerlast = millis();

  for(count=0; count < nHybrids; count++){
    digitalWrite(muxCtrl0, (bitRead(count,0) ) );
    digitalWrite(muxCtrl1, (bitRead(count,1) ) );
    digitalWrite(muxCtrl2, (bitRead(count,2) ) );
    hybridTemp[count] = hybridNTC.getDegC();

    if(hybridTemp[count] > hybridTHi || hybridTemp[count] < hybridTLo){ 
      bitWrite(hybridTnotOK, count, HIGH);
    }
    else{
      bitWrite(hybridTnotOK, count, LOW);
    }
  }

  //  unsigned long t_diff = millis()-timerlast;

  if(hybridTnotOK == 0){
    digitalWrite(duoLed, LOW);
  }
  else{
    digitalWrite(duoLed, HIGH);
  }

  // print Temperature values to lcd
  for(count = 0; count < nHybrids; count++){
    dtostrf(hybridTemp[count], 5, 1, dispTemp);
    lcd.setCursor((count*5),1);
    lcd.print(dispTemp);
  }

  // print timing information
  //lcd.setCursor(38,0);
  //lcd.print(t_diff);

  // do menu stuff
  int menuPotVal = analogRead(menuPot);// for use in menu
  // read menu button, debounce, increase menustate if necessary
  int buttonRead = digitalRead(menuButton);

  // check button is pressed, advance to next state
  if(buttonRead != prevButtonRead && buttonRead == HIGH){// leading edge
    delay(50);//require button to be pressed 50ms (= debounce as well)
    if( buttonRead == HIGH ){
      saveMenuPotVal = menuPotVal;//??
      menuState = nextMenuState;
      menuRePrint = LOW;
    }
  }
  prevButtonRead = buttonRead;

  // +=+=+=+=+=+=+=+=+ MENU STATE MACHINE +=+=+=+=+=+=+=+=+
  // menu state must rewrite entire first line. Avoid clearing LCD as it causes flicker
  //lcdClearLine(0);
  lcd.home();
  switch(menuState){
  case 0:
    lcd.print("H0-H7 T Limits: ");// 16 chars
    lcd.setCursor(16,0);
    dtostrf(hybridTLo, 5,1, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(21,0);
    lcd.print(" ");
    lcd.setCursor(22,0);
    dtostrf(hybridTHi, 5,1, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(27,0);
    lcd.print("   ");
    lcd.setCursor(30,0);
    if(menuPotVal > potClickHi){
      lcd.print("MENU   ");// is 7 now, should become 10 chars long
      nextMenuState = 1;
    }
    else{
      lcd.print("menu   ");
      nextMenuState = 0;
    }
    break;
  case 1:
    lcd.print("Set Temp. Limit:    ");//20 chars
    lcd.setCursor(20,0);

    if(menuPotVal < potClickLo){
      menuChoice = 1;// Low T limit
      nextMenuState = 2;
      lcd.print("LOW   cancel high");// is 17 now, should become 20 chars long
    }
    else if(menuPotVal > potClickHi){
      menuChoice = 2;// High T limit
      nextMenuState = 2;
      lcd.print("low   cancel HIGH");
    }
    else{
      nextMenuState = 0;
      lcd.print("low   CANCEL high");
    }
    break;
  case 2:
    lcd.print("Push to set Temp:   ");//20 chars
    setTemp = map(menuPotVal, 0, 1024, -25.0, 65.0);

    lcd.setCursor(20,0);
    dtostrf(setTemp, 5,1, dispTemp);
    lcd.print(dispTemp);
    lcd.setCursor(25,0);
    lcd.print("               ");
    nextMenuState = 3;
    break;
  case 3:
    if(menuChoice == 1){// low T limit
      hybridTLo = setTemp;
      EEPROM_writeAnything(addrHybridTLo, setTemp);
      lcd.print("Low Temp. limit set to:       ");
      lcd.setCursor(30,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else if(menuChoice == 2){
      hybridTHi = setTemp;
      EEPROM_writeAnything(addrHybridTHi, setTemp);
      lcd.print("High Temp. limit set to:    ");
      lcd.setCursor(30,0);
      dtostrf(setTemp, 5, 1, dispTemp);
      lcd.print(dispTemp);
    }
    else{
      lcd.print("Error: temperature limit not set");
    }
    nextMenuState = 1;
  }

  lcd.setCursor(37,0);
  lcd.print(menuChoice, DEC);
  lcd.setCursor(38,0);
  lcd.print(nextMenuState, DEC);
  lcd.setCursor(39,0);
  lcd.print(menuState, DEC);
}

// FUNCTIONS

// clear lcd line
void lcdClearLine(int lineNum){
  if(lineNum == 0 || lineNum == 1){
    lcd.setCursor(0,lineNum);
    lcd.print("                                        ");
  }
  return;
}




















