// -*- C++ -*-
#ifndef HERWIG_SudakovFormFactor_H
#define HERWIG_SudakovFormFactor_H
//
// This is the declaration of the <!id>SudakovFormFactor<!!id> class.
//
// CLASSDOC SUBSECTION Description:
//
// This is the abstract class from which all different types of <BR>
// Sudakov form factors derive from.
//
// CLASSDOC SUBSECTION See also:
//
// <a href="http:SplitFun.html">SplitFun.h</a>, <BR>
// <a href="http:ShowerAlpha.html">ShowerAlpha.h</a>. <BR>
// <a href="http:SplittingGenerator.html">SplittingGenerator.h</a>, <BR>
// <a href="http:QtoQGSudakovFormFactor.html">QtoQGSudakovFormFactor.h</a>, <BR>
// <a href="http:QtoQGammaSudakovFormFactor.html">QtoQGammaSudakovFormFactor.h</a>, <BR>
// <a href="http:GtoGGSudakovFormFactor.html">GtoGGSudakovFormFactor.h</a>, <BR>
// <a href="http:GtoQQbarSudakovFormFactor.html">GtoQQbarSudakovFormFactor.h</a>.
// 

#include "ShowerConfig.h"
#include "ThePEG/Pointer/Ptr.h"
#include "ThePEG/Pointer/ReferenceCounted.h"
#include "ThePEG/Pointer/PtrTraits.h"
#include "ThePEG/Pointer/RCPtr.h"
#include "Herwig++/Utilities/GlobalParameters.h"
#include "SplitFun.h"
#include "SplitFun1to2.h"
#include "ShowerAlpha.h"


namespace Herwig {

using namespace ThePEG;


class SudakovFormFactor: public ReferenceCounted {

public:

  inline SudakovFormFactor();
  inline SudakovFormFactor(const SudakovFormFactor &);
  virtual ~SudakovFormFactor();
  // Standard ctors and dtor.

  inline SudakovFormFactor( const SplitFunPtr inputSplitFun, 
			    const tShowerAlphaPtr inputShowerAlpha,
                            const Energy inputMinScale, const Energy inputMaxScale, 
			    const Energy inQ0);

  virtual Energy generateNextBranching( tPartCollHdlPtr ch, 
					const Energy startingScale,
					const bool reverseAngularOrder = false ) = 0;
  // Pure virtual method, to be defined in concrete derived classes.
  // It returns the scale of the next branching; if there is no 
  // branching then it returns Energy().
  // The <!id>ch<!!id> argument is used only for Initial State branching,
  // to get access to the PDFs; the <!id>reverseOrdering<!!id> is used 
  // (when it is not equal to the default, false, value) only for 
  // Final State branching of a decaying on-shell particle. 

  virtual void setupLookupTables();
  // This virtual method is defined as empty, and it should be
  // overriden only for those derived Sudakov form factor classes
  // that use lookup tables for numerical evaluations, rather
  // than using the Monte Carlo rejection (veto) method.
  // This method is called once, during initialization, by
  // the <!class>SplittingGenerator<!!class>. 
  // General methods, usable for any type of Sudakov form factor
  // that override this method, should be provided in this
  // class in the protected session.

  inline tSplitFunPtr splitFun() const;
  // It returns the pointer to the <!class>SplitFun<!!class> object.

  inline tShowerAlphaPtr alpha() const;
  // It returns the pointer to the <!class>ShowerAlpha<!!class> object.

  inline double z() const;
  inline double phi() const;
  inline Energy qtilde() const; 
  inline Energy resScale() const; 
  inline Energy kinScale() const; 
  // These methods returns other shower kinematics variables, kept
  // internally to this class, that have being generated by a call
  // to <!id>generateNextBranching<!!id> together with the energy
  // scale (which is the returned value of such method).
  // Notice that these variables cannot be set directly from the extern 
  // (outside this class) but only via a call to the method
  // <!id>generateNextBranching<!!id>, whose first action is
  // to reset them, to clear the values they got in the previous
  // call to the same method. In other words, the lifetime
  // of the values of these kinematics variables is between
  // to successive call to <!id>generateNextBranching<!!id>.
  // Finally, notice that at the moment these variables are meaninful
  // only for a 1->2 splitting, but in future other variables
  // could be added as well for describing also a 1->3 splitting.
 
protected:

  Energy _q;
  double _z;
  double _phi;
  // Internal variables to keep shower kinematics information
  // generated by a call to <!id>generateNextBranching<!!id>.

  //***LOOKHERE*** define eventually here some methods which are useful
  //               to compute numerically the sudakov form factor,
  //               in a very general way, independently from the
  //               initial state / final state and from the
  //               kind and multiplicity of the vertex.
  //               I am not sure if this is possible: maybe for
  //               all 1-&gt;2 splitfun, but not probably for 1-&gt;3.
  //               eventually assume 1-&gt;2 for this default case.
  //               notice that is important to define these methods
  //               in the protected part because they should be
  //               used only by the derived classes.

  // toy model: returns q according to powerlike (q^p) distribution
  // with cutoff q0, qmin < q0 < qmax.  qmin is chosen such that the
  // probability for a first branching is 1-R.  z is chosen from
  // 1/(1-z) with z0 = m/q and z0 < z < 1-z0; or (if znorm==false)
  // flatter, as z^2+(1-z)^2 with z0 < z < 1.
  void get_qz (bool znorm, double p, double R, Energy q0, Energy qmax, Energy &q, double &z);

  // three utility methods, very similar to those of the test
  // programs.  They do what their name sais and are adopted to the
  // abstract nature of SplitFun as well.
  double guessz (double z0, double z1);
  Energy2 guesst (Energy2 t0, Energy2 t1, double z0, double z1);
  void gettz (Energy2 tmax, Energy2 &t, double &z);

private:

  SudakovFormFactor & operator=(const SudakovFormFactor &);
  //  Private and non-existent assignment operator.

  SplitFunPtr _splitFun;
  tShowerAlphaPtr _alpha;
  Energy _minScale;
  Energy _maxScale;
  Energy _kinQ0;

  //***LOOKHERE*** define eventual here some data structure which is useful
  //               to implement the general methods for the numeric
  //               evaluation which are defined in the protected part.

};

}

#include "SudakovFormFactor.icc"

#endif /* HERWIG_SudakovFormFactor_H */
