#include "EvtShapes.h"

using namespace Herwig;
using namespace ThePEG;

// live and die

EvtShapes::EvtShapes() {}
EvtShapes::~EvtShapes() {}

EvtShapes::EvtShapes(const tPVector& part) {
  _pv = part; 
  _linTenDone = false; 
  _spherDone = false; 
  _thrustDone = false; 
  _hemDone = false; 
  _broadDone = false; 
  _useCmBoost = false; 
}

// some convenient single particle variables
double EvtShapes::getX(const Lorentz5Momentum & p, 
			      const Energy & Ebeam) {
  return(Ebeam > 0 ? p.vect().mag()/Ebeam : -1.); 
}

double EvtShapes::getXi(const Lorentz5Momentum & p, 
			       const Energy & Ebeam) {
  return((Ebeam > 0 && p.vect().mag() > 0) ? 
	 log(Ebeam/p.vect().mag()) : -1.); 
}

Energy EvtShapes::getPt(const Lorentz5Momentum & p) {
  return p.perp(); 
}

Energy EvtShapes::getRapidity(const Lorentz5Momentum & p) {
  if (p.e() > p.z()) return p.rapidity(); 
  else return 1000.0; 
}

// single particle variables related to shape axis
Energy EvtShapes::ptInT(const Lorentz5Momentum & p) {
  checkThrust(); 
  return p.vect()*_thrustAxis[1]; 
}

Energy EvtShapes::ptOutT(const Lorentz5Momentum & p) {
  checkThrust(); 
  return p.vect()*_thrustAxis[2]; 
}

double EvtShapes::yT(const Lorentz5Momentum & p) {
  checkThrust(); 
  if (p.e() > p.z()) return p.rapidity(_thrustAxis[0]);
  else return 1000.0; 
}

Energy EvtShapes::ptInS(const Lorentz5Momentum & p) { 
  checkSphericity(); 
  return p.vect()*_spherAxis[1]; 
}

Energy EvtShapes::ptOutS(const Lorentz5Momentum & p) {
  checkSphericity(); 
  return p.vect()*_spherAxis[2]; 
}

double EvtShapes::yS(const Lorentz5Momentum & p) {
  checkSphericity(); 
  if (p.e() > p.z()) return p.rapidity(_spherAxis[0]);
  else return 1000.0;
}

// thrust related 
double EvtShapes::thrust() {
  checkThrust(); 
  return _thrust[0];
}

double EvtShapes::thrustMajor() {
  checkThrust(); 
  return _thrust[1];
}

double EvtShapes::thrustMinor() {
  checkThrust(); 
  return _thrust[2];
}

double EvtShapes::oblateness() {
  checkThrust(); 
  return _thrust[1]-_thrust[2];
}

Vector3 EvtShapes::thrustAxis() {
  checkThrust(); 
  return _thrustAxis[0];
}

Vector3 EvtShapes::majorAxis() {
  checkThrust(); 
  return _thrustAxis[1];
}

Vector3 EvtShapes::minorAxis() {
  checkThrust(); 
  return _thrustAxis[2];
}

// linear tensor related
vector<double> EvtShapes::linTenEigenValues() {
  checkLinTen(); 
  return _linTen; 
}

vector<Vector3> EvtShapes::linTenEigenVectors() {
  checkLinTen(); 
  return _linTenAxis; 
}

double EvtShapes::CParameter() {
  checkLinTen(); 
  return 3.*(_linTen[0]*_linTen[1]+_linTen[1]*_linTen[2]
	     +_linTen[2]*_linTen[0]); 
}

double EvtShapes::DParameter() {
  checkLinTen(); 
  return 27.*(_linTen[0]*_linTen[1]*_linTen[2]); 
}

// quadratic tensor related
double EvtShapes::sphericity() {
  checkSphericity(); 
  return 3./2.*(_spher[1]+_spher[2]); 
}

double EvtShapes::aplanarity() {
  checkSphericity(); 
  return 3./2.*_spher[2];
}

double EvtShapes::planarity() {
  checkSphericity(); 
  return _spher[1]-_spher[2]; 
}

Vector3 EvtShapes::sphericityAxis() {
  checkSphericity(); 
  return _spherAxis[0]; 
}

vector<double> EvtShapes::sphericityEigenValues() {
  checkSphericity(); 
  return _spher; 
}

vector<Vector3> EvtShapes::sphericityEigenVectors() {
  checkSphericity(); 
  return _spherAxis; 
}

// jet mass related
double EvtShapes::Mhigh2() {
  checkHemispheres();
  return _mPlus; 
} 

double EvtShapes::Mlow2() {
  checkHemispheres();
  return _mMinus; 
} 

double EvtShapes::Mdiff2() {
  checkHemispheres();
  return _mPlus-_mMinus; 
} 

// jet broadening
double EvtShapes::Bmax() {
  checkBroadening(); 
  return _bPlus;
}

double EvtShapes::Bmin() {
  checkBroadening(); 
  return _bMinus;
}

double EvtShapes::Bsum() {
  checkBroadening(); 
  return _bPlus+_bMinus;
}

double EvtShapes::Bdiff() {
  checkBroadening(); 
  return _bPlus-_bMinus;
}

void EvtShapes::normalizeEEC(vector<double> & hi, long evts) {
  for (int bin = 0; bin < hi.size(); bin++) 
    bin /= (hi.size()*evts);
}

void EvtShapes::bookEEC(vector<double> & hi) {
  // hi is the histogram.  It is understood that hi.front() contains
  // the bin [-1 < cos(chi) < -1+delta] and hi.back() the bin [1-delta
  // < cos(chi) < 1].  Here, delta = 2/hi.size().
  Energy Evis;
  Lorentz5Momentum p_i, p_j; 
  for (int bin = 0; bin < hi.size(); bin++) {
    double delta = 2./((double) hi.size());
    double coschi = -1+bin*delta;
    if (_pv.size() > 1) {
      for (int i = 0; i < _pv.size()-1; i++) {
	p_i = _pv[i]->momentum();
	Evis += p_i.e(); 
	for (int j = i+1; j < _pv.size(); j++) {
	  p_j = _pv[j]->momentum();
	  double diff = abs(coschi-cos( p_i.vect().angle(p_j.vect()) )); 
	  if (delta > diff) 
	    hi[bin] += p_i.e()*p_j.e();
	}
      }
    }
    hi[bin] /= (Evis*Evis);
  }
}

double EvtShapes::AEEC(vector<double> & hi, double& coschi) {
  if (coschi > 0. && coschi <= 1.) {
    int i = (int) floor((-coschi+1.)/2.*hi.size()); 
    int j = (int) floor((coschi+1.)/2.*hi.size()); 
    return hi[i]-hi[j];
  } else {
    return 1e99;
  }
}

// private methods
inline void EvtShapes::checkLinTen() {
  if (!_linTenDone) {
    diagonalizeTensors(true, _useCmBoost); 
    _linTenDone = true;
  }
}

inline void EvtShapes::checkSphericity() {
  if (!_spherDone) {
    diagonalizeTensors(false, _useCmBoost); 
    _spherDone = true;
  }
}

void EvtShapes::diagonalizeTensors(bool linear, bool cmboost) {  
  // initialize
  HepSymMatrix Theta = HepSymMatrix(3);
  for(int i=0; i<3; i++) 
    for(int j=0; j<3; j++) 
      Theta[i][j] = 0.0;
  double sum = 0.; 
  Vector3 sumvec = Vector3();
  vector<double> lam;
  vector<Vector3> n; 
  // get cm-frame
  Lorentz5Momentum pcm = Lorentz5Momentum(); 
  tPVector::const_iterator cit;
  Vector3 beta; 
  if (cmboost) {
    for(cit=_pv.begin(); cit != _pv.end(); ++cit) 
      pcm += (*cit)->momentum();    
    beta = pcm.findBoostToCM(); 
  }
  // get Theta_ij
  for(cit=_pv.begin(); cit != _pv.end(); ++cit) {
    Lorentz5Momentum dum = (*cit)->momentum();
    if (cmboost) dum.boost( beta );
    Vector3 pvec = dum.vect();
    if (pvec.mag() > 0) {
      sumvec += pvec;
      if (linear) 
	sum += pvec.mag();
      else 
	sum += pvec.mag2();
      for(int i=0; i<3; i++) 
	for(int j=i; j<3; j++) 
	  if (linear) 
	    Theta[i][j] += (pvec[i])*(pvec[j])/(pvec.mag());      
	  else 
	    Theta[i][j] += (pvec[i])*(pvec[j]);      
    }
  }
  Theta /= sum;      
  // diagonalize it
  HepMatrix U = diagonalize(&Theta);
  for(int i=0; i<3; i++) {
    lam.push_back( Theta[i][i] );
    Vector3 ndum;
    for(int j=0; j<3; j++) 
      ndum[j] = U[j][i]; 
    n.push_back( ndum ); 
  }
  // sort according to size of eigenvalues
  // such that lam[0] > lam[1] > lam[2]
  if (lam[0] < lam[1]) {
    swap(lam[0], lam[1]); 
    swap(n[0], n[1]);     
  }
  if (lam[0] < lam[2]) {
    swap(lam[0], lam[2]); 
    swap(n[0], n[2]);         
  }
  if (lam[1] < lam[2]) {
    swap(lam[1], lam[2]); 
    swap(n[1], n[2]);     
  }
  if (linear) {
    _linTen = lam; 
    _linTenAxis = n; 
  } else {
    _spher = lam; 
    _spherAxis = n; 
  }
}

inline void EvtShapes::checkThrust() {
  if (!_thrustDone) {
    calculateThrust();
    _thrustDone = true;
  }
}

void EvtShapes::calculateThrust() { 
  EventShape eshape;
  TObjArray vecp;
  double dumv3[3];
  for (tPVector::const_iterator cit = _pv.begin(); cit != _pv.end(); ++cit) {
    dumv3[0] = (*cit)->momentum().x();
    dumv3[1] = (*cit)->momentum().y();
    dumv3[2] = (*cit)->momentum().z();
    vecp.Add(new TVector3(dumv3));
  }
  eshape.setPartList(&vecp);
  _thrust.clear();
  _thrust.push_back(eshape.thrust().X());
  _thrust.push_back(eshape.thrust().Y());
  _thrust.push_back(eshape.thrust().Z());	
  Vector3 dum; 
  dum.setX(eshape.thrustAxis().X());
  dum.setY(eshape.thrustAxis().Y());
  dum.setZ(eshape.thrustAxis().Z());
  _thrustAxis.push_back(dum); 
  dum.setX(eshape.majorAxis().X());
  dum.setY(eshape.majorAxis().Y());
  dum.setZ(eshape.majorAxis().Z());
  _thrustAxis.push_back(dum); 
  dum.setX(eshape.minorAxis().X());
  dum.setY(eshape.minorAxis().Y());
  dum.setZ(eshape.minorAxis().Z());
  _thrustAxis.push_back(dum); 
  vecp.Delete(); 
}

inline void EvtShapes::checkHemispheres() {
  if (!_hemDone) {
    calcHemisphereMasses(); 
    _hemDone = true;
  }
}

void EvtShapes::calcHemisphereMasses() {
  Lorentz5Momentum pos, neg;
  tPVector::const_iterator cit;
  for(cit = _pv.begin(); cit != _pv.end(); cit++) 
    if ((*cit)->momentum().vect()*thrustAxis() > 0) 
      pos += (*cit)->momentum();
    else neg += (*cit)->momentum();
  _mPlus = pos.m()/(pos+neg).e();
  _mPlus *= _mPlus;
  _mMinus = neg.m()/(pos+neg).e();
  _mMinus *= _mMinus; 
  if (_mPlus < _mMinus) swap(_mPlus, _mMinus);
}

inline void EvtShapes::checkBroadening() {
  if (!_broadDone) {
    calcBroadening(); 
    _broadDone = true;
  }
}

void EvtShapes::calcBroadening() {
  Energy pos, neg, den; 
  tPVector::const_iterator cit;
  for(cit = _pv.begin(); cit != _pv.end(); cit++) {
    if ((*cit)->momentum().vect()*thrustAxis() > 0) 
      pos += (*cit)->momentum().perp(thrustAxis());
    else 
      neg += (*cit)->momentum().perp(thrustAxis());
    den += (*cit)->momentum().vect().mag();
  }
  _bPlus = pos/den/2.;
  _bMinus = neg/den/2;
  if (_bPlus < _bMinus) swap(_bPlus, _bMinus);
}

